import json
import logging
import datetime

from flask import Blueprint, request, session, jsonify
from flask_jwt_extended import jwt_required, get_jwt_identity
from webargs import fields
from webargs.flaskparser import use_args
from peewee import DoesNotExist, JOIN
from median.models import (ListeModel, ListeItemModel, Historique, Gpao, Magasin, Stock, FListe, FListeError,
                           Endpoint, Product, User)
from median.views import RawConfig
from median.constant import HistoryType, TypeServiListe, TypeListe, RIEDL_PRIORITY, EtatListe
from common.status import (HTTP_200_OK, HTTP_400_BAD_REQUEST, HTTP_500_INTERNAL_SERVER_ERROR, HTTP_403_FORBIDDEN,
                           HTTP_204_NO_CONTENT, HTTP_501_NOT_IMPLEMENTED)
from common.log import log_riedl
from common.label import compose_container_label
from common.printer import printer_riedl_configuration
from common.util import send_to_printer, get_counter

from peewee import fn


riedl_output_blueprint = Blueprint('riedl_output', __name__)

logger = logging.getLogger('median')


@riedl_output_blueprint.errorhandler(400)
def handle_error(err):
    """
    When an error 400, return a formated custom error on the frontend
    """
    headers = err.data.get("headers", None)
    messages = err.data.get("messages", ["Invalid request."])
    if headers:
        return jsonify({"message": messages}), err.code, headers
    else:
        return jsonify({"message": messages}), err.code


@riedl_output_blueprint.route('endpoints', methods=['GET'])
@jwt_required()
def get_endpoints():
    endpoints = Endpoint.select(Endpoint.pk, Endpoint.libelle, Endpoint.secteur, Endpoint.type_peigne) \
        .where(Endpoint.type_dest == 'RIEDL').order_by(Endpoint.type_peigne)

    return {'list': [
        {'pk': item.pk,
         'label': item.libelle,
         'riedl_code': item.secteur,
         'value': item.type_peigne} for item in endpoints]}, HTTP_200_OK


@riedl_output_blueprint.route('priorities', methods=['GET'])
@jwt_required()
def get_priorities():
    return {'list': [{'value': i[0], 'name': i[1]} for i in RIEDL_PRIORITY]}, HTTP_200_OK


@riedl_output_blueprint.route('all', methods=['POST'])
@jwt_required()
def get_all():
    data = json.loads(request.data)

    stores = data.get('stores', [])
    states = data.get('states', [])

    andexpr = ((FListe.mode == TypeListe.Output.value) &
               (FListe.type_servi << [TypeServiListe.GlobaleBoite.value, TypeServiListe.RiedlBoite.value,
                                      TypeServiListe.Nominatif.value, TypeServiListe.RetraitLot.value,
                                      TypeServiListe.Perimee.value]) &
               (FListe.zone_deb == 'RIEDL'))

    if len(stores) > 0:
        andexpr = andexpr & (FListe.zone_fin << stores)

    if len(states) > 0:
        andexpr = andexpr & (FListe.etat << states)

    rdl_list = FListe.select(FListe, Magasin.libelle,
                             FListeError.message.alias('error_message')) \
        .join(FListeError, JOIN.LEFT_OUTER,
              on=(FListe.liste == FListeError.liste) & (FListe.mode == FListeError.mode)) \
        .switch(FListe) \
        .join(Magasin, JOIN.LEFT_OUTER, on=FListe.zone_fin == Magasin.type_mag) \
        .where(andexpr).order_by(FListe.service)

    res = []
    for lst in rdl_list:

        type_id = None

        if lst.type_servi in {TypeServiListe.GlobaleBoite.value, TypeServiListe.RiedlBoite.value}:
            type_id = 'outglobal'
        elif lst.type_servi in {TypeServiListe.Nominatif.value}:
            type_id = 'outnominative'
        elif lst.type_servi in {TypeServiListe.RetraitLot.value}:
            type_id = 'outlot'
        elif lst.type_servi in {TypeServiListe.Perimee.value}:
            type_id = 'outperime'

        obj = {
            'type': type_id,
            'text': '%s' % (lst.liste,),
            'id': 'model-%s' % (lst.liste,),
            'isSend': lst.selectionne,
            'ward': lst.service,
            'mag': {
                'label': lst.magasin.libelle if hasattr(lst, 'magasin') and lst.magasin is not None else '',
                'type': lst.zone_fin,
            },
            'username': lst.username or '',
            'etat': lst.etat,
            'message': lst.listeerrormodel.error_message if hasattr(lst, 'listeerrormodel') else '',
            'date_create': str(lst.date_creation) if lst.date_creation is not None else '',
            'order_num': lst.interface,
            'items_num': lst.nb_item,
            'date_update': str(lst.ddeb) if lst.ddeb is not None else '',
            'output': lst.no_pilulier,
            'priority': lst.pos_pilulier
        }

        res.append(obj)

    return {'listes': res}, HTTP_200_OK


@riedl_output_blueprint.route('deleteitem', methods=['DELETE'])
@jwt_required()
def deleteItem():
    """Delete one item by PK"""
    # logger.info(request.data)
    data = json.loads(request.data)

    try:
        itm = ListeItemModel.get(pk=data['item_pk'])
        lst = ListeModel.get(mode="S", liste=itm.liste)

        mag = Magasin.select(Magasin).where(Magasin.type_mag == lst.zone_fin).order_by(Magasin.pk)
        qte_tot = Stock.select(fn.SUM(Stock.quantite)).where(Stock.reference == itm.reference).scalar()

        # Create a GPAO line
        logger.info('Create a GPAO line for item %i in list [%s]' % (data['item_pk'], itm.liste))
        Gpao.create(
            chrono=datetime.datetime.now(),
            poste='MEDIANWEB',
            etat='A',
            ref=itm.reference,
            qte=0,
            type_mvt='S',
            liste=itm.liste,
            dest=itm.dest,
            user=session['username'],
            item=itm.item,
            info=itm.info,
            solde=1,
            item_wms=itm.item_wms,
            fraction=itm.fraction,
            qte_dem=itm.qte_dem,
            ipp=itm.num_ipp,
            sejour=itm.num_sej,
            id_robot=mag and mag[0].id_robot or 1,
            id_zone=mag and mag[0].id_zone or 1,
            magasin=mag and mag[0].mag or 'ST1',
        )

        # Create an history
        logger.info('Create a GPAO line for item %i' % data['item_pk'])
        Historique.create(
            chrono=datetime.datetime.now(),
            reference=itm.reference,
            adresse='',
            magasin=mag and mag[0].mag or 'L01',
            quantite_mouvement=0,
            quantite_totale=qte_tot,
            quantite_demande=itm.qte_dem,
            service=itm.dest,
            liste=itm.liste,
            item=itm.item,
            type_mouvement=HistoryType.Sortie.value,
            pmp=0,
            poste='MEDIANWEB',
            ipp=itm.num_ipp,
            sejour=itm.num_sej,
            fraction=itm.fraction,
            item_wms=itm.item_wms,
            utilisateur=session['username'],
            info="MedianWeb -> Supp item",
            commentaire="Manual",
        )

        logger.info("RIEDL: delete item [%s]: f_item.pk %i" % (data['item'], data['item_pk']))
        log_riedl(
            session['username'], 'out_del_item',
            'suppression item %s (%s) liste %s ' % (itm.item, itm.pk, itm.liste)
        )
        itm.delete_instance()

        return {'result': 'ok'}, HTTP_200_OK

    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_400_BAD_REQUEST


@riedl_output_blueprint.route('report', methods=['POST'])
def report():
    return {}, HTTP_501_NOT_IMPLEMENTED


@riedl_output_blueprint.route('printcontainer', methods=['POST'])
@use_args({
    "liste": fields.Str(required=True)
}, location="json", error_status_code=400, error_headers={"validation": "error"})
@jwt_required()
def printContainer(args):
    """Print the container label"""
    try:
        # Load the output list and check if container
        lst = ListeModel.get(mode="S", liste=args['liste'])
        if not lst.id_plateau:
            # Retrieve prefix
            tray_prefix = ""
            try:
                cfg = RawConfig('TOUS')
                tray_prefix = cfg.read('k_rdl_prefix_container').value
            except Exception:
                tray_prefix = ""
            # Retrieve code
            lst.id_plateau = '%s%010d' % (tray_prefix, int(get_counter("RIEDL_CONTAINER")))
            lst.save()
        pr_addr, pr_name, pr_offset = printer_riedl_configuration(lst.zone_fin)
        logger.info("Print the container label for %s on %s" % (lst.id_plateau, pr_name))
        label = compose_container_label(lst, pr_offset)
        send_to_printer(pr_addr, label)
        return {'result': 'ok', 'args': args}, HTTP_200_OK
    except DoesNotExist as error:
        logger.error(error.args)
        return {'message': "List %s does not eixsts" % args['liste']}, HTTP_400_BAD_REQUEST
    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_400_BAD_REQUEST


@riedl_output_blueprint.route('forcefullboxes/<string:list_pk>', methods=['GET'])
@jwt_required()
def force_full_boxes(list_pk):
    logger.info("force list %s to move on full boxes" % list_pk)
    try:
        lst = ListeModel.get(mode="S", pk=list_pk)
        query = ListeItemModel.update({ListeItemModel.num_face: 1}).where(
            ListeItemModel.mode == "S", ListeItemModel.liste == lst.liste)
        query.execute()
        lst.num_face = 1
        lst.save()
        logger.info("Force list %s to use a full boxes" % lst.liste)
        log_riedl(
            session['username'], 'out_update_list',
            'Force la sortie des boites complètes de la liste %s' % (lst.liste,)
        )
        return {'result': 'ok', 'args': [list_pk]}, HTTP_200_OK
    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_400_BAD_REQUEST


@riedl_output_blueprint.route('validate/<string:item_pk>', methods=['PATCH'])
@jwt_required()
def validate(item_pk):
    try:
        data = json.loads(request.data)
        value = data.get('readonly', False)
        itm = (ListeItemModel.select(ListeItemModel)
               .where(ListeItemModel.pk == item_pk)).get()

        itm.readonly = value
        itm.save()
        logger.info("RIEDL: update item [%s]: f_item.pk %i" % (itm.item, itm.pk))
        log_riedl(
            session['username'], 'out_update_item',
            'mise à jour item %s (%s) liste %s ' % (itm.item, itm.pk, itm.liste)
        )

        return {'result': 'ok'}, HTTP_200_OK

    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_400_BAD_REQUEST


@riedl_output_blueprint.route('<string:x_liste>/items', methods=['POST'])
@jwt_required()
def post(x_liste):
    logger.info("Récupérer les items de listes riedl : '%s'" % x_liste)
    try:

        query = (ListeItemModel.select(ListeItemModel.pk, ListeItemModel.item,
                                       ListeItemModel.etat, ListeItemModel.reference,
                                       ListeItemModel.fraction, Product.designation,
                                       ListeItemModel.qte_dem, ListeItemModel.qte_serv,
                                       ListeItemModel.moment, ListeItemModel.heure,
                                       ListeItemModel.readonly, Product.pk)
                 .join(Product, on=ListeItemModel.reference == Product.reference)
                 .where((ListeItemModel.mode == 'S') & (ListeItemModel.liste == x_liste))
                 .order_by(ListeItemModel.item.asc()))

        if query.count() == 0:
            logger.error('No list with name %s' % x_liste)
            return {'data': []}, HTTP_200_OK

        return {'data': [{
            'pk': i.pk,
            'list_name': x_liste,
            'item': i.item,
            'etat': i.etat,
            'reference': i.reference,
            'fraction': i.fraction,
            'designation': i.product.designation,
            'qte_demandee': i.qte_dem,
            'qte_servie': i.qte_serv,
            'moment': i.moment,
            'heure': i.heure,
            'readonly': i.readonly,
            'reference_pk': i.product.pk,
        } for i in query]}, HTTP_200_OK

    except Exception as error:
        logger.error(('Get reappro items Datatables raised an exception: ', error.args))
        return {'message': error.args}, HTTP_500_INTERNAL_SERVER_ERROR


@riedl_output_blueprint.route('<string:x_liste>/init', methods=['PUT'])
@jwt_required()
def reinit_list(x_liste):
    authorized_profil = ['DEENOVA', 'ECO-DEX', 'PHARMACIEN']
    user_id = get_jwt_identity()
    try:
        User.select(User.profil).where((user_id == User.pk) &
                                       (User.profil << authorized_profil)).get()
    except DoesNotExist:
        return {'message': 'rield.output.list.error.forbidden',
                }, HTTP_403_FORBIDDEN

    try:
        riedl_lst = ListeModel.select().where(ListeModel.pk == x_liste).get()
    except DoesNotExist:
        return {'message': 'rield.output.list.error.not_exist',
                }, HTTP_500_INTERNAL_SERVER_ERROR

    try:
        riedl_lst.interface = ''
        riedl_lst.etat = EtatListe.Vierge.value
        riedl_lst.selectionne = 0
        riedl_lst.save()
    except Exception as error:
        logger.error(('Reinit_list raised an exception: ', error.args))
        return {'message': error.args}, HTTP_500_INTERNAL_SERVER_ERROR

    return {}, HTTP_204_NO_CONTENT
