import json

import numpy as np
from common.status import HTTP_200_OK
from flask import Blueprint, request
from flask_jwt_extended import jwt_required
from median.constant import HistoryType, EcoType
from median.models import Historique, Magasin, ListeValide, ItemValide
from peewee import fn
from ressources.blueprint.stats_utils import estimate_coef, lineary_regression

trends_blueprint = Blueprint('acced_trends', __name__)


@trends_blueprint.route('acced/completion', methods=['POST'])
@jwt_required()
def get_completion_trends():
    data = json.loads(request.data)
    start = data['start']
    end = data['end']

    completions = (ListeValide.select(fn.WEEKOFYEAR(ListeValide.chrono).alias('week_num'),
                                      fn.SUM(ItemValide.quantite_dem).alias('prescription_qty'),
                                      fn.SUM(ItemValide.quantite_serv).alias('served_qty'))
                   .join(ItemValide, on=ItemValide.liste_pk == ListeValide.pk)
                   .where((ListeValide.chrono >= start) & (ListeValide.chrono <= end))
                   .order_by(fn.WEEKOFYEAR(ListeValide.chrono))
                   .group_by(fn.WEEKOFYEAR(ListeValide.chrono))
                   ).objects()

    val_counts = list(map(lambda r: (1 - (r.prescription_qty - r.served_qty) / r.prescription_qty) * 100.00,
                          completions))
    total_served = sum(map(lambda r: (r.prescription_qty - r.served_qty), completions))
    total_prescription = sum(map(lambda r: r.prescription_qty, completions))
    counts = np.array(val_counts)
    val_weeks = list(map(lambda r: int(r.week_num), completions))
    weeks = np.array(val_weeks)

    count_coeff = estimate_coef(weeks, counts) if len(counts) > 0 else None
    count_r2 = lineary_regression(count_coeff, weeks, counts) if count_coeff is not None and count_coeff[
        0] is not None and count_coeff[
                                                                     1] is not None else None

    return {
        'total': (1 - (total_served / total_prescription)) * 100.00 if total_prescription > 0 else 0,
        'datas': val_counts,
        'weeks': val_weeks,
        'trends': (count_coeff[0] / abs(
            count_coeff[1])) * 100.0 if count_r2 is not None and count_r2 > 0.65 else None,

    }


@trends_blueprint.route('acced/return', methods=['POST'])
@jwt_required()
def get_return_trends():
    data = json.loads(request.data)
    start = data['start']
    end = data['end']
    return _get_trends(type_mvts=[HistoryType.Entree.value], start=start, end=end,
                       eco_types=[EcoType.Coupe.value, EcoType.Cueillette.value]), HTTP_200_OK


@trends_blueprint.route('acced/cut', methods=['POST'])
@jwt_required()
def get_cuts_trends():
    data = json.loads(request.data)
    start = data['start']
    end = data['end']
    return _get_trends(type_mvts=[HistoryType.Coupe.value, HistoryType.Directe.value], start=start, end=end,
                       eco_types=[EcoType.Coupe.value, EcoType.Cueillette.value]), HTTP_200_OK


@trends_blueprint.route('acced/pick', methods=['POST'])
@jwt_required()
def get_picking_trends():
    data = json.loads(request.data)
    start = data['start']
    end = data['end']
    return _get_trends(type_mvts=[HistoryType.Sortie.value], start=start, end=end,
                       eco_types=[EcoType.Coupe.value, EcoType.Cueillette.value]), HTTP_200_OK


@trends_blueprint.route('riedl/input', methods=['POST'])
@jwt_required()
def get_riedl_input_trends():
    data = json.loads(request.data)
    start = data['start']
    end = data['end']
    return _get_trends(type_mvts=[HistoryType.Entree.value], start=start, end=end,
                       eco_types=[EcoType.Riedl.value]), HTTP_200_OK


@trends_blueprint.route('riedl/output', methods=['POST'])
@jwt_required()
def get_riedl_output_trends():
    data = json.loads(request.data)
    start = data['start']
    end = data['end']
    return _get_trends(type_mvts=[HistoryType.Sortie.value], start=start, end=end,
                       eco_types=[EcoType.Riedl.value]), HTTP_200_OK


@trends_blueprint.route('astus/input', methods=['POST'])
@jwt_required()
def get_astus_input_trends():
    data = json.loads(request.data)
    start = data['start']
    end = data['end']
    return _get_trends(type_mvts=[HistoryType.Entree.value], start=start, end=end,
                       eco_types=[EcoType.Astus.value]), HTTP_200_OK


@trends_blueprint.route('astus/output', methods=['POST'])
@jwt_required()
def get_astus_output_trends():
    data = json.loads(request.data)
    start = data['start']
    end = data['end']
    return _get_trends(type_mvts=[HistoryType.Sortie.value], start=start, end=end,
                       eco_types=[EcoType.Astus.value]), HTTP_200_OK


def _get_trends(type_mvts, start, end, eco_types):
    req_mag = Magasin.select(Magasin.mag).where(Magasin.eco_type << eco_types)
    is_acced = any(filter(lambda x: x in [EcoType.Coupe.value, EcoType.Cueillette.value], eco_types))
    is_not_acced_return = not is_acced or HistoryType.Entree.value not in type_mvts

    returns = (Historique.select(fn.WEEKOFYEAR(Historique.chrono).alias('date'),
                                 fn.SUM(Historique.quantite_mouvement).alias("qty"))
               .where((Historique.chrono >= start) & (Historique.chrono <= end) &
                      (Historique.type_mouvement << type_mvts) & (is_not_acced_return | (Historique.service != '')) &
                      (fn.SUBSTRING(Historique.adresse, 1, 3) << req_mag))
               .order_by(fn.WEEKOFYEAR(Historique.chrono))
               .group_by(fn.WEEKOFYEAR(Historique.chrono)))

    val_counts = list(map(lambda r: r.qty, returns))
    counts = np.array(val_counts)
    val_weeks = list(map(lambda r: int(r.date), returns))
    weeks = np.array(val_weeks)

    count_coeff = estimate_coef(weeks, counts) if len(counts) > 0 else None
    count_r2 = lineary_regression(count_coeff, weeks, counts) if count_coeff is not None and count_coeff[
        0] is not None and count_coeff[
                                                                     1] is not None else None

    return {
        'total': sum(counts),
        'datas': val_counts,
        'weeks': val_weeks,
        'trends': (count_coeff[0] / abs(
            count_coeff[1])) * 100.0 if count_r2 is not None and count_r2 > 0.65 else None,

    }
