import logging

from flask import Blueprint, request
from flask_jwt_extended import jwt_required
from median.models import Product, Ucd, Cip
from peewee import JOIN

suggestions_blueprint = Blueprint('suggestions', __name__)

logger = logging.getLogger('median')


@suggestions_blueprint.route('', methods=['GET'])
@jwt_required()
def get_all():
    args = request.args
    v_search_text = args['q'].replace(' ', '%')

    try:
        products = (Product.select(Product.reference, Product.ucd, Product.designation, Product.pk)
                    .where(Product.reference.contains(v_search_text) |
                           Product.designation.contains(v_search_text) |
                           Product.ucd.contains(v_search_text)))

        if products.count() > 0:
            logger.info('Lines : %s.' % len(products))
            return ([{
                'cdu': p.ucd,
                'value': p.pk,
                'text': '[ ' + p.reference + ' / ' + p.ucd + ' ] ' + p.designation,
                'designation': p.designation
            } for p in products])

        # si on ne trouve rien, on va chercher sur l'ucd, dans f_ref_ucd
        else:
            products = (Ucd.select(Ucd.reference.alias('reference'),
                                   Ucd.ucd.alias('ucd'),
                                   Product.pk.alias('product_pk'),
                                   Product.designation.alias('designation'))
                        .join(Product, JOIN.LEFT_OUTER, on=Product.reference == Ucd.reference)
                        .where(Ucd.ucd.contains(v_search_text)))
            if products.count() > 0:
                logger.info('Lines : %s.' % len(products))
                return ([{
                    'value': p.product_pk,
                    'text': f'[{p.cip} / {p.ucd}] {p.designation if p.designation is not None else p.ref}'
                } for p in products.objects])

            # si on ne trouve rien, on va chercher sur le cip, dans f_ucd_cip
            else:
                products = (Cip.select(Cip.cip.alias('cip'),
                                       Cip.ucd.alias('ucd'),
                                       Ucd.reference.alias('ref'),
                                       Product.pk.alias('product_pk'),
                                       Product.designation.alias('designation')).distinct()
                            .join(Ucd, on=(Ucd.ucd == Cip.ucd))
                            .join(Product, JOIN.LEFT_OUTER, on=Product.reference == Ucd.reference)
                            .where(Cip.cip.contains(v_search_text)))
                if products.count() > 0:
                    logger.info('Lines : %s.' % len(products))
                    return ([{
                        'value': p.product_pk,
                        'text': f'[{p.cip} / {p.ucd}] {p.designation if p.designation is not None else p.ref}'
                    } for p in products.objects()])
                else:
                    return []

    except Exception as error:
        return {'message': error.args}, 400
