import datetime
import json
import logging

from common.status import HTTP_200_OK, HTTP_500_INTERNAL_SERVER_ERROR, HTTP_503_SERVICE_UNAVAILABLE
from flask import Blueprint, request, session
from flask_jwt_extended import jwt_required
from median.constant import HistoryType
from median.models import Stock, Adresse, CodeBlocage, Product, Cip, Historique, Magasin, Gpao, Service
from peewee import JOIN, fn, DoesNotExist

stock_blueprint = Blueprint('stock', __name__)

logger = logging.getLogger('median')


def _padAddressField(adr):
    _adr_items = adr.split('.')
    # if len(_adr_items) != 5:
    #     return False
    _o = []
    for a in _adr_items:
        _o.append(a.rjust(3))

    return '.'.join(_o)


def _moveContainerForward(adr):
    _els = adr.split('.')
    _pos = int(_els[-1].lstrip())
    if _pos > 1:
        return False

    _back_adrs = []
    _els[-1] = '  2'
    _back_adrs.append('.'.join(_els))
    _els[-1] = '  3'
    _back_adrs.append('.'.join(_els))

    _a = (Adresse.select(Adresse.contenant)
          .where((Adresse.adresse == _back_adrs[0]) | (Adresse.adresse == _back_adrs[1])))

    _cont = '' if _a.count() == 0 else _a[0].contenant

    (Adresse.update({Adresse.contenant: _cont, Adresse.etat: 'O'})
     .where(Adresse.adresse == adr).execute())

    (Adresse.update({Adresse.contenant: '', Adresse.etat: 'L'})
     .where((Adresse.adresse == _back_adrs[0]) | (Adresse.adresse == _back_adrs[1]))
     .execute())

    (Stock.update({Stock.adresse: adr})
     .where((Stock.adresse == _back_adrs[0]) | (Stock.adresse == _back_adrs[1]))
     .execute())


def _convertUserFriendlyDateToDBDateTime(ufDate):
    _dl = ufDate.split('-')
    if len(_dl[0]) == 4:
        return ufDate + ' 00:00:00'
    return _dl[2] + '-' + _dl[1] + '-' + _dl[0] + ' 00:00:00'


@stock_blueprint.route('<string:ref>', methods=['PATCH'])
@jwt_required()
def get_all(ref):
    args = json.loads(request.data)
    v_filter_by_magasin = args.get('filterByMagasin', None)

    try:
        if not v_filter_by_magasin:

            filtered_stocks_query = (
                Stock
                .select(Stock, Adresse, CodeBlocage.libelle, CodeBlocage.valeur)
                .switch(Stock)
                .join(Adresse, JOIN.LEFT_OUTER, on=Adresse.adresse == Stock.adresse).alias('adr')
                .switch(Stock)
                .join(CodeBlocage, JOIN.LEFT_OUTER, on=CodeBlocage.valeur == Adresse.bloque)
                .join(Product, on=Stock.reference == Product.reference)
                .where(Product.pk == ref))
        else:
            ms = v_filter_by_magasin.split(",")

            filtered_stocks_query = (
                Stock
                .select(Stock.pk, Adresse, CodeBlocage.libelle, CodeBlocage.valeur)
                .switch(Stock)
                .join(Adresse, JOIN.LEFT_OUTER, on=Adresse.adresse == Stock.adresse).alias('adr')
                .switch(Stock)
                .join(CodeBlocage, JOIN.LEFT_OUTER, on=CodeBlocage.valeur == Adresse.bloque)
                .join(Product, on=Stock.reference == Product.reference)
                .where((Product.pk == ref) & (Stock.magasin << ms)))

        filtered_stocks = (filtered_stocks_query
                           .order_by(Stock.adresse, Stock.date_peremption))

        logger.debug('Lines : %s.' % len(filtered_stocks))

        return {'data': [{
            'pk': s.pk,
            'reference': s.reference,
            'format': s.adresse.format if type(s.adresse) is not str else '',
            'lock_code': s.codeblocage.valeur if hasattr(s, 'codeblocage') else 0,
            'lock_custom_msg': s.adresse.bloque_message if type(s.adresse) is not str else '',
            'lock_msg': s.codeblocage.libelle if hasattr(s, 'codeblocage') else '',
            'bloque': ('OUI' if s.bloque else 'NON'),
            'emplacement': s.adresse.adresse if type(s.adresse) is not str else s.adresse,
            'quantite': s.quantite,
            'ucd': s.ucd,
            'gtin': s.cip,
            'date_sortie': str(s.date_sortie or '-'),
            'date_entree': str(s.date_entree or '-'),
            'lot': s.lot,
            'date_peremption': str(s.date_peremption).split(' ')[0],
            'contenant': s.contenant,
            'fraction': s.fraction,
            'qte_blister_bac': s.qte_blister_bac,
            'nb_dose_cut': s.nb_dose_cut,
            'du': s.DU,
            'qte_prod_blister': s.qte_prod_blister,
            'magasin': s.magasin,
            'serial': s.serial
        } for s in filtered_stocks]}, HTTP_200_OK

    except Exception as error:
        logger.error('Get stock Datatables raised an exception: ', error.args)
        return {'message': error.args}, HTTP_500_INTERNAL_SERVER_ERROR


@stock_blueprint.route('<string:ref>', methods=['POST'])
@jwt_required()
def create_stock(ref):
    logger.info('Création d\'une ligne de stock...')

    args = request.form
    _dp = _convertUserFriendlyDateToDBDateTime(args['date_peremption'])
    _ucd = args['ucd'].strip()

    try:
        _calc_fifo = (Stock.select((fn.IFNULL(fn.MAX(Stock.id_fifo), 0) + 1).alias('new_fifo'))
                      .where(Stock.reference == ref))
        logger.info('Compute FIFO : "%s"' % (_calc_fifo))

        _calc_capa = (Cip.select(fn.MIN(Cip.qt_pass).alias('qt_pass')).where(Cip.ucd == '_ucd'))
        _cal = _calc_capa[0].qt_pass or 30
        logger.info('COmpute PASSbox quantity (qt_pass) : "%s"' % _cal)

        _existing_adr = Stock.select().where((Stock.adresse == args['adresse']) & (Stock.reference != ref))
        if len(_existing_adr):
            logger.warning('This location is actually occupied')
            return {'message': 'This location is actually occupied'}, HTTP_503_SERVICE_UNAVAILABLE

        logger.debug('Ajout ligne de stock, dans la table f_stock')
        Stock.create(
            adresse=args['adresse'],
            reference=ref,
            quantite=args['quantite'],
            id_fifo=_calc_fifo[0].new_fifo,
            lot=args['lot'],
            date_peremption=_dp,
            date_entree=str(datetime.datetime.now()).split('.')[0],
            contenant=args['containerCode'],
            magasin=args['magasin'],
            ucd=_ucd,
            fraction=args['fraction'],
            capa=_cal,
        )

        logger.debug('Mise à jour adresse "%s", état O & code contenant: "%s"' % (
            args['adresse'], args['containerCode']))
        (Adresse.update({Adresse.etat: 'O', Adresse.contenant: args['containerCode']})
         .where(Adresse.adresse == args['adresse']).execute())

        qte_tot = (Stock.select(fn.SUM(Stock.quantite)).where(Stock.reference == ref))

        logger.debug('Ecriture dans la table historique')
        Historique.create(
            chrono=datetime.datetime.now(),
            reference=ref,
            adresse=args['adresse'],
            magasin=args['adresse'][:3],
            quantite_mouvement=args['quantite'],
            quantite_totale=qte_tot,
            quantite_picking=qte_tot,
            service=args['service'],
            type_mouvement='ENT',
            lot=args['lot'][0],
            pmp=0,
            date_peremption=_dp,
            contenant=args['containerCode'],
            poste='MEDIANWEB',
            ucd=_ucd,
            fraction=args['fraction'],
            utilisateur=session['username']
        )

        _calc_id_robot = (Magasin.select(Magasin.id_robot, Magasin.id_zone).where(Magasin.mag == args['magasin']))
        _id_robot = _calc_id_robot[0].id_robot or 1

        logger.info('GPAO : Ajout d\'un mouvement de type ENTREE')
        Gpao.create(
            chrono=datetime.datetime.now(),
            poste='MEDIANWEB',
            etat='A',
            ref=ref,
            qte=args['quantite'],
            lot=args['lot'],
            type_mvt='E',
            dest=args['service'],
            tperemp=_dp,
            fraction=args['fraction'],
            id_robot=_id_robot,
            id_zone=_calc_id_robot[0].id_zone or 1,
            user=session['username'],
            magasin=args['adresse'][:3],
        )

    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_503_SERVICE_UNAVAILABLE

    logger.info('Create a stock line: success')
    return 'Success'


@stock_blueprint.route('<string:ref_pk>', methods=['DELETE'])
@jwt_required()
def delete_stock(ref_pk):
    args = request.args
    _pk = args['pk']
    _service = args['service']

    try:
        Stock.select().where(Stock.pk == _pk).get()
    except DoesNotExist:
        return {'alertMessage': 'ui.stock.unknown'}, HTTP_503_SERVICE_UNAVAILABLE

    try:
        reference = Product.select(Product.reference).where(Product.pk == ref_pk).get()
    except DoesNotExist:
        return {'alertMessage': 'ui.reference.unknown'}, HTTP_503_SERVICE_UNAVAILABLE

    logger.info('Delete stock line, pk: "%s"' % (_pk))

    try:
        Service.select().where(Service.code == _service).get()
    except DoesNotExist:
        return {'alertMessage': 'ui.ward.unknown'}, HTTP_503_SERVICE_UNAVAILABLE

    _padded_adr = _padAddressField(args['adresse'])
    if not _padded_adr:
        logger.warning('Error when formating address')
        return 'Error when formating address', 503

    _adrs = Stock.select().where(Stock.adresse == _padded_adr)

    try:
        logger.debug('Update address to become free: "%s"' % _padded_adr)
        Adresse.update({Adresse.etat: 'L'}).where(Adresse.adresse == _padded_adr).execute()

        logger.debug(f'Delete stock line at address {_padded_adr}...')
        for st in _adrs:
            st.delete_instance()
            if _adrs.count() == 0 and args['is_empty_container'] == 'false':
                logger.info('Move passbox from .3 to .1...')
                _moveContainerForward(_padded_adr)
            qte_tot = (Stock.select(fn.SUM(Stock.quantite)).where(Stock.reference == reference.reference))

            logger.debug('HISTO: Add delete movement')
            Historique.create(
                chrono=datetime.datetime.now(),
                reference=reference.reference,
                adresse=_padded_adr,
                magasin=_padded_adr[0:3],
                quantite_mouvement=st.quantite,
                quantite_totale=qte_tot,
                service=_service,
                type_mouvement=HistoryType.Sortie.value,
                lot=st.lot,
                pmp=0,
                date_peremption=st.date_peremption,
                contenant=st.contenant,
                poste='MEDIANWEB',
                ucd=st.ucd.strip(),
                fraction=st.fraction,
                serial=st.serial,
                quantite_picking=qte_tot,
                utilisateur=session['username']
            )

            _calc_id_robot = Magasin.get_or_none(mag=_padded_adr[0:3])

            logger.info("GPAO: add delete movement.")
            Gpao.create(
                poste='MEDIANWEB',
                chrono=datetime.datetime.now(),
                etat='A',
                ref=reference.reference,
                qte=st.quantite,
                lot=st.lot.strip(),
                type_mvt='S',
                dest=_service,
                tperemp=st.date_peremption,
                fraction=st.fraction,
                serial=st.serial,
                ucd=st.ucd.strip(),
                magasin=_padded_adr[0:3],
                contenant=st.contenant,
                user=session['username'],
                id_zone=_calc_id_robot.id_zone if _calc_id_robot else 1,
                id_robot=_calc_id_robot.id_robot if _calc_id_robot else 1
            )

    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_503_SERVICE_UNAVAILABLE

    logger.info('Delete stock line with success, pk: "%s"' % (_pk))
    return 'Success'
