import json
import logging

from flask import Blueprint, request
from flask_jwt_extended import jwt_required
from median.models import Rights, Profil

from common.status import HTTP_200_OK, HTTP_204_NO_CONTENT
from peewee import DoesNotExist, JOIN

rights_blueprint = Blueprint('rights', __name__)

logger = logging.getLogger('median')


@rights_blueprint.route('all', methods=['GET'])
@jwt_required()
def get_all_rights():
    req = (Rights.select(Rights.ressource.alias('ressource'), Rights.group.alias('group'))
           .where(Rights.group.is_null(False)))
    return {'list': [{
        'resource': i.ressource,
        'group': i.group,
        # 'add': 0,
        # 'edit': 0,
        # 'delete': 0,
        # 'visu': 0
    } for i in req]}, HTTP_200_OK


# Helper function for save_right
def process_right(data):
    logging.info(f"Processing right with data: {data}")

    # Validate required fields
    required_fields = ['mode', 'resource', 'profil', 'valeur']
    for field in required_fields:
        if field not in data:
            raise ValueError(f"Missing required field: {field}")

    _mode = data['mode']
    _resource = data['resource']
    _profile = data['profil']
    _value = data['valeur']
    try:
        profile = (Profil.select(Profil)
                   .where((Profil.ressource == _resource) & (Profil.profil == _profile))
                   .get())
    except DoesNotExist:
        profile = Profil()
        profile.ressource = _resource
        profile.profil = _profile

    if _mode == 'visu':
        profile.visu = _value
    elif _mode == 'edit':
        profile.edit = _value
    elif _mode == 'delete':
        profile.supp = _value
    elif _mode == 'add':
        profile.add = _value
    else:
        raise ValueError(f"invalid mode: {_mode}")

    try:
        profile.save()
    except Exception as e:
        logging.error(f"Failed to save profile: {e}")
        raise


@rights_blueprint.route('', methods=['PUT'])
# @jwt_required()
def save_right():
    data = json.loads(request.data)
    print(data)

    if isinstance(data, list):
        for item in data:
            process_right(item)
    else:
        process_right(data)

    return {}, HTTP_204_NO_CONTENT


@rights_blueprint.route('<string:profil_id>', methods=['GET'])
@jwt_required()
def get_profil(profil_id):
    req = (Rights.select(Profil.cree.alias('cree'),
                         Profil.profil.alias('profil'),
                         Profil.edit.alias('edit'), Profil.visu.alias('visu'),
                         Profil.supp.alias('supp'), Rights.ressource.alias('ressource'),
                         Rights.group.alias('group'))
           .join(Profil, JOIN.LEFT_OUTER,
                 on=(Rights.ressource == Profil.ressource) & (Profil.profil == profil_id))
           .where((Rights.group.is_null(False))))
    return {'list': [{
        'resource': i.ressource,
        'group': i.group,
        'label': i.profil,
        'add': i.cree,
        'edit': i.edit,
        'delete': i.supp,
        'visu': i.visu} for i in req.objects()]}, HTTP_200_OK
