import datetime
import json
import socket
from datetime import date
from urllib.parse import urlparse

from flask import Blueprint, request, session
from flask_jwt_extended import jwt_required

from median.models import Magasin, Config, Product, ListeItemModel, Poste, FListe, FItem, Service, Stock, LotRetire, \
    Ucd
from median.constant import TypeListe, EtatListe
from common.status import (HTTP_200_OK, HTTP_201_CREATED, HTTP_415_UNSUPPORTED_MEDIA_TYPE, HTTP_400_BAD_REQUEST,
                           HTTP_405_METHOD_NOT_ALLOWED, HTTP_404_NOT_FOUND, HTTP_204_NO_CONTENT)
from common.riedl import get_url_dispatcher, add_reference_inventory
from common.exception import RiedlDispatcherException
from median.views import RawConfig, RiedlView
from peewee import DoesNotExist
from common.util import logger
import requests
from common.exception import RiedlPrinterException, PrinterException
from common.label import compose_container_label
from common.log import log_riedl
from common.printer import printer_riedl_configuration
from common.util import get_counter, is_multiple, send_to_printer, read_printer_template

riedl_blueprint = Blueprint('riedl', __name__)


def get_diff(rield, base_riedl):
    obj = {}
    if rield['axis_state'] != base_riedl['axis_state']:
        obj['pk'] = rield['pk']
        obj['axis_state'] = rield['axis_state']

    if rield['dispatcher_state'] != base_riedl['dispatcher_state']:
        obj['pk'] = rield['pk']
        obj['dispatcher_state'] = rield['dispatcher_state']

    if rield['dispatcher_date'] != base_riedl['dispatcher_date']:
        obj['pk'] = rield['pk']
        obj['dispatcher_date'] = rield['dispatcher_date']

    if rield['door_state'] != base_riedl['door_state']:
        obj['pk'] = rield['pk']
        obj['door_state'] = rield['door_state']

    if rield['gripper_state'] != base_riedl['gripper_state']:
        obj['pk'] = rield['pk']
        obj['gripper_state'] = rield['gripper_state']

    if rield['mode_state'] != base_riedl['mode_state']:
        obj['pk'] = rield['pk']
        obj['mode_state'] = rield['mode_state']
    return obj


def get_riedls():
    mags = Magasin \
        .select(Magasin.pk, Magasin.mag, Magasin.type_mag, Magasin.avatar,
                Magasin.eco_type, Magasin.libelle, Magasin.type_machine,
                Config.select(Config.value)
                .where((Config.poste == Magasin.type_mag) & (Config.cle == 'status') &
                       (Config.propriete == 'k_rdl_state_axis')).alias('axis_state'),
                Config.select(Config.value)
                .where((Config.poste == Magasin.type_mag) & (Config.cle == 'status') &
                       (Config.propriete == 'k_rdl_state_dispatcher')).alias('dispatcher_state'),
                Config.select(Config.value)
                .where((Config.poste == Magasin.type_mag) & (Config.cle == 'status') &
                       (Config.propriete == 'k_rdl_state_dispatcher_date')).alias('dispatcher_date'),
                Config.select(Config.value)
                .where((Config.poste == Magasin.type_mag) & (Config.cle == 'status') &
                       (Config.propriete == 'k_rdl_state_door')).alias('door_state'),
                Config.select(Config.value)
                .where((Config.poste == Magasin.type_mag) & (Config.cle == 'status') &
                       (Config.propriete == 'k_rdl_state_gripper')).alias('gripper_state'),
                Config.select(Config.value)
                .where((Config.poste == Magasin.type_mag) & (Config.cle == 'status') &
                       (Config.propriete == 'k_rdl_state_mode')).alias('mode_state'),
                Config.select(Config.value)
                .where((Config.poste == Magasin.type_mag) & (Config.cle == 'status') &
                       (Config.propriete == 'k_rdl_maintenance')).alias('maintenance_state'),
                ) \
        .where(Magasin.eco_type == 'L') \
        .order_by(Magasin.type_mag)
    return [{
        'pk': m.pk,
        'mag': m.mag,
        'type_mag': m.type_mag,
        'eco_type': m.eco_type,
        'libelle': m.libelle,
        'type_machine': m.type_machine,
        'axis_state': m.axis_state,
        'dispatcher_state': m.dispatcher_state,
        'dispatcher_date': m.dispatcher_date,
        'door_state': m.door_state,
        'gripper_state': m.gripper_state,
        'mode_state': m.mode_state,
        'maintenance_state': m.maintenance_state,
        'avatar': m.avatar,

    } for m in mags]


def _rield_maintenance(type_mag, val):
    try:
        config = (Config.select()
                  .where((Config.poste == type_mag) &
                         (Config.cle == 'status') &
                         (Config.propriete == 'k_rdl_maintenance')).get())
    except DoesNotExist:
        config = Config()
        config.poste = type_mag
        config.cle = 'status'
        config.propriete = 'k_rdl_maintenance'

    config.value = val
    config.save()
    return {}, HTTP_204_NO_CONTENT


@riedl_blueprint.route('/<string:type_mag>/maintenance', methods=['PUT'])
@jwt_required()
def set_riedl_maintenance(type_mag):
    _rield_maintenance(type_mag=type_mag, val=1)
    return {}, HTTP_204_NO_CONTENT


@riedl_blueprint.route('/<string:type_mag>/active', methods=['PUT'])
@jwt_required()
def set_riedl_active(type_mag):
    _rield_maintenance(type_mag=type_mag, val=0)
    return {}, HTTP_204_NO_CONTENT


@riedl_blueprint.route('/inventory', methods=['POST'])
@jwt_required()
def get_inventory_riedl():
    data = json.loads(request.data)

    product_mag = data.get('mag', None)
    product_code = data.get('product', None)
    gtin_version_zero = []

    logger.info("Inventory request for %s on %s by %s" % (product_code, product_mag, session['username']))

    try:
        mag = Magasin.get(mag=product_mag)
    except DoesNotExist:
        return {'message': 'reference.rield.inventory.error.mag'}, 500

    try:
        url = get_url_dispatcher(mag.type_mag)
    except RiedlDispatcherException:
        return {'message': 'reference.rield.inventory.error.internal'}, 500

    try:
        pro = Product.get(reference=product_code)
        gtins = pro.gtin_list()
        for g in gtins:
            if g.dossier == "0" and g.cip not in gtin_version_zero:
                gtin_version_zero.append(g.cip)
        logger.info('Ask inventory to this GTIN: ' + ','.join(gtin_version_zero))
        logger.info("Url of the dispatcher: %s" % url)

        # Create a list and item for this product, 1 item per CIP
        current_liste = ""
        for gt in gtin_version_zero:
            current_liste, _ = add_reference_inventory(
                mag.type_mag, product_code, gt, session['username']
            )

        try:
            items = ListeItemModel.select(ListeItemModel).where(
                ListeItemModel.mode == TypeListe.Inventory.value, ListeItemModel.liste == current_liste
            ).order_by(ListeItemModel.item)
            logger.info("Number of GTIN to inventory %i" % len(items))
            for itm in items:
                if itm.etat == 'V':
                    resp = requests.post('%s/RiedlVMessage' % url, json={
                        "warehouseCode": mag.mag,
                        "orderNumber": str(itm.id_chargement),
                        "barcode": str(itm.contenant),
                        "batchNumber": "",
                        "externalIdCode": "",
                    })
                    if resp.status_code != 202:
                        logger.error('Error when sending http request %s -> %s' % (resp.status_code, resp.text))
                    itm.etat = 'E'
                    itm.save()
        except Exception:

            raise RiedlDispatcherException()

    except RiedlDispatcherException:
        return {'message': 'reference.rield.inventory.error.internal'}, 500

    return {'message': 'reference.rield.inventory.success'}, HTTP_201_CREATED


@riedl_blueprint.route('/all', methods=['GET'])
@jwt_required()
def get_all_riedl():
    return {
        'data': get_riedls()
    }, HTTP_200_OK


zpl_template = """^XA
    ^LS{offset}^FS
    ^FS^FS
    ^PON^FS
    ^KL3^FS

    ^FO50,30^GB372,731,4,B,2^FS
    ^FO106,96^A0R,52,36^FD{libelle1}^FS
    ^FO56,96^A0R,52,36^FD{libelle2}^FS
    ^FO360,336^A0R,38,28^FDPC: {gtin}^FS
    ^FO312,336^A0R,38,28^FDSN: {serial}^FS
    ^FO264,336^A0R,38,28^FDEXP: {expiry:%Y-%m}^FS
    ^FO216,336^A0R,38,28^FDLOT: {batch}^FS
    ^FO166,550^A0R,52,36^FDQTE: {quantity}^FS
    ^FO200,100^BXR,7,200,26,26,6,_^FD_101{gtin}17{expiry:%y%m%d}10{batch}_192{fraction}_1711{cip}^FS
    ^FO120,720^A0N,30,30^FDMEDIAN / DEENOVA^FS

    ^FO450,30^GB372,731,4,B,2^FS
    ^FO506,96^A0R,52,36^FD{libelle1}^FS
    ^FO456,96^A0R,52,36^FD{libelle2}^FS
    ^FO760,336^A0R,38,28^FDPC: {gtin}^FS
    ^FO712,336^A0R,38,28^FDSN: {serial}^FS
    ^FO664,336^A0R,38,28^FDEXP: {expiry:%Y-%m}^FS
    ^FO616,336^A0R,38,28^FDLOT: {batch}^FS
    ^FO566,550^A0R,52,36^FDQTE: {quantity}^FS
    ^FO600,100^BXR,7,200,26,26,6,_^FD_101{gtin}17{expiry:%y%m%d}10{batch}_192{fraction}_1711{cip}^FS
    ^FO520,720^A0N,30,30^FDMEDIAN / DEENOVA^FS

    ^FO850,30^GB372,731,4,B,2^FS

    ^FO906,96^A0R,52,36^FD{libelle1}^FS
    ^FO856,96^A0R,52,36^FD{libelle2}^FS
    ^FO1160,336^A0R,38,28^FDPC: {gtin}^FS
    ^FO1112,336^A0R,38,28^FDSN: {serial}^FS
    ^FO1064,336^A0R,38,28^FDEXP: {expiry:%Y-%m}^FS
    ^FO1016,336^A0R,38,28^FDLOT: {batch}^FS
    ^FO966,550^A0R,52,36^FDQTE: {quantity}^FS
    ^FO1000,100^BXR,7,200,26,26,6,_^FD_101{gtin}17{expiry:%y%m%d}10{batch}_192{fraction}_1711{cip}^FS
    ^FO920,720^A0N,30,30^FDMEDIAN / DEENOVA^FS

    ^PQ1^FS
    ^XZ
"""


@riedl_blueprint.route('/return', methods=['GET'])
@jwt_required()
def get():
    return {'message': 'ApiRiedlReturn::get not implemented'}, HTTP_405_METHOD_NOT_ALLOWED


@riedl_blueprint.route('/return', methods=['POST'])
@jwt_required()
def post():
    if not request.is_json:
        return {'message': 'Content-Type must be application/json'}, HTTP_415_UNSUPPORTED_MEDIA_TYPE

    content = request.get_json()

    try:
        ward = Service.get(code=content.get('ward', ''))
    except DoesNotExist:
        err_message = "Ward [%s] not found" % content.get('ward', '')
        logger.error(err_message)
        return {'message': err_message}, HTTP_400_BAD_REQUEST

    try:
        drug = Product.get(pk=content.get('drugs', ''))
    except DoesNotExist:
        err_message = "Drugs [%s] not found" % content.get('drugs', '')
        logger.error(err_message)
        return {'message': err_message}, HTTP_400_BAD_REQUEST

    try:
        mag = Magasin.get(type_mag=content.get('riedl', ''))
    except DoesNotExist:
        err_message = "Riedl [%s] not found" % content.get('riedl', '')
        logger.error(err_message)
        return {'message': err_message}, HTTP_400_BAD_REQUEST

    batch = content.get('batch', '-')

    if any(LotRetire.select(LotRetire.pk)
                    .join(Ucd, on=LotRetire.ucd == Ucd.ucd)
                    .where((LotRetire.lot == batch) & (Ucd.reference == drug.reference))):
        return {'message': 'riedl.error.withdrawnbatch'}, HTTP_400_BAD_REQUEST

    quantity = content.get('quantity', 0)
    batch = content.get('batch', '-')
    expiry = content.get('expiry', None)
    cip = content.get('cip', '')

    try:
        expiry = date.fromisoformat(expiry)
        today = datetime.datetime.now()

        if expiry <= today.date():
            return {'message': 'riedl.error.expiration_date.sup'}, HTTP_400_BAD_REQUEST

    except ValueError:
        err_message = 'expiry date error, need yyyy-mm-dd, receive %s' % (expiry,)
        logger.error(err_message)
        return {'message': err_message}, HTTP_400_BAD_REQUEST

    rdlview = RiedlView(mag.type_mag)
    ret_id = rdlview.create_ward_return(ward.code, drug.reference, quantity, batch, expiry, cip)

    return {'message': 'ApiRiedlReturn::get success', 'id': ret_id.pk}, HTTP_201_CREATED


@riedl_blueprint.route('/return', methods=['PUT'])
@jwt_required()
def put():
    return {'message': 'ApiRiedlReturn::put not implemented'}, HTTP_405_METHOD_NOT_ALLOWED


@riedl_blueprint.route('/return', methods=['DELETE'])
@jwt_required()
def delete():
    return {'message': 'ApiRiedlReturn::delete not implemented'}, HTTP_405_METHOD_NOT_ALLOWED


@riedl_blueprint.route('/printer', methods=['POST'])
@jwt_required()
def printer():
    if not request.is_json:
        return {'message': 'Content-Type must be application/json'}, HTTP_415_UNSUPPORTED_MEDIA_TYPE

    content = request.get_json()

    poste = content.get("poste", "")
    mode = content.get("mode", "")
    mode_id = int(content.get("id", 0))
    mode_test = int(content.get("test", 0))
    logger.info("printing mode: %s -> id: %i" % (mode, mode_id))

    try:
        Poste.get(poste=poste)
    except DoesNotExist:
        err_message = "Poste not found: %s" % poste
        logger.error(err_message)
        return {'message': err_message}, HTTP_400_BAD_REQUEST

    if mode not in ('list', 'item'):
        err_message = "Incorrect mode: %s" % mode
        logger.error(err_message)
        return {'message': err_message}, HTTP_400_BAD_REQUEST

    try:
        printer_cfg = _printer_configuration(poste)
    except RiedlPrinterException as e:
        err_message = e.message
        logger.error(err_message)
        return {'message': err_message}, HTTP_400_BAD_REQUEST

    if mode == 'list':
        # List mode detect, we print all items
        try:
            lst_model = FListe.get(pk=mode_id)
        except DoesNotExist:
            err_message = "List not exists: %i" % mode_id
            logger.error(err_message)
            return {'message': err_message}, HTTP_400_BAD_REQUEST

        itms = FItem.select(FItem).where(FItem.liste == lst_model.liste, FItem.mode == lst_model.mode)
        for line in itms:
            try:
                label = _compose_label_from_item(line, printer_cfg[2], printer_cfg[3])
                if not mode_test:
                    _send_to_printer(printer_cfg[0], label)
            except RiedlPrinterException as e:
                err_message = e.message
                logger.error(err_message)
                return {'message': err_message}, HTTP_400_BAD_REQUEST

    if mode == 'item':
        # Print only one item
        try:
            item_model = FItem.get(pk=mode_id)
        except DoesNotExist:
            err_message = "Item not exists: %i" % mode_id
            logger.error(err_message)
            return {'message': err_message}, HTTP_400_BAD_REQUEST

        try:
            label = _compose_label_from_item(item_model, printer_cfg[2], printer_cfg[3])
            if not mode_test:
                _send_to_printer(printer_cfg[0], label)
        except RiedlPrinterException as e:
            err_message = e.message
            logger.error(err_message)
            return {'message': err_message}, HTTP_400_BAD_REQUEST

    return {"message": "Print send sucessfully", "printer": printer_cfg}, HTTP_201_CREATED


def _compose_label_from_item(itemId=None, offset=0, zpl_templ_file=''):
    """Compose a label in ZPL language"""
    try:
        pro = Product.get(reference=itemId.reference)
    except DoesNotExist:
        return None

    zpl_template = read_printer_template(zpl_templ_file)

    try:
        label_datas = {
            'offset': offset,
            'reference': itemId.reference or '-',
            'libelle1': pro.designation[:30],
            'libelle2': pro.designation[30:60],
            'gtin': "{0:014}".format(int(itemId.contenant)),
            'cip': "{0:013}".format(int(itemId.contenant)),
            'serial': '',
            'ucd': itemId.info or '',
            'expiry': itemId.tperemp,
            'batch': itemId.lot or '',
            'quantity': itemId.qte_dem or 0,
            'fraction': itemId.fraction or 100
        }
        return zpl_template.format(**label_datas)
    except TypeError:
        return ''


def _printer_configuration(riedl=None) -> tuple:
    """
    Retrieve defautl printer associate to the RIEDL
    """
    try:
        cfg = RawConfig(riedl)
        printer_number = cfg.read('k_rdl_printer').value
    except Exception:
        printer_number = 1

    try:
        printer_key = "k_rdl_printer_adr_%i" % printer_number
        name_key = "k_rdl_printer_name_%i" % printer_number
        offset_key = "k_rdl_printer_offset_%i" % printer_number
        template_key = "k_rdl_tk_template_%i" % printer_number
        cfg = RawConfig('TOUS')

        pr_addr = cfg.read(printer_key).value
        pr_name = cfg.read(name_key).value
        pr_offset = cfg.read(offset_key).value
        pr_template = cfg.read(template_key).value
    except Exception as ex:
        logger.error('error retrieve printer conf %s' % repr(ex))
        raise RiedlPrinterException("Error to retrieve printer configuration!")
    return pr_addr, pr_name, pr_offset, pr_template


def _send_to_printer(printer_address=None, printer_content=None):
    """Send ZPL content to the printer"""
    try:
        printer_url = urlparse(printer_address)
        host_port = printer_url.netloc.split(':')
        logger.info("Send to the printer %s:%s" % (host_port[0], host_port[1]))
        s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        # TODO: add timeout as parameter
        s.settimeout(5)
        s.connect((host_port[0], int(host_port[1])))
        s.send(bytes(printer_content, "utf-8"))
        s.close()
    except Exception as e:
        logger.error("Printer error: %s" % str(e))
        raise RiedlPrinterException("Error to send label to printer configuration!" + str(e))
    return True


@riedl_blueprint.route('command_sorties', methods=['GET'])
@jwt_required()
def get_command():
    return "1"


@riedl_blueprint.route('/command_sorties', methods=['POST'])
@jwt_required()
def update_commande():
    # REcuperation de l'id de la commande

    args = json.loads(request.data)
    liste = args['liste']
    priority = args.get('priority', 3)
    exit_out = args.get('exit', 1)
    res = {
        'liste': liste,
    }

    # Récupération de la liste pour savoir sur quel RIEDL la lancer
    try:
        lst = FListe.get(
            FListe.mode == TypeListe.Output.value,
            FListe.liste == liste,
            FListe.zone_deb == 'RIEDL')

        if lst.selectionne:
            # List already launch, we reject execution
            return {'message': 'list %s already launch for unload!' % (liste,)}, HTTP_400_BAD_REQUEST

        if not lst.interface:
            # Generate a new counter
            lst.interface = get_counter("RIEDL_ORDERCODE")

        if not lst.id_plateau:
            # Retrieve prefix
            tray_prefix = ""
            try:
                cfg = RawConfig('TOUS')
                tray_prefix = cfg.read('k_rdl_prefix_container').value
            except Exception:
                tray_prefix = ""
            # Retrieve code
            lst.id_plateau = '%s%010d' % (tray_prefix, int(get_counter("RIEDL_CONTAINER")))
        lst.no_pilulier = exit_out
        lst.pos_pilulier = priority
        lst.ddeb = datetime.datetime.now()
        lst.etat = EtatListe.EnCours.value
        lst.username = session['username']
        lst.selectionne = 1
        lst.save()
    except DoesNotExist:
        logger.error('list %s for rield does not exists' % (liste,))
        return {'message': 'list %s for rield does not exists' % (liste,)}, HTTP_404_NOT_FOUND

    mag = None
    try:
        mag = Magasin.get(Magasin.type_mag == lst.zone_fin, Magasin.eco_type == 'L')
    except DoesNotExist:
        # TODO: THis case is not possible
        logger.error("No warehouse found for %s" % lst.zone_fin)

    try:
        itms = FItem.select(FItem).where(FItem.liste == lst.liste, FItem.mode == lst.mode)
        # for each items, check if quantity is a multiple of the full box
        for it in itms:
            stk = Stock.select(Stock).where(
                Stock.magasin == mag.mag, Stock.reference == it.reference,
                Stock.bloque == 0, Stock.quantite == Stock.capa).order_by(Stock.date_peremption)
            for st in stk:
                if is_multiple(it.qte_dem, st.quantite):
                    logger.info("Quantity for %s is a multiple (%i, %i)" % (
                        it.reference, int(it.qte_dem), int(st.quantite)))
                    it.num_face = 1
                    it.save()
                else:
                    logger.info("Quantity for %s is a not multiple (%i, %i)" % (
                        it.reference, int(it.qte_dem), int(st.quantite)))
                break  # parse only the first stock to retrieve quantity

    except DoesNotExist:
        logger.error("parse item return an error")

    res['riedl'] = lst.zone_fin
    res['exit'] = exit_out
    res['priority'] = priority

    # Check if we need to print the container label
    try:
        cfg = RawConfig('TOUS')
        if cfg.read('k_rdl_print_container').value == '1':
            # Print the label
            pr_addr, pr_name, pr_offset = printer_riedl_configuration(lst.zone_fin)
            logger.info("Print the container label for %s on %s" % (lst.id_plateau, pr_name))
            label = compose_container_label(lst, pr_offset)
            send_to_printer(pr_addr, label)
    except RiedlPrinterException:
        logger.error("Error when retrieve printer configuration!")
    except PrinterException:
        logger.error("send to printer failed !")
    except Exception:
        logger.error("Config Key k_rdl_print_container cannot be retrieve")

    try:
        # We launch teh standard list
        _launch_liste(lst.zone_fin, lst)
        log_riedl(
            session['username'], 'out_send_liste',
            'Lancement de la liste %s (%s)' % (lst.liste, lst.pk)
        )
    except ApiRiedlException:
        logger.error('dispatcher URL for rield %s does not exists' % (lst.zone_fin,))
        return {'message': 'dispatcher URL for rield %s does not exists' % (lst.zone_fin,)}, HTTP_404_NOT_FOUND

    return res


def _launch_liste(r_poste, list_obj):
    # Récupération de l'URL du dispatcher
    try:
        cfg = Config.get(poste=r_poste, cle='cfg', propriete='k_rdl_dispatcher_url')
    except DoesNotExist:
        logger.error('dispatcher URL for rield %s does not exists' % (r_poste,))
        raise ApiRiedlException('dispatcher URL for rield %s does not exists' % (r_poste,))

    # Envoi de la requete avec requests
    try:
        url = cfg.value
        if url.endswith('/'):
            url = url[:-1]
        requests.post('%s/RiedlListUpdate' % (url,), json={'id': list_obj.pk}, timeout=10)
    except requests.exceptions.RequestException as e:
        logger.error('Call dispatcher  %s failed' % (cfg.value,))
        return {'message': 'Call dispatcher  %s failed' % (cfg.value,), 'trace': str(e)}, HTTP_404_NOT_FOUND


class ApiRiedlException(Exception):
    pass
