import json
import logging
import datetime
import math
from random import random

from flask import Blueprint, request, session
from flask_jwt_extended import jwt_required
from median.models import Product, Ucd, Stock, Historique, Gpao, Magasin, Adresse, Seuil, FItem, FListe, Cip
from median.constant import PatientGlobal
from peewee import DoesNotExist, fn, JOIN

from common.status import HTTP_200_OK, HTTP_400_BAD_REQUEST

from common.status import HTTP_404_NOT_FOUND, HTTP_503_SERVICE_UNAVAILABLE

from common.rest import WebResourceException, WebResource

product_blueprint = Blueprint('product', __name__)

logger = logging.getLogger('median')


@product_blueprint.route('unitunload', methods=['POST'])
@jwt_required()
def unload_unit_doses():
    args = request.json
    logger.info(args)
    logger.info("Sortie unitaire a l'adresse [%s]: quantite de: %s" % (
        args.get('emplacement'), args.get('qte', 0)))

    pk = args.get('pk', 0)
    qty_to_unload = int(args.get('qte', 0))
    service = args.get('service', False)
    magasin = args.get('magasin', False)
    stock_data = {}

    if not service:
        logger.error("unload_unit_doses service requis")
        return {'message': "ward is required!"}, HTTP_400_BAD_REQUEST

    if not magasin:
        logger.error("unload_unit_doses service requis")
        return {'message': "warehouse is required!"}, HTTP_400_BAD_REQUEST

    if qty_to_unload == 0:
        logger.error("unload_unit_doses quantite a zero")
        return {'message': "quantity cannot be zero"}, HTTP_400_BAD_REQUEST

    # recupération du mouvement de stock
    try:
        stk = Stock.get_by_id(pk)
    except DoesNotExist:
        logger.error('stock line not found: %s' % str(pk))
        return {'message': 'stock line not found: %s' % str(pk)}, HTTP_400_BAD_REQUEST

    stock_data['ref'] = stk.reference
    stock_data['adresse'] = stk.adresse
    stock_data['qty_line'] = stk.quantite
    stock_data['lot'] = stk.lot
    stock_data['expiry'] = stk.date_peremption
    stock_data['ucd'] = stk.ucd
    stock_data['cip'] = stk.cip
    stock_data['serial'] = stk.serial
    stock_data['fraction'] = stk.fraction
    stock_data['mag'] = stk.magasin
    stock_data['contenant'] = stk.contenant
    stock_data['qte_blister'] = stk.qte_prod_blister

    adr = Adresse.get_or_none(adresse=stk.adresse)

    if qty_to_unload > stk.quantite:
        logger.error("unload_unit_doses quantite superieur a quantie en stock sur %s" % args.get('emplacement'))
        return {'message': "quantity cannot be greather than stock quantity"}, HTTP_400_BAD_REQUEST

    final_quantity = int(stock_data['qty_line']) - int(qty_to_unload)
    # If quantity to unload equal to the stock line, we remove it
    if qty_to_unload == stk.quantite:
        logger.debug('Deletion of the stock line')
        Stock.delete().where(Stock.pk == pk).execute()

        nb_lignes_stocks = (Stock.select(Stock.quantite).where(Stock.adresse == stk.adresse))

        if nb_lignes_stocks.count() == 0:
            logger.debug('''Update address, address become free: "%s"''' % (stk.adresse))
            Adresse.update({Adresse.etat: 'L'}).where(Adresse.adresse == stk.adresse).execute()
    else:
        logger.info('change stock line quantity for pk: "%s", qty: "%s"' % (pk, qty_to_unload))
        qte_blister_bac = 0
        nb_dose_cut = 0

        if adr.format.startswith("TIROIR"):
            logger.info("format is a tiroir, we need to compute blister quantity")

            # If tiroir, we need to have a multiple of blister
            if final_quantity % int(stock_data['qte_blister']) != 0:
                logger.error("final quantity must be a multiple of blister")
                return {'message': "final quantity must be a multiple of blister"}, HTTP_400_BAD_REQUEST

            qte_blister_bac = math.ceil(final_quantity / int(stock_data['qte_blister']))
            nb_dose_cut = int(qty_to_unload) % int(stock_data['qte_blister'])
            if (nb_dose_cut == 0):
                nb_dose_cut = stock_data['qte_blister']

        Stock.update({
            Stock.quantite: final_quantity,
            Stock.qte_blister_bac: qte_blister_bac,
            Stock.nb_dose_cut: nb_dose_cut
        }).where((Stock.pk == pk)).execute()

    qte_tot = (Stock.select(fn.SUM(Stock.quantite)).where(Stock.reference == stock_data['ref']))

    logger.debug('Ecriture dans la table historique')
    Historique.create(
        chrono=datetime.datetime.now(),
        reference=stock_data['ref'],
        adresse=stock_data['adresse'],
        quantite_mouvement=qty_to_unload,
        quantite_totale=qte_tot,
        service=service,
        type_mouvement='SOR',
        lot=stock_data['lot'],
        pmp=0,
        date_peremption=stock_data['expiry'],
        contenant=stock_data['contenant'],
        poste='MEDIANWEB',
        ucd=stock_data['ucd'],
        fraction=stock_data['fraction'],
        utilisateur=session['username'],
        serial=stock_data['serial'],
        commentaire='Manual',
        magasin=stock_data['mag'],
        ipp=PatientGlobal.Ipp.value,
        sejour=PatientGlobal.Sejour.value
    )

    current_mag = Magasin.get(mag=magasin)

    logger.info("GPAO: Ajout d'un mouvement de type sortie")
    Gpao.create(
        chrono=datetime.datetime.now(),
        poste=current_mag.type_mag,
        sequence='MEDIANWEB',
        user=session['username'],
        etat='A',
        ref=stock_data['ref'],
        qte=qty_to_unload,
        lot=stock_data['lot'],
        type_mvt='S',
        dest=service,
        tperemp=stock_data['expiry'],
        fraction=stock_data['fraction'],
        id_robot=current_mag.id_robot,
        id_zone=current_mag.id_zone,
        ucd=stock_data['ucd'],
        cip=stock_data['cip'],
        contenant=stock_data['contenant'],
        magasin=current_mag.mag,
        ipp=PatientGlobal.Ipp.value,
        sejour=PatientGlobal.Sejour.value
    )

    logger.info('Unit unload OK')

    return {'message': "OK"}, HTTP_200_OK


@product_blueprint.route('', methods=['POST'])
@jwt_required()
def create_product():
    args = json.loads(request.data)
    _ref = args['ref']
    _designation = args['designation']
    _com_med = args['com_med']

    logger.info('Création de produit, ref: "%s"' % (_ref))

    if _ref:
        if not _isProductReferenceAvailable(_ref):
            logger.warning('La référence existe déjà')
            return {'status': 'Error', 'ref': _ref}
    else:
        logger.info('On crée une nouvelle référence')
        _ref = _generateProductReference()

    logger.info('On crée le produit en base, ref: "%s", designation: "%s",'
                ' com_med: "%s"', (_ref, _designation, _com_med))
    Product.create(reference=_ref,
                   designation=_designation,
                   com_med=_com_med)

    return {'status': 'Success', 'ref': _ref}


@product_blueprint.route('<string:ref_pk>', methods=['GET'])
@jwt_required()
def get_product(ref_pk):
    logger.info('read drug: ' + ref_pk)
    try:
        p = Product.get(Product.pk == ref_pk)
        ref = p.reference
        stock_info = _GetProductStockInfo(ref)
        stock_by_magasin = _getStockByMagasin(ref)
        ucd_cip_list = _getProductUCDAndCIPListJSON(ref)

    except DoesNotExist:
        logger.error("""Le produit n'existe pas => Réf: "%s" """ % (ref_pk))
        return {'message': 'Product does not exist'}, HTTP_404_NOT_FOUND
    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_503_SERVICE_UNAVAILABLE

    return {
        'id': p.pk,
        'reference': p.reference,
        'designation': p.designation,
        'dci': p.dci,
        'ucd': p.ucd,
        'desig_bis': p.desig_bis,
        'com_med': p.com_med if p.com_med is not None else '',
        'risk': p.risque,
        'narcotic': p.stup,
        'external': p.externe,
        'stock': stock_info['stock'],
        'stock_by_mag': stock_by_magasin,
        'ucd_cip_list': ucd_cip_list,
        'tiroir_bas': p.tiroir_bas
    }


@product_blueprint.route('<string:ref_pk>', methods=['PUT'])
@jwt_required()
def update_product(ref_pk):
    """Update drugs"""
    args = request.form
    try:
        r = WebResource()
        designation = r.mandatory(args.get('designation', ''), "Drug name is mandatory")
        com_med = args['com_med'] if 'com_med' in args else ''
        ucd = args['ucd'] if 'ucd' in args else ''
        risk = r.boolean(args['risk'] if 'risk' in args else 'false')
        narco = r.boolean(args['narcotic'] if 'narcotic' in args else 'false')
        extern = r.boolean(args['external'] if 'external' in args else 'false')
    except WebResourceException as e:
        logger.warning(e.message)
        return {'message': e.message}, HTTP_400_BAD_REQUEST

    logger.info('Drug editing, ref: "%s"' % ref_pk)

    if not ref_pk:
        logger.warning('Missing drug reference')
        return {'message': 'Référence manquante!'}, HTTP_400_BAD_REQUEST

    if len(designation) > 100:
        logger.warning('drug description too long!')
        return {'message': 'drug description too long!'}, HTTP_400_BAD_REQUEST

    try:
        p = Product.get(Product.pk == ref_pk)
        ref = p.reference
        n = (Product.update({
            Product.designation: designation,
            Product.com_med: com_med,
            Product.risque: risk,
            Product.stup: narco,
            Product.ucd: ucd,
            Product.externe: extern
        }).where(Product.pk == ref_pk))
        n.execute()

    except DoesNotExist:
        logger.error('Ce produit n\'existe pas! Réf: "%s"' % ref_pk)
        return {'message': 'Product does not exist'}, HTTP_404_NOT_FOUND
    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_503_SERVICE_UNAVAILABLE

    logger.info("Drug modification with success. ref: %s" % ref)
    return {'result': 'OK', 'reference': ref, 'method': 'put'}


def _GetProductStockInfo(ref):
    st = (Product
          .select(Product, fn.SUM(Stock.quantite).alias('stock'))
          .join(Stock, on=(Product.reference == Stock.reference))
          .where(Product.reference == ref)
          .group_by(Product))

    if not st:
        stock = 0
    else:
        stock = st[0].stock

    s = (Product
         .select(Product,
                 fn.SUM(Seuil.stock_mini).alias('stock_mini'),
                 fn.SUM(Seuil.stock_maxi).alias('stock_maxi'))
         .join(Seuil, on=(Product.reference == Seuil.reference))
         .group_by(Product))

    return {
        'stock': stock,
        'stock_mini': s and s[0].stock_mini or '-',
        'stock_maxi': s and s[0].stock_maxi or '-'
    }


def _getStockByMagasin(ref):
    mags = (Magasin
            .select(Magasin.mag, Magasin.type_mag, fn.SUM(Stock.quantite).alias('stock'), Magasin.libelle)
            .join(Stock, JOIN.LEFT_OUTER, on=((ref == Stock.reference) & (Magasin.mag == Stock.magasin)))
            # .where(Magasin.eco_type == 'C')
            .group_by(Magasin.mag).order_by(Magasin.eco_type))

    _out = []
    for m in mags:
        if m.type_mag != 'RETOUR':
            q = (FListe.select((fn.SUM(FItem.qte_dem - FItem.qte_serv)).alias('cmd')).join(
                FItem, on=(
                    (FListe.liste == FItem.liste)
                    & (FListe.mode == FItem.mode)
                    & (FItem.reference == ref)))
                 .where(  # noqa
                (FListe.mode == 'E')
                & (FListe.etat == 'V')
                & (FListe.zone_fin == m.type_mag))
                 .group_by(FListe.zone_fin))

            cmd = q and q[0].cmd or 0
            if cmd < 0:
                cmd = 0
            _out.append({
                'mag': m.mag,
                'type_mag': m.type_mag,
                'libelle': m.libelle,
                'stock': m.stock or 0,
                'commande': cmd or 0,
            })

    return _out


def _generateProductReference():
    while True:
        rand_ref = 'P' + str(round(random() * 100000000))
        if _isProductReferenceAvailable(rand_ref):
            break
    return rand_ref


def _isProductReferenceAvailable(ref):
    if (Product.select(Product.reference)
               .where(Product.reference == ref).count()) == 0:
        return True
    return False


def _getProductUCDAndCIPListJSON(ref):
    _out = []

    ucd_list = (
        Ucd.select(Ucd.ucd)
        .distinct()
        .where(Ucd.reference == ref)
        .order_by(Ucd.ucd)
    )

    for u in ucd_list:
        ucd_cip_list = Cip.select(Cip.ucd, Cip.cip).distinct().where(Cip.ucd == u.ucd)
        logger.info('Lines : %s.' % len(ucd_cip_list))
        _o = {
            'label': u.ucd,
            'header': 'UCD',
            'children': []
        }
        for c in ucd_cip_list:
            _c = {
                'label': c.cip,
                'header': 'CIP',
                'children': []
            }
            cip_version = Cip.select(Cip).where(Cip.ucd == u.ucd, Cip.cip == c.cip)
            for v in cip_version:
                _v = {
                    'label': v.dossier,
                    'box': v.qt_boite,
                    'pass': v.qt_pass,
                    'blister': v.qt_blister,
                    'header': 'Version',
                    'icon': 'file',
                    'children': []
                }
                _c['children'].append(_v)
            _o['children'].append(_c)
        _out.append(_o)

    return _out
