import logging
import traceback

from flask import Blueprint, json, request, session
from flask_jwt_extended import jwt_required
from median.models import Patient, Sejour, Prescription, PrescriptionItem, Product
from peewee import DoesNotExist, JOIN, fn, Case
from common.status import (
    HTTP_200_OK, HTTP_204_NO_CONTENT, HTTP_500_INTERNAL_SERVER_ERROR,
    HTTP_400_BAD_REQUEST, HTTP_409_CONFLICT, HTTP_404_NOT_FOUND)
from common.database import mysql_server_version

patients_blueprint = Blueprint('patients', __name__)

logger = logging.getLogger('median')


def patient_to_dict(patient):
    return {
        'pk': patient.pk,
        'ipp': patient.ipp,
        'nom': patient.nom,
        'prenom': patient.prenom,
        'date_maj': patient.date_maj,
        'date_naissance': patient.date_naissance,
        'sexe': patient.sexe,
        'nom_jeune_fille': patient.nom_jeune_fille,
        'derniere_date_entree': getattr(patient, 'date_entree', None),
        'derniere_date_sortie': getattr(patient, 'date_sortie', None),
        'derniere_chambre': getattr(patient, 'chambre', None),
        'dernier_lit': getattr(patient, 'lit', None),
        'sejours': [sejour_to_dict(sejour) for sejour in getattr(patient, 'sejours', [])],
        'prescriptions': [prescription_to_dict(prescription) for prescription in getattr(patient, 'prescriptions', [])]
    }


def sejour_to_dict(sejour):
    return {
        'pk': sejour.pk,
        'ipp': sejour.ipp,
        'sejour': sejour.sejour,
        'date_entree': sejour.date_entree,
        'uf_hosp': sejour.uf_hosp,
        'uf_alit': sejour.uf_alit,
        'date_sortie': sejour.date_sortie,
        'chambre': sejour.chambre,
        'lit': sejour.lit
    }


def prescription_to_dict(prescription):
    line = prescription.line if hasattr(prescription, 'line') else None
    return {
        'ent_pk': prescription.pk,
        'line_pk': line.pk if line else '',
        'sequence': line.sequence if line else '',
        'subsequence': line.sub_sequence if line else '',
        'reference': line.reference if line else '',
        'date_debut': line.date_debut if line else '',
        'date_fin': line.date_fin if line else '',
        'posologie': line.posologie if line else '',
        'libelle': (
            line.product.libelle
            if line and hasattr(line, 'product') and line.product
            else 'patient.prescription.nodesignation'
        ),
        'nom': prescription.nom,
        'prenom': prescription.prenom,
        'nom_jeune_fille': prescription.nom_jeune_fille,
        'sexe': prescription.sexe,
        'chambre': prescription.chambre,
        'sejour': prescription.sejour,
        'ipp': prescription.ipp,
        'lit': prescription.lit,
        'date_naissance': prescription.date_naissance
    }


@patients_blueprint.route('all', methods=['POST'])
@jwt_required()
def get_all():
    try:
        data = json.loads(request.data)
        v_limit = data.get('length', 25)
        v_offset = data.get('page', 0) * v_limit

        # Parametres
        filter = data.get('filter', None)

        criterias = data.get('criterias', {})
        # print('filter criterias:')
        # print(criterias)

        # Calcul de la date la plus récente des entrées
        # Le filtrage se fera ici pour un gain de temps de traitement
        filter_words = filter.split() if filter else []

        filter_conditions = [(Patient.nom.contains(word) | Patient.prenom.contains(word) |
                              Patient.nom_jeune_fille.contains(word) |
                              Patient.ipp.contains(word)) for word in filter_words]

        # MYSQL VERSION CHECK - LATERAL EXIST SINCE MYSQL 8.0.14
        # OLDER VERSION WILL RUN SLOWLY /!\
        if mysql_server_version() >= (8, 0, 14):

            # choix du type de jointure pour exclure ou non les patients sans sejours
            joinType = 'INNER JOIN LATERAL'  # the peewee helper does not exist but this is valid SQL
            if criterias.get('withNoEpisodes'):
                joinType = JOIN.LEFT_LATERAL

            subQuery = Sejour.select(
                Sejour.ipp.alias('ipp'), Sejour.date_sortie.alias('date_sortie'),
                Sejour.date_entree.alias('date_entree'),
                Sejour.chambre.alias('chambre'), Sejour.lit.alias('lit')
                ).where(
                    Sejour.ipp == Patient.ipp
                ).order_by(
                    Sejour.date_entree.desc(), Sejour.pk.desc()
                ).limit(1).alias('last_dates')

            query = Patient.select(
                    Patient, subQuery.c.date_entree, subQuery.c.date_sortie, subQuery.c.chambre, subQuery.c.lit
                ).join(
                    subQuery, joinType, on=True
                ).where(
                    *filter_conditions if filter_conditions else (True,)
                ).order_by(
                    subQuery.c.date_sortie.is_null(False),  # Put null dates on top
                    -(subQuery.c.date_sortie == '0000-00-00 00:00:00'),  # Put default (0000...) dates on top
                    -subQuery.c.date_sortie,  # Order by date_sortie descending
                    +Patient.nom,
                    +Patient.prenom
                )

        else:
            # QUERY WITHOUT LATERAL

            # choix du type de jointure pour exclure ou non les patients sans sejours
            joinType = JOIN.INNER  # the peewee helper does not exist but this is valid SQL
            if criterias.get('withNoEpisodes'):
                joinType = JOIN.LEFT_OUTER

            max_tentree_subquery = Sejour.select(
                Sejour.ipp.alias('ipp'),
                fn.MAX(
                    Case(
                        None,
                        (
                            (Sejour.date_entree.is_null(), '9999-01-01 00:00:00'),
                            (Sejour.date_entree == '0000-00-00 00:00:00', '9999-01-01 00:00:00'),
                        ),
                        Sejour.date_entree
                    )
                ).alias('max_tentree')
            ).group_by(Sejour.ipp).alias('t2')

            latest_entry_subquery = Sejour.select(
                Sejour.ipp, Sejour.date_entree, Sejour.date_sortie, Sejour.sejour, Sejour.pk
                ).join(
                    max_tentree_subquery,
                    JOIN.INNER,
                    on=(
                        (Sejour.ipp == max_tentree_subquery.c.ipp) &
                        (
                            (Sejour.date_entree == max_tentree_subquery.c.max_tentree) |
                            ((Sejour.date_entree.is_null()) &
                             (max_tentree_subquery.c.max_tentree == '9999-01-01 00:00:00')) |
                            ((Sejour.date_entree == '0000-00-00 00:00:00') &
                             (max_tentree_subquery.c.max_tentree == '9999-01-01 00:00:00'))
                        )
                    )
                ).where(
                    Sejour.pk == (
                        Sejour.select(fn.MAX(Sejour.pk))
                        .where(
                            (Sejour.ipp == max_tentree_subquery.c.ipp) &
                            (
                                (Sejour.date_entree == max_tentree_subquery.c.max_tentree) |
                                ((Sejour.date_entree.is_null()) &
                                 (max_tentree_subquery.c.max_tentree == '9999-01-01 00:00:00')) |
                                ((Sejour.date_entree == '0000-00-00 00:00:00') &
                                 (max_tentree_subquery.c.max_tentree == '9999-01-01 00:00:00'))
                            )
                        )
                    )
                ).alias('latest_entry')

            query = Patient.select(
                Patient.pk, Patient.ipp, Patient.nom, Patient.prenom, Patient.nom_jeune_fille,
                latest_entry_subquery.c.x_num_sej, latest_entry_subquery.c.x_tentree, latest_entry_subquery.c.x_tsortie
            ).join(
                latest_entry_subquery,
                joinType,
                on=(Patient.ipp == latest_entry_subquery.c.x_num_ipp)
            ).where(
                *filter_conditions if filter_conditions else (True,)
                ).order_by(
                    latest_entry_subquery.c.x_tsortie.is_null(False),  # Put null dates on top
                    -(latest_entry_subquery.c.x_tsortie == '0000-00-00 00:00:00'),  # Put default (0000...) dates on top
                    -latest_entry_subquery.c.x_tsortie,  # Order by date_sortie descending
                    +Patient.nom,
                    +Patient.prenom
                )

            # print('MYSQL SANS LATERAL')
            # print(query)

        # Utilisation de "objects" pour conserver le champ calculé "max_entry", sinon peewee tente
        # de coller au modele Patient
        patients_list = []
        for patient in query.limit(v_limit).offset(v_offset).objects():
            # exclusion des patients sans sejour si filtre actif
            patients_list.append(patient_to_dict(patient))

        return {'patients': patients_list, 'total': query.count()}, HTTP_200_OK

    except Exception as error:
        logger.error(('Get patients Datatables raised an exception: ', error.args))
        print(traceback.format_exc())
        return {'message': error.args}, HTTP_500_INTERNAL_SERVER_ERROR


@patients_blueprint.route('create', methods=['POST'])
@jwt_required()
def create_patient():
    try:
        if not request.data:
            return {'message': 'No data provided'}, HTTP_400_BAD_REQUEST

        args = json.loads(request.data)

        v_nom_pat = args['nom']
        v_num_ipp = args['ipp']
        v_prenom_pat = args['prenom']
        v_date_maj = args['date_maj']
        v_dnaiss = args['date_naissance']
        v_sexe = args['sexe']
        v_nom_jf = args['nom_jeune_fille']
    except KeyError as e:
        logger.error(f'Patient Create : Key error ({e.args[0]}')
        return {'message': 'patient.error.wrong_property'}, HTTP_400_BAD_REQUEST

    nb_patients = Patient.select().where(Patient.ipp == v_num_ipp).count()
    if nb_patients > 0:
        return {'message': "patient.error.exists_already"}, HTTP_409_CONFLICT

    Patient.create(
        ipp=v_num_ipp,
        nom=v_nom_pat,
        prenom=v_prenom_pat,
        date_maj=v_date_maj,
        date_naissance=v_dnaiss,
        sexe=v_sexe,
        nom_jeune_fille=v_nom_jf)

    return {}, HTTP_204_NO_CONTENT


@patients_blueprint.route('<string:id>', methods=['GET'])
@jwt_required()
def get_patient(id):
    try:
        patient = Patient.get((Patient.pk == id))
        patient.sejours = Sejour.select().where(Sejour.ipp == patient.ipp).order_by(-Sejour.pk)

        patient.prescriptions = Prescription.select(
            Prescription.pk, PrescriptionItem.pk,
            Prescription.nom, Prescription.prenom, Prescription.nom_jeune_fille, Prescription.sexe,
            Prescription.chambre, Prescription.sejour, Prescription.ipp, Prescription.lit, Prescription.date_naissance,
            PrescriptionItem.sequence, PrescriptionItem.sub_sequence,
            PrescriptionItem.reference, PrescriptionItem.date_debut, PrescriptionItem.date_fin,
            PrescriptionItem.posologie, Product.designation.alias('libelle')
            ).join(
                PrescriptionItem, JOIN.INNER,
                on=(
                    (PrescriptionItem.ordre == Prescription.ordre)
                    & (PrescriptionItem.ipp == Prescription.ipp)
                    & (PrescriptionItem.sejour == Prescription.sejour)
                ),
                attr='line'
            ).join(
                Product, JOIN.LEFT_OUTER, on=Product.reference == PrescriptionItem.reference,
                attr='product'
            ).where(
                Prescription.ipp == patient.ipp
            ).order_by(
                -PrescriptionItem.sejour, +PrescriptionItem.sequence, +PrescriptionItem.sub_sequence
            )

        # print(patient.prescriptions)

        return patient_to_dict(patient), HTTP_200_OK
    except DoesNotExist:
        return {'message': 'patient.error.not_found'}, HTTP_400_BAD_REQUEST


@patients_blueprint.route('<string:id>', methods=['PUT'])
@jwt_required()
def update_patient(id):
    try:
        patient = Patient.get(pk=id)
        args = json.loads(request.data)

        for key, value in args.items():
            if hasattr(patient, key):
                setattr(patient, key, value)
            else:
                raise KeyError(f"Attribute {key} does not exist in Patient model")

        patient.save()
    except KeyError as e:
        logger.error(e)
        return {'message': 'patient.error.wrong_property'}, HTTP_400_BAD_REQUEST
    except DoesNotExist:
        return {'message': 'patient.error.not_found'}, HTTP_400_BAD_REQUEST
    except Exception as e:
        return {'message': str(e)}, HTTP_500_INTERNAL_SERVER_ERROR

    return patient_to_dict(patient), HTTP_200_OK


@patients_blueprint.route('<string:id>/delete', methods=['DELETE'])
@jwt_required()
def delete_patient(id):
    """Delete the patient by the primary key"""
    try:
        patient = Patient.get(pk=id)
        logger.info(
            "Delete patient [%s %s] (%i) by %s",
            patient.nom, patient.prenom, patient.pk, session['username']
        )
        patient.delete_instance()
        return {}, HTTP_204_NO_CONTENT
    except DoesNotExist:
        return {'message': 'patient.error.not_found'}, HTTP_400_BAD_REQUEST
    except Exception as e:
        return {'message': str(e)}, HTTP_500_INTERNAL_SERVER_ERROR


@patients_blueprint.route('<string:id>/sejours/', methods=['POST'])
@jwt_required()
def add_sejour(id):
    try:
        patient = Patient.get(pk=id)

        if not request.data:
            logger.error('Patient - Add Sejour : No data provided')
            return {'message': 'sejour.error.no_data_provided'}, HTTP_400_BAD_REQUEST

        try:
            args = json.loads(request.data)
        except json.JSONDecodeError:
            logger.error('Patient - Add Sejour : Invalid JSON')
            return {'message': 'sejour.error.invalid_json'}, HTTP_400_BAD_REQUEST

        Sejour.create(
            ipp=patient.ipp,
            sejour=args['sejour'],
            date_entree=args['date_entree'],
            date_sortie=args.get('date_sortie', None),
            uf_hosp=args['uf_hosp'],
            uf_alit=args.get('uf_alit', ''),
            chambre=args['chambre'],
            lit=args['lit']
        )

        patient = Patient.get(pk=id)
        patient.sejours = Sejour.select().where(Sejour.ipp == patient.ipp)
        updated_patient = patient_to_dict(patient)
    except KeyError as e:
        logger.error('Patient - Add Sejour : An error occured -> ', str(e))
        return {'message': 'sejour.error.wrong_property'}, HTTP_400_BAD_REQUEST
    except Exception as e:
        logger.error('Patient - Add Sejour : An error occured -> ', str(e))
        return {'message': str(e)}, HTTP_500_INTERNAL_SERVER_ERROR

    return updated_patient, HTTP_200_OK


@patients_blueprint.route('<string:id>/sejours/<string:sejour_id>', methods=['DELETE'])
@jwt_required()
def delete_sejour(id, sejour_id):
    try:
        patient = Patient.get(pk=id)
        sejour = Sejour.get(pk=sejour_id)

        if patient.ipp != sejour.ipp:  # if the sejour does not belong to the patient, raise an exception
            return {'message': 'sejour.error.corrupted_data'}, HTTP_500_INTERNAL_SERVER_ERROR

        sejour.delete_instance()

    except DoesNotExist:
        return {'message': 'sejour.error.not_found'}, HTTP_400_BAD_REQUEST
    except Exception as e:
        logger.error('Sejour - Delete : An error occured -> ', str(e))
        return {'message': str(e)}, HTTP_500_INTERNAL_SERVER_ERROR

    patient.sejours = Sejour.select().where(Sejour.ipp == patient.ipp)
    return patient_to_dict(patient), HTTP_200_OK


@patients_blueprint.route('/sejours/<string:sejour_id>', methods=['PUT'])
@jwt_required()
def update_sejour(sejour_id):
    try:

        if not request.data:
            logger.error('Sejour - Update : No data provided')
            return {'message': 'sejour.error.no_data_provided'}, HTTP_400_BAD_REQUEST

        args = json.loads(request.data)
        ipp = args.get('ipp')
        new_end_date = args.get('date_end')

        if not ipp or not new_end_date:
            logger.error('Sejour - Update : Missing required fields')
            return {'message': 'sejour.error.missing_fields'}, HTTP_400_BAD_REQUEST

        Patient.get(ipp=ipp)  # will trigger DoesNotExist if the IPP is linked to a patient
        Sejour.get(pk=sejour_id)

        sejour = Sejour.update(date_sortie=new_end_date).where((Sejour.pk == sejour_id) & (Sejour.ipp == ipp))
        sejour.execute()

        # patient.sejours = Sejour.select().where(Sejour.ipp == patient.ipp)
    except KeyError as e:
        logger.error(e)
        return {'message': 'sejour.error.wrong_property'}, HTTP_400_BAD_REQUEST
    except DoesNotExist:
        return {'message': 'sejour.error.not_found'}, HTTP_404_NOT_FOUND
    except Exception as e:
        logger.error('Sejour - Update : An error occured -> ', str(e))
        return {'message': str(e)}, HTTP_500_INTERNAL_SERVER_ERROR

    return {}, HTTP_204_NO_CONTENT
