import json
import logging
import datetime
import operator
from functools import reduce

from flask import Blueprint, request
from flask_jwt_extended import jwt_required

from median.models import Magasin,  FListe, FItem, \
    WorkItemModel, Product

from common.status import HTTP_200_OK

from peewee import fn

inventory_blueprint = Blueprint('inventory', __name__)

logger = logging.getLogger('median')


@inventory_blueprint.route('all', methods=['POST'])
@jwt_required()
def get_all():
    data = json.loads(request.data)
    criterias = data.get('criterias', [])
    equipments = data.get('equipments', [])
    expr = (FListe.mode == 'I')

    if len(criterias) > 0:
        lst = list(map(lambda s: (
            (FListe.liste.contains(s.strip())) |
            (Magasin.libelle.contains(s.strip()))
        ), criterias))

        search = reduce(operator.and_, lst)

        expr = reduce(operator.and_, [expr, search])

    if len(equipments) > 0:
        expr = expr & (FListe.zone_fin << equipments)

    sub_query = WorkItemModel.raw('(SELECT MIN(`t3`.`x_qte_dem` = (SELECT COALESCE(SUM(`t4`.`x_qte_dem`), %s) FROM '
                                  '`f_item_w` AS `t4`'
                                  'WHERE (`t3`.`x_pk` = `t4`.`x_pk_item`))) FROM `f_item` AS `t3` WHERE ('
                                  '`t3`.`x_liste` =`t1`.`x_liste`)) as \'isOk\'', [0])

    inventories = (FListe
                   .select(FListe.liste, Magasin.libelle, FListe.date_creation, Magasin.pk,
                           FListe.pk, sub_query)
                   .join(Magasin, on=FListe.zone_fin == Magasin.type_mag)
                   .where(expr))

    return {
        'list': [
            {
                'pk': item.pk,
                'label': item.liste,
                'isOk': item.isOk,
                'equipment': {
                    'label': item.magasin.libelle,
                    'pk': item.magasin.pk
                },
                'creationDate': datetime.datetime.utcfromtimestamp(item.date_creation.timestamp())

            } for item in inventories
        ]}, HTTP_200_OK


@inventory_blueprint.route('detail/<string:pk>', methods=['POST'])
@jwt_required()
def get_inventory_detail(pk):
    item = (WorkItemModel.select(WorkItemModel.lot, fn.SUM(WorkItemModel.quantite_dem).alias('quantity'))
            .where(WorkItemModel.pk_item == pk)
            .group_by(WorkItemModel.lot))

    return {
        'list': [
            {
                'batch_num': item.lot,
                'quantity': item.quantity,

            } for item in item.objects()
        ],

    }, HTTP_200_OK


@inventory_blueprint.route('<string:pk>', methods=['POST'])
@jwt_required()
def get_inventory(pk):
    criterias = []
    expr = (FListe.pk == pk)

    if len(criterias) > 0:
        lst = list(map(lambda s: (
            (FListe.liste.contains(s.strip())) |
            (Magasin.libelle.contains(s.strip()))
        ), criterias))

        search = reduce(operator.and_, lst)

        expr = reduce(operator.and_, [expr, search])

    inventories_stock = (FItem.select(FItem.reference, FItem.fraction,
                                      FItem.qte_dem.alias('quantity'),
                                      FItem.pk,
                                      Product.designation.alias('designation'),
                                      (WorkItemModel
                                       .select(fn.SUM(WorkItemModel.quantite_dem).alias('quantity'))
                                       .where((WorkItemModel.pk_item == FItem.pk))).alias('quantity_equipment'))
                         .join(FListe, on=FItem.liste == FListe.liste)
                         .switch(FItem)
                         .join(Product, on=Product.reference == FItem.reference)
                         .where(expr))

    return {
        'list': [
            {
                'pk': item.pk,
                'quantity': item.quantity,
                'reference': item.reference,
                'fraction': item.fraction,
                'label': item.designation,
                'quantity_equipment': item.quantity_equipment if item.quantity_equipment is not None else 0,
            } for item in inventories_stock.objects()
        ],

    }, HTTP_200_OK
