import json
import logging
import operator
import os
import uuid

from functools import reduce
from flask import Blueprint, request, send_file
from flask_jwt_extended import jwt_required
from median.models import Patient, Service, Product, Gpao
from peewee import DoesNotExist, JOIN, fn
from ressources.astus.utils import generate_excel_writer, add_excel_worksheet, close_excel_writer
from ressources.blueprint.GpaoMvt import GpaoMvt

interface_blueprint = Blueprint('interface', __name__)

logger = logging.getLogger('median')


def getExpression(v_date_debut, v_date_fin, v_search, v_mvts, v_state, v_robots, v_buffers):
    mvts = [GpaoMvt.INPUT.value, GpaoMvt.OUTPUT.value,
            GpaoMvt.STOCK_LVL.value, GpaoMvt.COMPLEMENT.value,
            GpaoMvt.LOST_OR_BACK.value, GpaoMvt.REPLENISHMENT.value]

    if (v_search is None) | (v_search == ''):
        v_search_list = []
    else:
        v_search_list = json.loads(v_search.replace("'", "\""))

    expr_list = [(Gpao.chrono >= v_date_debut),
                 ((v_date_fin is None) | (Gpao.chrono <= v_date_fin)),
                 (Gpao.type_mvt != 'S') | (Gpao.etat != 'N'),
                 Gpao.type_mvt << mvts]

    if len(v_mvts) > 0:
        expr_list.append(Gpao.type_mvt << v_mvts)

    if len(v_state) > 0:
        expr_list.append(Gpao.etat << v_state)

    if any(v_robots):
        expr_list.append(Gpao.id_robot << v_robots)

    if any(v_buffers):
        expr_list.append(Gpao.id_zone << v_buffers)

    andexpr = reduce(operator.and_, expr_list)

    if len(v_search_list) > 0:
        lst = list(map(lambda s: (
            (Product.reference.contains(s.strip())) |
            (Product.designation.contains(s.strip())) |
            (Service.libelle.contains(s.strip())) |
            (Service.code.contains(s.strip())) |
            (Gpao.id_robot.contains(s.strip()))

        ), v_search_list))

        search = reduce(operator.and_, lst)

        expr = reduce(operator.and_, [andexpr, search])
    else:
        expr = andexpr

    return expr


def get_all_interface(args):
    v_date_debut = args['date_debut']
    v_date_fin = args['date_fin']
    v_search = args['search']
    v_mvts = args.get("mvts", [])
    v_state = args.get("states", [])
    v_robots = args.get("robots", [])
    v_buffers = args.get("buffers", [])
    return Gpao.select(Gpao.chrono, Product.reference, Gpao.pk,
                       Product.designation, Service.code, Service.libelle,
                       Gpao.type_mvt, Gpao.id_robot, Gpao.qte, Gpao.id_zone,
                       Gpao.etat) \
        .join(Product,
              on=(Product.reference == Gpao.ref)) \
        .switch(Gpao) \
        .join(Service, on=Service.code == Gpao.dest) \
        .where(getExpression(v_date_debut=v_date_debut,
                             v_search=v_search,
                             v_date_fin=v_date_fin,
                             v_mvts=v_mvts,
                             v_state=v_state,
                             v_robots=v_robots,
                             v_buffers=v_buffers)) \
        .order_by(Gpao.chrono)


def get_interface_by_pk(pk):
    return Gpao.select(
        Gpao.chrono,
        Product.reference, Gpao.pk,
        Product.designation, Service.code, Service.libelle,
        Gpao.type_mvt, Gpao.id_robot, Gpao.qte,
        Gpao.etat, Gpao.ipp, Gpao.sejour, Gpao.tenvoi,
        Gpao.item_wms, Patient.prenom, Patient.nom) \
        .join(Product,
              on=(Product.reference == Gpao.ref)) \
        .switch(Gpao) \
        .join(Service, on=Service.code == Gpao.dest) \
        .switch(Gpao) \
        .join(Patient, JOIN.LEFT_OUTER, on=Patient.ipp == Gpao.ipp) \
        .where(Gpao.pk == pk) \
        .get()


def get_obj_interface(i):
    return {
        'pk': i.pk,
        'chrono': i.chrono,
        'state': i.etat,
        'type_mvt': i.type_mvt,
        'buffer': i.id_zone,
        'productReference': i.product.reference if hasattr(i, 'product') else None,
        'productLabel': i.product.designation if hasattr(i, 'product') else None,
        'wardCode': i.service.code if hasattr(i, 'service') else None,
        'wardLabel': i.service.libelle if hasattr(i, 'service') else None,
        'id_robot': i.id_robot,
        'qte': i.qte,
    }


@interface_blueprint.route('states', methods=['POST'])
@jwt_required()
def get_states_interface():
    args = json.loads(request.data)

    v_date_debut = args['date_debut']
    v_date_fin = args['date_fin']
    v_search = args['search']
    v_robots = args.get("robots", [])
    v_buffers = args.get("buffers", [])
    v_mvts = args.get("mvts", [])

    try:
        states = Gpao.select(Gpao.etat,
                             fn.Count(Gpao.type_mvt).alias('total')) \
            .join(Product,
                  on=(Product.reference == Gpao.ref)) \
            .switch(Gpao) \
            .join(Service, on=Service.code == Gpao.dest) \
            .where(getExpression(v_date_debut=v_date_debut,
                                 v_search=v_search,
                                 v_date_fin=v_date_fin,
                                 v_mvts=v_mvts,
                                 v_state=[],
                                 v_robots=v_robots,
                                 v_buffers=v_buffers)) \
            .group_by(Gpao.etat)

        return {'list': [
            {
                'state': state.etat,
                'total': state.total
            } for state in states
        ]}
    except Exception as error:
        logger.error(('get_states_interface raised an exception: ', error.args))
        return {'list': [], }


@interface_blueprint.route('buffers', methods=['POST'])
@jwt_required()
def get_buffers_interface():
    args = json.loads(request.data)

    v_date_debut = args['date_debut']
    v_date_fin = args['date_fin']
    v_search = args['search']
    v_mvts = args.get("mvts", [])
    v_state = args.get("states", [])
    v_robots = args.get("v_robots", [])

    try:
        buffers = Gpao.select(Gpao.id_zone,
                              fn.Count(Gpao.id_zone).alias('total')) \
            .join(Product,
                  on=(Product.reference == Gpao.ref)) \
            .switch(Gpao) \
            .join(Service, on=Service.code == Gpao.dest) \
            .where(getExpression(v_date_debut=v_date_debut,
                                 v_search=v_search,
                                 v_date_fin=v_date_fin,
                                 v_mvts=v_mvts,
                                 v_state=v_state,
                                 v_robots=v_robots,
                                 v_buffers=[]
                                 )) \
            .group_by(Gpao.id_zone)

        return {'list': [
            {
                'buffer': buffer.id_zone,
                'total': buffer.total
            } for buffer in buffers
        ]}
    except Exception as error:
        logger.error(('get_buffers_interface raised an exception: ', error.args))
        return {'list': [], }


@interface_blueprint.route('robots', methods=['POST'])
@jwt_required()
def get_robots_interface():
    args = json.loads(request.data)

    v_date_debut = args['date_debut']
    v_date_fin = args['date_fin']
    v_search = args['search']
    v_mvts = args.get("mvts", [])
    v_state = args.get("states", [])
    v_buffers = args.get("buffers", [])

    try:
        robots = Gpao.select(Gpao.id_robot,
                             fn.Count(Gpao.id_robot).alias('total')) \
            .join(Product,
                  on=(Product.reference == Gpao.ref)) \
            .switch(Gpao) \
            .join(Service, on=Service.code == Gpao.dest) \
            .where(getExpression(v_date_debut=v_date_debut,
                                 v_search=v_search,
                                 v_date_fin=v_date_fin,
                                 v_mvts=v_mvts,
                                 v_state=v_state,
                                 v_robots=[],
                                 v_buffers=v_buffers
                                 )) \
            .group_by(Gpao.id_robot)

        return {'list': [
            {
                'robot': robot.id_robot,
                'total': robot.total
            } for robot in robots
        ]}
    except Exception as error:
        logger.error(('get_type_mvts_interface raised an exception: ', error.args))
        return {'list': [], }


@interface_blueprint.route('typemvts', methods=['POST'])
@jwt_required()
def get_type_mvts_interface():
    args = json.loads(request.data)

    v_date_debut = args['date_debut']
    v_date_fin = args['date_fin']
    v_search = args['search']
    v_state = args.get("states", [])
    v_robots = args.get("robots", [])
    v_buffers = args.get("buffers", [])
    try:
        type_mvts = Gpao.select(Gpao.type_mvt,
                                fn.Count(Gpao.type_mvt).alias('total')) \
            .join(Product,
                  on=(Product.reference == Gpao.ref)) \
            .switch(Gpao) \
            .join(Service, on=Service.code == Gpao.dest) \
            .where(getExpression(v_date_debut=v_date_debut,
                                 v_search=v_search,
                                 v_date_fin=v_date_fin,
                                 v_mvts=[],
                                 v_state=v_state,
                                 v_robots=v_robots,
                                 v_buffers=v_buffers)) \
            .group_by(Gpao.type_mvt)

        return {'list': [
            {
                'type': mvt.type_mvt,
                'total': mvt.total
            } for mvt in type_mvts
        ]}
    except Exception as error:
        logger.error(('get_type_mvts_interface raised an exception: ', error.args))
        return {'list': [], }


@interface_blueprint.route('count', methods=['POST'])
@jwt_required()
def get_count_interface():
    args = json.loads(request.data)
    try:
        total_stocks_count = get_all_interface(args)
        return {'recordsTotal': total_stocks_count.count(), }
    except Exception as error:
        logger.error(('get_count_interface raised an exception: ', error.args))
        return {'recordsTotal': 0, }


@interface_blueprint.route('', methods=['POST'])
@jwt_required()
def get_interfaces():
    args = json.loads(request.data)

    try:
        v_start = args['start']
        v_length = args['length']
        total_stocks_count = get_all_interface(args)
        paged_emplacements = total_stocks_count.limit(v_length).offset(v_start)

        list_interface = []

        for hist in paged_emplacements:
            list_interface.append(get_obj_interface(hist))

        return {
            'list': list_interface,
        }

    except DoesNotExist:
        logger.error('Get historic items Datatables raised a DoesNotExist exception')
        return {
            'list': [],
        }
    except Exception as error:
        logger.error(('get_interface raised an exception: ', error.args))
        return {
            'list': [],
        }


@interface_blueprint.route('<string:gpao_pk>', methods=['GET'])
@jwt_required()
def get_interface(gpao_pk):
    try:

        i = get_interface_by_pk(gpao_pk)

        return {
            'data': {
                'pk': i.pk,
                'chrono': i.chrono,
                'state': i.etat,
                'type_mvt': i.type_mvt,
                'productReference': i.product.reference if hasattr(i, 'product') else None,
                'productLabel': i.product.designation if hasattr(i, 'product') else None,
                'wardCode': i.service.code if hasattr(i, 'service') else None,
                'wardLabel': i.service.libelle if hasattr(i, 'service') else None,
                'id_robot': i.id_robot,
                'send': i.tenvoi,
                'item_wms': i.item_wms,
                'ipp': i.ipp,
                'sejour': i.sejour,
                'patientFirstName': i.patient.prenom if hasattr(i, 'patient') else None,
                'patientName': i.patient.nom if hasattr(i, 'patient') else None,
                'qte': i.qte,
            },
        }

    except Exception as error:
        logger.error(('get_interface raised an exception: ', error.args))
        return {
            'data': {},
        }


def get_obj_summary(i):
    return {
        'type_mvt': i.type_mvt,
        'productReference': i.product.reference if hasattr(i, 'product') else None,
        'productLabel': i.product.designation if hasattr(i, 'product') else None,
        'id_robot': i.id_robot,
        'qte': i.qte,
    }


def get_all_summary(args):
    v_date_debut = args['date_debut']
    v_date_fin = args['date_fin']
    v_search = args['search']
    v_mvts = args.get("mvts", [])
    v_state = args.get("states", [])
    return Gpao.select(Product.reference,
                       Product.designation,
                       Gpao.type_mvt, Gpao.id_robot, fn.SUM(Gpao.qte).alias('qte')) \
        .join(Product,
              on=(Product.reference == Gpao.ref)) \
        .switch(Gpao) \
        .join(Service, on=Service.code == Gpao.dest) \
        .where(getExpression(v_date_debut=v_date_debut,
                             v_search=v_search,
                             v_date_fin=v_date_fin,
                             v_mvts=v_mvts,
                             v_state=v_state)) \
        .group_by(Product.reference,
                  Product.designation,
                  Gpao.type_mvt, Gpao.id_robot) \
        .order_by(Product.reference)


@interface_blueprint.route('export', methods=['PATCH'])
@jwt_required()
def export():
    data = json.loads(request.data)
    headers = data['translations']
    sheet_names = data['sheetNames']
    name = os.sep.join(
        [os.getcwd(), "tmp_export", "%s.xlsx" % uuid.uuid4()])
    interfaces = get_all_interface(data)
    summaries = get_all_summary(data)
    writer = generate_excel_writer(name)
    add_excel_worksheet(headers=headers, items=interfaces,
                        transform_function=get_obj_interface,
                        writer=writer, work_sheet_name=sheet_names['all'])
    add_excel_worksheet(headers=headers, items=summaries,
                        transform_function=get_obj_summary,
                        writer=writer, work_sheet_name=sheet_names['summary'])
    close_excel_writer(writer=writer)

    return send_file(name, as_attachment=True)
