import json
import operator
from functools import reduce

from flask import Blueprint, request
from flask_jwt_extended import jwt_required

from median.models import Format, Reform, Product
from peewee import JOIN, DoesNotExist
from common.status import HTTP_204_NO_CONTENT

format_blueprint = Blueprint('format', __name__)


@format_blueprint.route('<string:ref_pk>', methods=['PUT'])
@jwt_required()
def update(ref_pk):
    data = json.loads(request.data)

    try:
        reform = (Reform.select(Reform)
                  .join(Format, on=Reform.format == Format.format)
                  .join(Product, on=Reform.reference == Product.reference)
                  .where((Format.pk == data['pk']) & (Product.pk == ref_pk))).get()
    except DoesNotExist:
        product = Product.get(Product.pk == ref_pk)
        reform = Reform()
        reform.format = data['format']
        reform.reference = product.reference

    reform.capacity = float(data['capacity'])
    reform.save()

    return 'success', HTTP_204_NO_CONTENT


@format_blueprint.route('', methods=['GET'])
@jwt_required()
def get_all_formats():
    req = (Format.select(Format.pk, Format.format)
           .order_by(Format.format)
           )

    return {'list': [
        {
            'pk': item.pk,
            'format': item.format,
        } for item in req]
    }, 200


@format_blueprint.route('<string:ref_pk>', methods=['POST'])
@jwt_required()
def get_all(ref_pk):
    data = json.loads(request.data)
    search_list = data.get('criterias', [])
    andexpr = True

    if len(search_list) > 0:
        lst = list(map(lambda s: (
            (Format.format.contains(s.strip()))), search_list))
        search = reduce(operator.and_, lst)
        expr = reduce(operator.and_, [andexpr, search])
    else:
        expr = andexpr

    req = (Format.select(Format.pk, Format.format, Reform.capacity, Reform.ordre, Reform.nb_div)
           .join(Product, JOIN.LEFT_OUTER, on=(Product.pk == ref_pk))
           .switch(Format)
           .join(Reform, JOIN.LEFT_OUTER,
                 on=(Reform.format == Format.format) & (Reform.reference == Product.reference))
           .where(expr)
           .order_by(Format.format)
           )

    return {'list': [
        {
            'pk': item.pk,
            'format': item.format,
            'division': item.reform.nb_div if hasattr(item, 'reform') and item.reform is not None else None,
            'capacity': item.reform.capacity if hasattr(item, 'reform') else None,
            'order': item.reform.ordre if hasattr(item, 'reform') and item.reform is not None else None,
        } for item in req]
    }, 200
