import json
import logging

from flask import Blueprint, request, session
from flask_jwt_extended import jwt_required

from median.models import Cip, User, Gtin, Ucd, Product

from common.status import HTTP_500_INTERNAL_SERVER_ERROR, HTTP_204_NO_CONTENT, HTTP_403_FORBIDDEN, HTTP_200_OK

cip_blueprint = Blueprint('cip', __name__)

auth_profils = ['ECO-DEX', 'REFERENT', 'PHARMACIEN', 'ADMIN']

logger = logging.getLogger('median')


def file_to_base64(cip):
    encoded_string = None
    return encoded_string


@cip_blueprint.route('reference/<string:ref_pk>', methods=['GET'])
@jwt_required()
def get_by_reference(ref_pk):
    try:

        cips = (
            Gtin.select(Gtin.cip).join(Ucd, on=Gtin.ucd == Ucd.ucd)
            .join(Product, on=Product.reference == Ucd.reference)
            .where(Product.pk == ref_pk)
            .group_by(Gtin.cip))

        logger.debug('Lines : %s.' % len(cips))

        return {'data': [{
            'cip': s.cip,
        } for s in cips]}, HTTP_200_OK

    except Exception as error:
        logger.error('Get Ucd raised an exception: ', error.args)
        return {'message': error.args}, HTTP_500_INTERNAL_SERVER_ERROR


@cip_blueprint.route('<string:cip_pk>', methods=['PUT'])
@jwt_required()
def update_cip(cip_pk):
    profil = User.select(User.profil).where((User.pk == session['user_id']) & User.profil << auth_profils).count()

    if profil == 1:
        data = json.loads(request.data)
        qty = data['qtePass']
        cip = Cip.select().where(Cip.pk == cip_pk).get()

        if qty.isnumeric() and cip.dossier == '0':
            cip.qt_pass = qty
            cip.save()
            return {}, HTTP_204_NO_CONTENT
        else:
            return {
                'message': 'ucd_cip.error.index0' if cip.dossier == '0' else 'ucd_cip.error.qty.notnumeric'
                }, HTTP_500_INTERNAL_SERVER_ERROR

    else:
        return {'message': 'ucd_cip.error.profil'
                }, HTTP_403_FORBIDDEN


@cip_blueprint.route('', methods=['POST'])
@jwt_required()
def create_cip():
    profil = User.select(User.profil).where((User.pk == session['user_id']) & User.profil << auth_profils)

    if any(profil):
        data = json.loads(request.data)
        cip = data['cip']
        ucd = data['ucd']

        gtins = Gtin.select(Gtin.pk).where((Gtin.ucd == ucd) & (Gtin.cip == cip['cip']))

        if not any(gtins):
            gtin = Gtin()
            gtin.ucd = ucd
            gtin.cip = cip['cip']
            gtin.qt_blister = cip['qte_blister']
            gtin.qt_pass = cip['qtePass']
            gtin.qt_boite = cip['qte_boite']
            gtin.save()

            return {
            }, HTTP_204_NO_CONTENT
        else:
            return {'message': 'ucd_cip.create.error.cip.exist'
                    }, HTTP_500_INTERNAL_SERVER_ERROR
    else:
        return {'message': 'ucd_cip.error.profil'
                }, HTTP_403_FORBIDDEN


@cip_blueprint.route('<string:ucd>', methods=['GET'])
@jwt_required()
def get(ucd):
    try:

        filtered_cip = (
            Cip.select(Cip.pk, Cip.ucd, Cip.cip, Cip.qt_pass, Cip.qt_boite, Cip.qt_blister, Cip.dossier)
            .where(Cip.ucd == ucd)).order_by(Cip.ucd, Cip.cip, Cip.dossier.desc())

        logger.debug('Lines : %s.' % len(filtered_cip))

        return {'data': [{
            'image': file_to_base64(s),
            'pk': s.pk,
            'ucd': s.ucd,
            'cip': s.cip,
            'qte_pass': s.qt_pass,
            'qte_boite': s.qt_boite,
            'qte_blister': s.qt_blister,
            'index': s.dossier
        } for s in filtered_cip]}, HTTP_200_OK

    except Exception as error:
        logger.error('Get Ucd raised an exception: ', error.args)
        return {'message': error.args}, HTTP_500_INTERNAL_SERVER_ERROR
