import logging

from common.util import get_counter
from flask import Blueprint
from flask_jwt_extended import jwt_required
from median.models import Adresse, CodeBlocage, Stock, Product
from peewee import JOIN, fn

address_blueprint = Blueprint('address', __name__)

logger = logging.getLogger('median')


@address_blueprint.route('<string:adr>', methods=['POST'])
@jwt_required()
def get_container_counter(adr):
    logger.info('Retrieve a new container code for adress: "%s"' % adr)
    new_counter = get_counter('CONTENANT_PASS')
    logger.info('New container code for address "%s": "%09d"' % (adr, new_counter))
    return '%09d' % new_counter


@address_blueprint.route('all/<string:mag>', methods=['GET'])
@jwt_required()
def get_all(mag):
    try:
        adr = (Adresse.select(Adresse, CodeBlocage.libelle, Adresse.bloque_message,
                              Product.designation, fn.SUM(Stock.quantite).alias('qte_total'))
               .join(CodeBlocage, JOIN.LEFT_OUTER, on=CodeBlocage.valeur == Adresse.bloque)
               .join(Stock, JOIN.LEFT_OUTER, on=Stock.adresse == Adresse.adresse)
               .switch(Adresse)
               .join(Product, JOIN.LEFT_OUTER, on=Product.reference == Stock.reference)
               .where(Adresse.magasin == mag)
               .group_by(Adresse)
               .order_by(Adresse.adresse))

        logger.info('Lines : %s.' % len(adr))

    except Exception as error:
        logger.error('Exception levée en cherchant des adresses, '
                     'format: "%s", magasin: "%s". Message d\'erreur: "%s"' % (format, mag, error.args))
        return {'message': error.args}, 503

    if not len(adr):
        logger.warning('Adresses non trouvées, format: "%s", magasin: "%s"' % (format, mag))

    return ([{
        'adresse': a.adresse,
        'etat': a.etat,
        'contenant': a.contenant,
        'bloque': a.bloque,
        'designation': a.product.designation if hasattr(a, 'product') else '',
        'lock_custom_msg': a.bloque_message,
        'lock_msg': a.codeblocage.libelle if hasattr(a, 'codeblocage') else '',
        'pk': a.pk
    } for a in adr])
