import json
from datetime import datetime, timedelta

from flask import Blueprint, request
from flask_jwt_extended import jwt_required
from median.models import Magasin, Stock, Adresse, Product, Config, Gpao
from median.models.magasin import Reform
from median.views import RawConfig
from peewee import DoesNotExist, JOIN, fn

from common.status import HTTP_200_OK

astus_blueprint = Blueprint('astus', __name__)


@astus_blueprint.route('/partner', methods=['GET'])
@jwt_required()
def get_astus_partner():
    try:
        req_partner = (Config.select(Config.value)
                       .where((Config.poste == 'TOUS')
                              & (Config.cle == 'cfg')
                              & (Config.propriete == 'k_eco_superviseur'))
                       .get())
        partner = req_partner.value
    except DoesNotExist:
        partner = None

    return {'partner': partner}, HTTP_200_OK


@astus_blueprint.route('/<string:astus_pk>/inventory', methods=['PATCH'])
@jwt_required()
def send_inventory(astus_pk):
    stocks = (Stock.select(Stock.reference.distinct().alias('ref'), Stock.magasin.alias("astus"),
                           Magasin.id_robot, Magasin.id_zone)
              .join(Magasin, on=Magasin.mag == Stock.magasin)
              .where(Magasin.pk == astus_pk))
    nb_lines = 0
    for stock in stocks:
        gpao = Gpao()
        gpao.chrono = datetime.now()
        gpao.poste = 'MEDIANWEB'
        gpao.etat = 'A'
        gpao.ref = stock.ref
        gpao.magasin = stock.astus
        gpao.qte = 0
        gpao.type_mvt = 'K'
        gpao.id_robot = stock.magasin.id_robot
        gpao.id_zone = stock.magasin.id_zone

        gpao.save()
        nb_lines = nb_lines + 1

    return {'lines': nb_lines}, HTTP_200_OK


@astus_blueprint.route('/summary', methods=['GET'])
@jwt_required()
def get_astus_summary():
    d = datetime.now()

    nb_peremption = RawConfig().read('k_eco_nb_péremption_astus')
    nb_peremption = nb_peremption.value if nb_peremption is not None else 10
    date_max = (datetime.now() + timedelta(days=nb_peremption)).strftime('%Y-%m-%d')
    date_max_30 = (datetime.now() + timedelta(days=nb_peremption - 30)).strftime('%Y-%m-%d')

    outstocks = Product \
        .select(Magasin.mag.alias('code_magasin'), Product.designation.alias('designation'),
                Product.reference.alias('reference'),
                Stock
                .select(fn.Count(Stock.pk))
                .where((Stock.date_peremption > d) & (Stock.adresse.startswith(Magasin.mag))
                       & (Stock.zone_admin == 'TMP')).alias('qte_total_tampon')
                ) \
        .join(Adresse, on=((Adresse.emplacement == Product.reference) & (Adresse.bloque_reappro == 0))) \
        .join(Magasin, on=(Adresse.adresse.startswith(Magasin.mag)) & (Magasin.eco_type == 'A')) \
        .join(Stock, JOIN.LEFT_OUTER,
              on=((Stock.reference == Product.reference) & (Stock.date_peremption > d) &
                  (Stock.adresse.startswith(Magasin.mag)))) \
        .where(Stock.quantite.is_null()) \
        .order_by(Magasin.mag, Product.designation) \
        .group_by(Magasin.mag, Product.designation)

    mags = Magasin.select(Magasin.pk,
                          Magasin.libelle,
                          Magasin.eco_type,
                          Magasin.type_machine,
                          Magasin.avatar,
                          Magasin.last_reap,
                          Magasin.mag,
                          outstocks.c.designation.alias('designation'),
                          outstocks.c.reference.alias('reference'),
                          outstocks.c.qte_total_tampon.alias('qte_total_tampon'),
                          Adresse
                          .select(fn.Count(Adresse.pk))
                          .where((Adresse.magasin == Magasin.mag)
                                 & (Adresse.bloque != 1)).alias('adress_total'),
                          Adresse
                          .select(fn.Count(Adresse.pk))
                          .where((Adresse.magasin == Magasin.mag)
                                 & (Adresse.emplacement != "")
                                 & (Adresse.bloque != 1)).alias('fill_total'),
                          Adresse
                          .select(fn.Count(Adresse.pk))
                          .where((Adresse.magasin == Magasin.mag) & (Adresse.bloque == 1)).alias('bloque'),
                          Stock.select(fn.SUM(Stock.quantite))
                          .where((Stock.magasin == Magasin.mag) &
                                 (Stock.date_peremption < date_max)).alias('expiration_qty'),
                          Stock.select(fn.SUM(Stock.quantite))
                          .where((Stock.magasin == Magasin.mag) &
                                 (Stock.date_peremption < date_max_30) &
                                 (Stock.date_peremption >= date_max)).alias('expiration_30_qty')
                          ) \
        .join(outstocks, JOIN.LEFT_OUTER, on=(Magasin.mag == outstocks.c.code_magasin)) \
        .where(Magasin.eco_type == 'A') \
        .order_by(Magasin.mag)

    list_mag = []

    for mag in mags:
        t = list(filter(lambda a: a['mag'] == mag.mag, list_mag))
        if len(t) == 0:
            obj = {
                'mag': mag.mag,
                'avatar': mag.avatar,
                'eco_type': mag.eco_type,
                'type_machine': mag.type_machine,
                'libelle': mag.libelle,
                'lastReap': mag.last_reap,
                'expiration_30_qty': mag.expiration_30_qty,
                'expiration_qty': mag.expiration_qty,
                'occupancyRate': str(
                    round((float(mag.fill_total) / float(mag.adress_total)) * float(100.00), 2)) + " %"
                if mag.adress_total != 0 else '-',
                'bufferQty': mag.product.qte_total_tampon if hasattr(mag, 'product') else 0,
                'nb_lock': mag.bloque,
                'pk': mag.pk,
                'outStock': []}
            list_mag.append(obj)
        else:
            obj = t[0]

        if hasattr(mag, 'product'):
            obj['outStock'].append({
                'designation': mag.product.designation,
                'reference': mag.product.reference,
            })

    return {'data': list_mag
            }, 200


@astus_blueprint.route('/all', methods=['GET'])
@jwt_required()
def get_all_astus():
    mags = Magasin \
        .select() \
        .where(Magasin.eco_type == 'A') \
        .order_by(Magasin.type_mag)

    return {'data': [{
        'pk': m.pk,
        'mag': m.mag,
        'type_mag': m.type_mag,
        'eco_type': m.eco_type,
        'libelle': m.libelle,
        'last_reap': m.last_reap,
        'type_machine': m.type_machine,
        'nb_col': m.dim_3,
        'nb_line': m.dim_2
    } for m in mags]
    }, 200


@astus_blueprint.route('/stock/<string:astus_pk>', methods=['GET'])
@jwt_required()
def get_stock_astus(astus_pk):
    try:
        mag = Magasin.get(pk=astus_pk)

        stocks = Stock.select(Stock, Adresse, Product) \
            .join(Adresse, JOIN.LEFT_OUTER, on=(Adresse.adresse == Stock.adresse)) \
            .switch(Stock) \
            .join(Product, JOIN.LEFT_OUTER, on=(Product.reference == Stock.reference)) \
            .where(Stock.magasin == mag.mag) \
            .order_by(Adresse.adresse.asc())

        listStock = []

        for stock in stocks:
            obj = {'contenant': stock.contenant, 'batch': stock.lot, 'quantity': stock.quantite,
                   'entry': stock.date_entree, 'expiry': stock.date_peremption, 'zone': stock.zone_admin,
                   'reference': stock.reference}

            if not isinstance(stock.adresse, str):
                obj['adress'] = stock.adresse.adresse
                obj['format'] = stock.adresse.format
            else:
                obj['adress'] = stock.adresse

            if stock.product is not None:
                obj['designation'] = stock.product.designation

            listStock.append(obj)

        return {'data': listStock
                }, 200

    except DoesNotExist:
        return {'message': 'Magasin does not exist'}, 400


@astus_blueprint.route('/adresses/<string:astus_pk>/maxtray', methods=['GET'])
@jwt_required()
def get_max_tray(astus_pk):
    try:
        mag = Magasin.get(pk=astus_pk)
        data = request.args
        quantity = None
        max_quantity = None

        rows_max = Reform.select() \
            .where((Reform.format == data.get('format'))
                   & (Reform.reference == data.get('reference')))
        if len(rows_max) > 0:
            max_quantity = rows_max[0].capacity

        rows = Adresse.select(Adresse) \
            .where((Adresse.adresse == data.get('adress')) & Adresse.quantite_posage.is_null(False))
        if len(rows) == 0:
            rows = Adresse.select(Adresse) \
                .where((Adresse.magasin == mag.mag)
                       & (Adresse.format == data.get('format'))
                       & (Adresse.emplacement == data.get('reference'))
                       & Adresse.quantite_posage.is_null(False))

            if len(rows) == 0:
                rows = Adresse.select(Adresse) \
                    .where((Adresse.format == data.get('format'))
                           & (Adresse.emplacement == data.get('reference'))
                           & Adresse.quantite_posage.is_null(False))
                if len(rows) > 0:
                    quantity = rows[0].quantite_posage
            else:
                quantity = rows[0].quantite_posage
        else:
            quantity = rows[0].quantite_posage

        return {'quantity': quantity, 'max': max_quantity}, HTTP_200_OK
    except DoesNotExist:
        return {'message': 'Magasin does not exist'}, 400


@astus_blueprint.route('/adresses/<string:astus_pk>/dotation', methods=['POST'])
@jwt_required()
def newDotation(astus_pk):
    try:
        mag = Magasin.get(pk=astus_pk)
        data = json.loads(request.data)

        """ Mise à jour quantité dotation"""
        Adresse \
            .update(quantite_posage=data['quantity']) \
            .where((Adresse.magasin == mag.mag) &
                   (Adresse.emplacement == data['reference']) &
                   Adresse.format == data['format']
                   ) \
            .execute()
        """ Mise à jour max """
        try:
            reform = Reform \
                .select() \
                .where((Reform.reference == data['reference']) &
                       (Reform.format == data['format'])) \
                .get()
            Reform \
                .update(capacity=data['max']) \
                .where(Reform.pk == reform.pk).execute()
        except DoesNotExist:
            Reform \
                .create(reference=data['reference'],
                        format=data['format'],
                        capacity=data['max'])

        if data['new']:
            Adresse \
                .update(emplacement=data['reference'],
                        nouvelle_quantite_posage=data['quantity'],
                        nouvel_emplacement=data['reference'],
                        service=False,
                        bloque_reappro=True
                        ) \
                .where(Adresse.adresse.startswith(data['adress'])) \
                .execute()
        else:
            Adresse \
                .update(emplacement=data['reference'],
                        nouvelle_quantite_posage=0,
                        nouvel_emplacement='',
                        service=False,
                        bloque_reappro=False
                        ) \
                .where(Adresse.adresse.startswith(data['adress'])) \
                .execute()

        return {'adress': getAdress(mag.mag, data['adress'])}, HTTP_200_OK

    except DoesNotExist:
        return {'message': 'Magasin does not exist'}, 400


def getAdress(magasin, adress):
    adrs = Adresse \
        .select(Adresse, Product.designation,
                Stock.serial,
                Stock.date_entree,
                Stock.date_peremption,
                Stock.quantite,
                Stock.lot,
                Stock.contenant,
                Reform
                .select(Reform.capacity)
                .where((Adresse.emplacement == Reform.reference) & (Reform.format == Adresse.format))
                .alias('max'),
                Product.select(Product.designation).where(
                    Product.reference == Adresse.nouvel_emplacement).alias('new_designation'),
                ) \
        .join(Product, JOIN.LEFT_OUTER, on=(Product.reference == Adresse.emplacement)) \
        .join(Stock, JOIN.LEFT_OUTER, on=(Stock.adresse == Adresse.adresse)) \
        .where((Adresse.magasin == magasin) & (Adresse.adresse == adress)) \
        .order_by(Adresse.adresse.asc())

    list_adrs = []
    for adr in adrs:
        t = list(filter(lambda a: a['pk'] == adr.pk, list_adrs))
        if len(t) == 0:

            obj = {
                'pk': adr.pk,
                'contenant': adr.contenant,
                'quantity': adr.quantite_posage,
                'new_quantity': adr.nouvelle_quantite_posage,
                'reference': adr.emplacement,
                'new_reference': adr.nouvel_emplacement,
                'adress': adr.adresse, 'format': adr.format, 'lock_replenish': adr.bloque_reappro,
                'new_designation': adr.new_designation,
                "stocks": [],
                'lock': adr.bloque, 'max': adr.max,
                'state': adr.etat,
                'service': adr.service}

            if getattr(adr, 'product', None) is not None:
                obj['designation'] = adr.product.designation
                if hasattr(adr.product, 'stock'):
                    obj['stocks'].append({'pk': adr.product.stock.pk,
                                          'quantity': adr.product.stock.quantite,
                                          'entry': adr.product.stock.date_entree,
                                          'expiry': adr.product.stock.date_peremption,
                                          'batch': adr.product.stock.lot,
                                          'container': adr.product.stock.contenant})

            list_adrs.append(obj)
        else:
            t[0]['stocks'].append({'pk': adr.product.stock.pk,
                                   'quantity': adr.product.stock.quantite,
                                   'entry': adr.product.stock.date_entree,
                                   'expiry': adr.product.stock.date_peremption,
                                   'batch': adr.product.stock.lot,
                                   'container': adr.product.stock.contenant})
    return list_adrs[0]


@astus_blueprint.route('/adresses/<string:astus_pk>/dotation', methods=['DELETE'])
@jwt_required()
def deleteDotation(astus_pk):
    try:
        mag = Magasin.get(pk=astus_pk)
        data = json.loads(request.data)

        if data['new_reference']:
            Adresse.update(
                nouvelle_quantite_posage=0,
                nouvel_emplacement='',
                bloque_reappro=False,
                service=False) \
                .where((Adresse.magasin == mag.mag) &
                       (Adresse.adresse.startswith(data['adress']))) \
                .execute()
        else:
            Adresse.update(
                emplacement='',
                quantite_posage=0,
                nouvelle_quantite_posage=0,
                nouvel_emplacement='',
                bloque_reappro=False,
                service=False) \
                .where((Adresse.magasin == mag.mag) &
                       (Adresse.adresse.startswith(data['adress']))) \
                .execute()

        return {'adress': getAdress(mag.mag, data['adress'])}, HTTP_200_OK
    except DoesNotExist:
        return {'message': 'Magasin does not exist'}, 400


@astus_blueprint.route('/adresses/<string:astus_pk>/lock', methods=['PUT'])
@jwt_required()
def lock(astus_pk):
    try:
        mag = Magasin.get(pk=astus_pk)
        data = json.loads(request.data)

        Adresse.update(bloque=data['lock']) \
            .where(
            (Adresse.adresse == data['adress']) &
            (Adresse.magasin == mag.mag)) \
            .execute()

        return {}, HTTP_200_OK
    except DoesNotExist:
        return {'message': 'Magasin does not exist'}, 400


@astus_blueprint.route('/adresses/<string:astus_pk>/lockreplenish', methods=['PUT'])
@jwt_required()
def lockreplenish(astus_pk):
    try:
        mag = Magasin.get(pk=astus_pk)
        data = json.loads(request.data)

        Adresse.update(bloque_reappro=data['lock']) \
            .where(
            (Adresse.adresse == data['adress']) &
            (Adresse.magasin == mag.mag)) \
            .execute()

        return {}, HTTP_200_OK
    except DoesNotExist:
        return {'message': 'Magasin does not exist'}, 400


@astus_blueprint.route('/adresses/<string:astus_pk>', methods=['GET'])
@jwt_required()
def get_adresses_astus(astus_pk):
    try:
        mag = Magasin.get(pk=astus_pk)

        adrs = Adresse \
            .select(Adresse, Product.designation,
                    Stock.serial,
                    Stock.date_entree,
                    Stock.date_peremption,
                    Stock.quantite,
                    Stock.lot,
                    Stock.contenant,
                    Reform
                    .select(Reform.capacity)
                    .where((Adresse.emplacement == Reform.reference) & (Reform.format == Adresse.format))
                    .alias('max'),
                    Product.select(Product.designation).where(
                        Product.reference == Adresse.nouvel_emplacement).alias('new_designation'),
                    ) \
            .join(Product, JOIN.LEFT_OUTER, on=(Product.reference == Adresse.emplacement)) \
            .join(Stock, JOIN.LEFT_OUTER, on=(Stock.adresse == Adresse.adresse)) \
            .where((Adresse.magasin == mag.mag)) \
            .order_by(Adresse.adresse.asc())

        list_adrs = []
        for adr in adrs:
            t = list(filter(lambda a: a['pk'] == adr.pk, list_adrs))
            if len(t) == 0:

                obj = {
                    'pk': adr.pk,
                    'contenant': adr.contenant,
                    'quantity': adr.quantite_posage,
                    'new_quantity': adr.nouvelle_quantite_posage,
                    'reference': adr.emplacement,
                    'new_reference': adr.nouvel_emplacement,
                    'adress': adr.adresse, 'format': adr.format, 'lock_replenish': adr.bloque_reappro,
                    'new_designation': adr.new_designation,
                    "stocks": [],
                    'lock': adr.bloque, 'max': adr.max,
                    'state': adr.etat,
                    'service': adr.service}

                if getattr(adr, 'product', None) is not None:
                    obj['designation'] = adr.product.designation
                    if hasattr(adr.product, 'stock'):
                        obj['stocks'].append({'pk': adr.product.stock.pk,
                                              'quantity': adr.product.stock.quantite,
                                              'entry': adr.product.stock.date_entree,
                                              'expiry': adr.product.stock.date_peremption,
                                              'batch': adr.product.stock.lot,
                                              'container': adr.product.stock.contenant})

                list_adrs.append(obj)
            else:
                obj = t[0]['stocks'].append({'pk': adr.product.stock.pk,
                                             'quantity': adr.product.stock.quantite,
                                             'entry': adr.product.stock.date_entree,
                                             'expiry': adr.product.stock.date_peremption,
                                             'batch': adr.product.stock.lot,
                                             'container': adr.product.stock.contenant})

        return {'data': list_adrs
                }, 200

    except DoesNotExist:
        return {'message': 'Magasin does not exist'}, 400
