import datetime
import json
import logging

from flask import Blueprint, request
from flask_jwt_extended import jwt_required
from median.base import BaseViewException
from median.constant import TypeServiListe, EcoType
from median.models import FListe, FItem, Product, Magasin, Config
from median.views import Seuil, RawConfig
from common.status import HTTP_200_OK, HTTP_503_SERVICE_UNAVAILABLE, HTTP_400_BAD_REQUEST
from median import views as v
from peewee import DoesNotExist

acced_replenish_blueprint = Blueprint('acced_replenish', __name__)

logger = logging.getLogger('median')


@acced_replenish_blueprint.route('', methods=['GET'])
@jwt_required()
def get():
    logger.info("Récupérer les listes de réappro...")
    lis = (FListe.select(FListe.pk, FListe.liste,
                         FListe.zone_fin,
                         FListe.date_creation,
                         ).where(
        FListe.mode == 'E', FListe.zone_deb == ''
    )
           .join(Magasin, on=FListe.zone_fin == Magasin.type_mag)
           .where(Magasin.eco_type << [EcoType.Cueillette.value, EcoType.Coupe.value])
           .order_by(FListe.zone_fin, FListe.pk))
    logger.info('Récupérer les listes de réappro ACCED... %s' % lis.count())
    return [{'pk': li.pk,
             'label': li.liste,
             'mag': li.zone_fin,
             'date_creation': str(li.date_creation)} for li in lis], HTTP_200_OK


@acced_replenish_blueprint.route('<string:pk_liste>', methods=['POST'])
@jwt_required()
def get_detail(pk_liste):
    # nb_item = 0
    try:
        args = json.loads(request.data)
    except Exception as err:
        print(err)

    v_limit = args['length']
    v_offset = args['start']
    v_filter_by_magasin = args['filterByMagasin']
    v_search = args['search'].strip().upper() if args['search'] is not None else None

    logger.info("Récupérer les items de listes de réappro, pk de la liste : '%s'" % pk_liste)

    try:
        pk_liste = int(pk_liste)
        # on récupère le libellé de la liste
        # c'est nul, mais c'est comme ça qu'on retrouve les items pour l'instant
        _liste = FListe.select(FListe.liste, FListe.nb_item).where((FListe.pk == pk_liste) & (FListe.mode == 'E'))

        if len(_liste) == 0:
            logger.error('No list for id %s' % pk_liste)
            return {'data': []}, HTTP_200_OK

        # nb_item = _liste[0].nb_item

        ms = v_filter_by_magasin.split(",")

        _items = (FItem.select(
            FItem.pk,
            FItem.etat,
            FItem.reference,
            FItem.fraction,
            Product.designation,
            Product.pk,
            FItem.qte_dem,
            FItem.qte_serv
        ).join(Product, on=(Product.reference == FItem.reference))
                  .where(((v_search == '') | ((v_search != '') & (Product.designation.contains(v_search)))) &
                         (FItem.liste == _liste[0].liste) & (FItem.etat << ms)))
        nb = _items.count()
        _items = _items.limit(v_limit).offset(v_offset).order_by(FItem.item)

        return {
            'recordsTotal': nb,
            'data': [{
                'pk': i.pk,
                'etat': _get_etat_human_label(i.etat),
                'reference': i.reference,
                'reference_pk': i.product.pk,
                'fraction': i.fraction,
                'designation': i.product.designation,
                'qte_demandee': i.qte_dem,
                'qte_servie': i.qte_serv
            } for i in _items]}, HTTP_200_OK

    except Exception as error:
        logger.error('Get reappro items Datatables raised an exception: ', error.args)
        return {'messages': error.args}, HTTP_200_OK


@acced_replenish_blueprint.route('', methods=['DELETE'])
@jwt_required()
def delete():
    args = request.args
    _pk = args['pk']
    logger.info("Suppression d'une liste de réappro: %s" % _pk)

    try:

        # get list label (needed to fetch items)
        _l = FListe.select(FListe.liste).where(FListe.pk == _pk)

        if len(_l) == 0:
            logger.error("Liste de réappro non trouvée: %s" % _pk)
            return {'message': "Liste de réappro non trouvée!"}, HTTP_503_SERVICE_UNAVAILABLE

        FItem.delete().where((FItem.liste == _l[0].liste) & (FItem.mode == 'E')).execute()
        FListe.delete().where((FListe.pk == _pk) & (FListe.mode == 'E')).execute()

        logger.info("Suppression réussie d'une liste de réappro: %s" % _pk)
        return {'message': 'Success', 'id': None}

    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_400_BAD_REQUEST


@acced_replenish_blueprint.route('calcul_reappro', methods=['POST'])
@jwt_required()
def replenishment_calculation():
    logger.info("Calcul d'une liste d'entrée de reappro")

    args = json.loads(request.data)
    _mag_long = args['mag']
    res = ""

    try:
        _m = Magasin.select(Magasin.mag).where(Magasin.type_mag == _mag_long)

        if len(_m) == 0:
            logger.error("Magasin non trouvé: " + _mag_long)
            return {'message': "Magasin non trouvé: " + _mag_long}, HTTP_400_BAD_REQUEST

        res = v.Seuil().reappro_calcul(_mag_long)

    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_400_BAD_REQUEST

    logger.info("Génération automatique d'une liste d'entrée... REUSSI")

    return res


@acced_replenish_blueprint.route('reappro', methods=['GET'])
@jwt_required()
def get_counter_replenishment():
    # n = Config.select(Config.value).where(Config.cle == 'CPT_LISTE_E' & Config.poste == 'TOUS')
    try:
        n = Config.get(poste='TOUS', cle='CPT_LISTE_S')
    except DoesNotExist as error:
        return {'message': error.args}, 400

    c = str(int(n.value) + 1)

    Config.update({Config.value: c}).where((Config.poste == 'TOUS') & (Config.cle == 'CPT_LISTE_S')).execute()

    return c.zfill(5)


@acced_replenish_blueprint.route('reappro', methods=['POST'])
@jwt_required()
def create_replenishment():
    logger.info("ApiReappro::post")
    res = request.get_json(force=True)
    logger.info(res)

    liste = res["liste"]
    type_magasin = res["type_magasin"]
    gpao = res["gpao"]

    service = _transfert_service()
    mag = _check_type_magasin(type_magasin)

    # Création de la liste d'entrée
    if len(res["lines"]) > 0:
        list = FListe.create(
            liste=liste, date_creation=datetime.datetime.now(), etat='V',
            mode='E', fusion='REASSORT', service=service,
            date_modification=datetime.datetime.now(), zone_fin=type_magasin,
            type_servi=TypeServiListe.GlobaleBoite.value, id_servi=2,
            nb_item=len(res["lines"])
        )

        # Création des items
        for r in res["lines"]:
            Seuil().reappro_item(liste, r["item"], r["ref"], r["frac"], r["qte"],
                                 type_magasin, service, mag.id_zone, mag.id_robot, gpao)

        return {'pk': list.pk}, HTTP_200_OK
    else:
        # Création f_gpao
        if gpao:
            Seuil().creer_gpao(mag.id_zone)

    return "ok"


def _transfert_service():
    """Retrieve teh code of the service to transfert stock"""
    service = RawConfig().read(param='k_ua_transfert').value
    # if service is None:
    #     raise BaseViewException("k_ua_transfet n'est pas définit")

    return service


def _check_type_magasin(type_magas):
    """Check if magasin exists

        :param type_magas: code magasin a rechercher
        :type  type_magas: str
        :return: Magasin model
        :rtype: median.models.Magasin
        """
    try:
        m = Magasin.get(type_mag=type_magas)
    except DoesNotExist:
        logger.error("Le type magasin %s n'existe pas" % type_magas)
        raise BaseViewException("Le magasin n'existe pas dans la base")
    return m


def _get_etat_human_label(etat_short_label):
    options = {
        'V': 'Vierge',
        'E': 'Entamé',
        'S': 'Soldé'
    }
    return options.get(etat_short_label, 'Inconnu')
