
import logging
import time
import io
from flask import Blueprint, make_response, send_file
from flask_jwt_extended import jwt_required
# from marshmallow import Schema, fields
from median.models import FItem, ListeModel, ListeItemModel, Product
from peewee import DoesNotExist, JOIN
# from ressources.astus.utils import generate_excel_file
from common.pdf import TablePDF
from common.status import HTTP_204_NO_CONTENT, HTTP_500_INTERNAL_SERVER_ERROR
from fpdf.errors import FPDFException


acced_reappro_blueprint = Blueprint('reappro', __name__)

logger = logging.getLogger('median')


@acced_reappro_blueprint.route('line/<string:item_pk>', methods=['DELETE'])
@jwt_required()
def delete_lines(item_pk):
    try:
        FItem.delete().where(FItem.pk == item_pk).execute()
    except Exception as error:
        logger.error(error.args)
        return {
            'alertMessage': 'replenishment.line.delete.error',
            'param': [item_pk]
        }, HTTP_500_INTERNAL_SERVER_ERROR
    return {}, HTTP_204_NO_CONTENT


@acced_reappro_blueprint.route('print/<string:list_pk>', methods=['GET'])
@jwt_required()
def printing(list_pk):
    res = []
    logger.info("Print replenishment %s" % list_pk)
    try:
        reappro = ListeModel.get_by_id(list_pk)

        lines = ListeItemModel.select(
            ListeItemModel.pk, ListeItemModel.reference, ListeItemModel.qte_dem,
            ListeItemModel.qte_serv, ListeItemModel.fraction,
            Product.designation.alias('libelle')
        ).join(
            Product, JOIN.INNER, on=Product.reference == ListeItemModel.reference
        ).where(
            ListeItemModel.mode == reappro.mode, ListeItemModel.liste == reappro.liste
        ).order_by(Product.designation)
        # print(lines)
        for li in lines:
            res.append([
                '[%s] %s' % (li.reference, li.product.libelle),
                li.qte_dem,
                li.fraction,
            ])
        logger.info('number of line for replenishment: %i' % len(res))
        filename = 'reappro-%s-%s' % (reappro.zone_fin, time.strftime('%Y%m%d_%H%M%S'))
        file_io = io.BytesIO()

        head = [
            ('Reference', 80, 'L'),
            ('Quantite', 10, 'R'),
            ('Fraction', 10, 'C')
        ]

        try:
            pdf_file = TablePDF()
            pdf_file.doc_font(size=7)
            pdf_file.doc_title(reappro.liste)
            pdf_file.grid_header(head)
            pdf_file.grid_rows(res)
            pdf_file.doc_save(file_io)

        except FPDFException as ex:
            logger.error("Error when generate PDF for replenishment %s" % list_pk)
            logger.error(str(ex))
            return {
                'alertMessage': 'replenishment.header.pdf.error',
                'param': [str(ex)]
            }, HTTP_500_INTERNAL_SERVER_ERROR

        file_io.seek(0)
        response = make_response(send_file(
            file_io, download_name="%s.pdf" % filename,
            mimetype='application/pdf', as_attachment=True, max_age=0))
        response.headers.set('file-name', "%s.pdf" % filename)
        return response

    except DoesNotExist:
        logger.error("Replenishment %s not found" % list_pk)
        return {
            'alertMessage': 'replenishment.header.print.error',
            'param': [list_pk]
        }, HTTP_500_INTERNAL_SERVER_ERROR
