import logging
import socket

from median.models import Compteur
from median.database import mysql_db
from median.views import RawConfig
from common.exception import PrinterException

from urllib.parse import urlparse
from pathlib import Path

logger = logging.getLogger('median.webserver')
data_dir = Path(__file__).resolve().parent


def Success(message="", id=None, **kwargs):
    """Format a JSON response"""
    res = {
        'id': id,
        'message': message,
    }
    for arg in kwargs:
        res[arg] = kwargs[arg]

    return res


def compute_checksum(number):
    """Calculate the EAN check digit for 13-digit numbers. The number passed
    should not have the check bit included."""
    if len(number) != 12:
        raise Exception("Invalid length")

    checksum_digit = str(
        (10 - sum((3, 1)[i % 2] * int(n) for i, n in enumerate(reversed(number)))) % 10
    )

    return str(number) + checksum_digit


def send_to_printer(printer_address: str = None, printer_content: str = None, socket_timeout: float = 5):
    """
    Send the content to the printer

    :param printer_address: adresse of the printer asan URI (eg: zebra://127.0.0.1:9100/)
    :param printer_content: context as a string value
    :param socket_timeout: timeout before returning an exception (5 seconds by default)
    """
    try:
        printer_url = urlparse(printer_address)
        host_port = printer_url.netloc.split(':')
        s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        s.settimeout(socket_timeout)
        s.connect((host_port[0], int(host_port[1])))
        s.send(bytes(printer_content, "utf-8"))
        s.close()
    except Exception as e:
        logger.error("Printing error on [%s]" % printer_address)
        logger.error(str(e))
        raise PrinterException("Error to send label to printer configuration!" + str(e))
    return True


def get_counter(count_name=None):
    """Retrieve a counter and increment it"""
    with mysql_db.atomic():
        cpt = (
            Compteur.select()
            .where(Compteur.cle == count_name)
            .for_update()
            .get()
        )
        result = cpt.val
        cpt.val = cpt.val + 1
        cpt.save()
    return result


def is_multiple(value: int, divider: int) -> bool:
    """
    Check if value is a multiple of divider

    :param value: Value to check
    :param divider: miltiplication parameters
    :returns: True if multiple of False (0 also return false)
    """
    if divider == 0:
        return False

    if value % divider == 0:
        return True
    else:
        return False


def zero_padding(value: int, number_of_zero: int) -> str:
    return str(value).zfill(number_of_zero)


def get_config_global_or_local(poste: str, prop: str) -> str:
    """
    Retrieve the configuration parameters for hte poste, if not found we return the
    gloabl paramter, otherwise we return and empty string
    """
    cfg_value = RawConfig(poste).read(param=prop)
    if cfg_value is None:
        cfg_value = RawConfig().read(param=prop)

    return cfg_value and cfg_value.value or ""


def read_printer_template(filename: str = None) -> str:
    """Read the template name path and return the content
    """
    logger.info('Read %s template file' % filename)
    template_content = ""
    with open(data_dir / "data" / "riedl" / filename, 'r', encoding='utf-8') as f:
        template_content = f.read()
    return template_content
