from median.models import Magasin, Endpoint, CodeBlocage, Config, Compteur, ContainerFormat, Adresse
from median.views import Riedl
from median.base import BaseViewException
from peewee import DoesNotExist
import logging
import json
import os

logger = logging.getLogger('migration.riedl')


def riedl_migration():
    """Manage riedl migration"""
    logger.info("RIEDL Migration")
    whs = riedl_complete_magasin()
    riedl_format_default()
    riedl_location_default()
    riedl_check_endpoint(whs)
    riedl_list_defaut()
    riedl_ward_default_endpoint(whs)
    riedl_printer()
    riedl_dispatcher()
    _riedl_counter()
    logger.info("RIEDL Migration done")


def riedl_complete_magasin():
    mags = Magasin().select(Magasin).where(Magasin.eco_type == 'L')
    for m in mags:
        m.lib_2 = 'ENTREE'
        m.lib_3 = 'SORTIE'
        m.lib_4 = 'URGENCE'
        m.nb_dim = 3
        m.save()
    return mags


def riedl_format_default():
    logger.info("Check format")
    mags = Magasin().select(Magasin).where(Magasin.eco_type == 'L')
    for m in mags:
        try:
            ContainerFormat.get(code="SHELF")
            logger.info("SHELF format exists")
        except DoesNotExist:
            logger.info("format SHELF not exists, we create it")
            cf = ContainerFormat()
            cf.code = "SHELF"
            cf.save()


def riedl_location_default():
    logger.info("Location by default")
    mags = Magasin().select(Magasin).where(Magasin.eco_type == 'L')
    for m in mags:
        try:
            Adresse.get(adresse=m.mag, magasin=m.mag)
            logger.info("%s exists, we do nothing" % m.mag)
        except DoesNotExist:
            logger.info("Default %s not exists, we create it" % m.mag)
            adr = Adresse()
            adr.adresse = m.mag
            adr.magasin = m.mag
            adr.format = "SHELF"
            adr.etat = "M"
            adr.save()


def riedl_check_endpoint(mags=[]):
    for m in mags:
        try:
            rdl = Riedl(m.type_mag)
        except BaseViewException as e:
            logger.info(e.message)
            continue

        # Check the urgency output
        logger.info("Number of urgency output in %s -> %i" % (m.mag, m.dim_4))
        for i in range(1, m.dim_4 + 1):
            ep_name = "%s.UR%02i" % (m.mag, i)
            try:
                Endpoint.get(code=ep_name)
                logger.info("%s exists" % ep_name)
            except DoesNotExist:
                logger.info("%s does not exists" % ep_name)
                rdl.add_endpoint(ep_name, "Urgence %i / Loading Bay %i" % (i, i), 1, i)

        logger.info("Number of normal output in %s -> %i" % (m.mag, m.dim_3))
        for i in range(1, m.dim_3 + 1):
            ep_name = "%s.EX%02i" % (m.mag, i)
            try:
                Endpoint.get(code=ep_name)
                logger.info("%s exists" % ep_name)
            except DoesNotExist:
                logger.info("%s does not exists" % ep_name)
                rdl.add_endpoint(ep_name, "Sortie %i / Exit %i" % (i, i), 3, i)

        # We add the default endpoint on f_mag.x_dest
        if m.dest == '':
            m.dest = ep_name = "%s.EX01" % (m.mag,)
            m.save()


def riedl_ward_default_endpoint(mags=[]):
    for m in mags:
        try:
            Riedl(m.type_mag)
        except BaseViewException as e:
            logger.info(e.message)
            continue


def riedl_list_defaut():
    filename = os.path.join(os.path.dirname(__file__), '..', 'data', 'riedl_defaut.json')
    # logger.info(filename)
    if os.path.exists(filename):
        # logger.info("File found")
        with open(filename) as json_file:
            data = json.load(json_file)
            for d in data:
                # logger.info(d['id'], d['label'].encode('latin9'))
                try:
                    cb = CodeBlocage.get(valeur=d['id'])
                    cb.libelle = d['label'].encode('latin9')
                    cb.save()
                    # logger.info('CodeBlocage %i exists' % cb.valeur)
                except DoesNotExist:
                    logger.info('CodeBlocage %i not exists' % d['id'])
                    cb = CodeBlocage()
                    cb.valeur = d['id']
                    cb.libelle = d['label'].encode('latin9')
                    cb.save()


def riedl_printer():
    """Define configuration file for Zebra printer"""
    for i in range(9):
        addr = 'k_rdl_printer_adr_%i' % (i + 1)
        name = 'k_rdl_printer_name_%i' % (i + 1)
        offs = 'k_rdl_printer_offset_%i' % (i + 1)
        tmpl = "k_rdl_tk_template_%i" % (i + 1)
        try:
            Config.get(poste='TOUS', cle='cfg', propriete=addr)
            print('Printer %i key exists' % (i + 1))
        except DoesNotExist:
            zpl_cfg = Config()
            zpl_cfg.ref_id = '_RDLPRN0%i' % (i + 1)
            zpl_cfg.poste = 'TOUS'
            zpl_cfg.cle = 'cfg'
            zpl_cfg.propriete = addr
            zpl_cfg.type_value = 'C'
            zpl_cfg.value = 'zebra+tcp://172.25.212.X:9100'
            zpl_cfg.hierarchie_menu = """RIEDL\\Imprimante\\%i-Adresse""" % (i + 1)
            zpl_cfg.type_champ = 'reg'
            zpl_cfg.ordre = 10 + i
            zpl_cfg.save()
            print('Printer %i created' % (i + 1))

        try:
            Config.get(poste='TOUS', cle='cfg', propriete=name)
            print('Printer name %i key exists' % (i + 1))
        except DoesNotExist:
            zpl_cfg = Config()
            zpl_cfg.ref_id = '_RDLPRN1%i' % (i + 1)
            zpl_cfg.poste = 'TOUS'
            zpl_cfg.cle = 'cfg'
            zpl_cfg.propriete = name
            zpl_cfg.type_value = 'C'
            zpl_cfg.value = 'Zebra 430 GX'
            zpl_cfg.hierarchie_menu = """RIEDL\\Imprimante\\%i-Nom""" % (i + 1)
            zpl_cfg.type_champ = 'reg'
            zpl_cfg.ordre = 10 + i
            zpl_cfg.save()
            print('Printer name %i created' % (i + 1))

        try:
            Config.get(poste='TOUS', cle='cfg', propriete=offs)
            print('Printer offset %i key exists' % (i + 1))
        except DoesNotExist:
            zpl_cfg = Config()
            zpl_cfg.ref_id = '_RDLPRN2%i' % (i + 1)
            zpl_cfg.poste = 'TOUS'
            zpl_cfg.cle = 'cfg'
            zpl_cfg.propriete = offs
            zpl_cfg.type_value = 'N'
            zpl_cfg.value = '30'
            zpl_cfg.hierarchie_menu = """RIEDL\\Imprimante\\%i-Offset""" % (i + 1)
            zpl_cfg.type_champ = 'num'
            zpl_cfg.ordre = 10 + i
            zpl_cfg.save()
            print('Printer offset %i created' % (i + 1))

        try:
            Config.get(poste='TOUS', cle='cfg', propriete=tmpl)
            print('Printer template %i key exists' % (i + 1))
        except DoesNotExist:
            zpl_cfg = Config()
            zpl_cfg.ref_id = '_RDLPRN3%i' % (i + 1)
            zpl_cfg.poste = 'TOUS'
            zpl_cfg.cle = 'cfg'
            zpl_cfg.propriete = tmpl
            zpl_cfg.type_value = 'C'
            zpl_cfg.value = 'box_uncomplete_102x68.tzpl'
            zpl_cfg.hierarchie_menu = """RIEDL\\Imprimante\\%i-template""" % (i + 1)
            zpl_cfg.type_champ = 'reg'
            zpl_cfg.ordre = 10 + i
            zpl_cfg.save()
            print('Printer template %i created' % (i + 1))

    # For each RIedl we can declare the default printer.
    # Declare 0 if no printer default
    mags = _rield_list()
    for m in mags:
        try:
            Config.get(poste=m.type_mag, cle='cfg', propriete='k_rdl_printer')
            print('Printer %s key exists' % m.type_mag)
        except DoesNotExist:
            zpl_cfg = Config()
            zpl_cfg.ref_id = ''
            zpl_cfg.poste = m.type_mag
            zpl_cfg.cle = 'cfg'
            zpl_cfg.propriete = 'k_rdl_printer'
            zpl_cfg.type_value = 'N'
            zpl_cfg.value = '0'
            zpl_cfg.hierarchie_menu = """RIEDL\\Imprimante\\Selection"""
            zpl_cfg.type_champ = 'num'
            zpl_cfg.ordre = 15
            zpl_cfg.save()
            print('Printer %s key created' % m.type_mag)


def riedl_dispatcher():
    """Create an URL per dispatcher"""
    # For each mag declare a printer number
    mags = _rield_list()
    for m in mags:
        try:
            Config.get(poste=m.type_mag, cle='cfg', propriete='k_rdl_dispatcher_url')
            print('Dispatcher URL %s key exists' % m.type_mag)
        except DoesNotExist:
            zpl_cfg = Config()
            zpl_cfg.ref_id = '_RDLDISP01'
            zpl_cfg.poste = m.type_mag
            zpl_cfg.cle = 'cfg'
            zpl_cfg.propriete = 'k_rdl_dispatcher_url'
            zpl_cfg.type_value = 'C'
            zpl_cfg.value = 'http://127.0.0.1:5005'
            zpl_cfg.hierarchie_menu = """RIEDL\\Dispatcher\\URL"""
            zpl_cfg.type_champ = 'reg'
            zpl_cfg.ordre = 15
            zpl_cfg.save()
            print('Dispatcher URL %s created' % m.type_mag)


def _rield_list():
    mags = Magasin().select(Magasin).where(Magasin.eco_type == 'L')
    return mags


def _riedl_counter():
    """Add counter for taking mode or return"""

    # Create the counter for return and taking mode
    try:
        Compteur.get(cle='RIEDL_RETOUR')
        print('Return counter exists')
    except DoesNotExist:
        cpt = Compteur()
        cpt.cle = 'RIEDL_RETOUR'
        cpt.val = 1
        cpt.save()
        print('Return counter create')

    try:
        Compteur.get(cle='RIEDL_PRELEVEMENT')
        print('Taking mode counter exists')
    except DoesNotExist:
        cpt = Compteur()
        cpt.cle = 'RIEDL_PRELEVEMENT'
        cpt.val = 1
        cpt.save()
        print('Taking mode counter create')

    try:
        Compteur.get(cle='RIEDL_GLOBALE')
        print('Globale counter exists')
    except DoesNotExist:
        cpt = Compteur()
        cpt.cle = 'RIEDL_GLOBALE'
        cpt.val = 1
        cpt.save()
        print('Globale counter create')

    try:
        Compteur.get(cle='RIEDL_COMMAND')
        print('Command counter exists')
    except DoesNotExist:
        cpt = Compteur()
        cpt.cle = 'RIEDL_COMMAND'
        cpt.val = 90000000
        cpt.save()
        print('Command counter create')

    try:
        Compteur.get(cle='RIEDL_INVENTORY')
        print('Inventory counter exists')
    except DoesNotExist:
        cpt = Compteur()
        cpt.cle = 'RIEDL_INVENTORY'
        cpt.val = 1
        cpt.save()
        print('Inventory counter create')

    try:
        Compteur.get(cle='RIEDL_CONTAINER')
        print('Container counter exists')
    except DoesNotExist:
        cpt = Compteur()
        cpt.cle = 'RIEDL_CONTAINER'
        cpt.val = 1
        cpt.save()
        print('Container counter create')

    # Create the prefix for return and taking mode
    prefix = 'k_rdl_prefix_return'
    try:
        Config.get(poste='TOUS', cle='cfg', propriete=prefix)
        print('Prefix for return exists key exists')
    except DoesNotExist:
        zpl_cfg = Config()
        zpl_cfg.ref_id = '_RDLPREF01'
        zpl_cfg.poste = 'TOUS'
        zpl_cfg.cle = 'cfg'
        zpl_cfg.propriete = prefix
        zpl_cfg.type_value = 'C'
        zpl_cfg.value = '233001'
        zpl_cfg.hierarchie_menu = """RIEDL\\Codebarre\\Prefix Retour"""
        zpl_cfg.type_champ = 'reg'
        zpl_cfg.ordre = 10
        zpl_cfg.save()
        print('Prefix for return created')

    prefix = 'k_rdl_prefix_takingmode'
    try:
        Config.get(poste='TOUS', cle='cfg', propriete=prefix)
        print('Prefix for taking mode exists key exists')
    except DoesNotExist:
        zpl_cfg = Config()
        zpl_cfg.ref_id = '_RDLPREF02'
        zpl_cfg.poste = 'TOUS'
        zpl_cfg.cle = 'cfg'
        zpl_cfg.propriete = prefix
        zpl_cfg.type_value = 'C'
        zpl_cfg.value = '255001'
        zpl_cfg.hierarchie_menu = """RIEDL\\Codebarre\\Prefix Prelevement"""
        zpl_cfg.type_champ = 'reg'
        zpl_cfg.ordre = 11
        zpl_cfg.save()
        print('Prefix for taking mode created')

    # Activate/deactivate print label for container
    prefix = 'k_rdl_print_container'
    try:
        Config.get(poste='TOUS', cle='cfg', propriete=prefix)
        print('Print container exists key exists')
    except DoesNotExist:
        pc_cfg = Config()
        pc_cfg.ref_id = '_RDLPREF03'
        pc_cfg.poste = 'TOUS'
        pc_cfg.cle = 'cfg'
        pc_cfg.propriete = prefix
        pc_cfg.type_value = 'N'
        pc_cfg.value = '0'
        pc_cfg.hierarchie_menu = """RIEDL\\Codebarre\\Etiquette Contenant"""
        pc_cfg.type_champ = 'reg'
        pc_cfg.ordre = 12
        pc_cfg.save()
        print('Print container number created')

    # Activate/deactivate print label for container
    prefix = 'k_rdl_prefix_container'
    try:
        Config.get(poste='TOUS', cle='cfg', propriete=prefix)
        print('Prefix container exists key exists')
    except DoesNotExist:
        pc_cfg = Config()
        pc_cfg.ref_id = '_RDLPREF04'
        pc_cfg.poste = 'TOUS'
        pc_cfg.cle = 'cfg'
        pc_cfg.propriete = prefix
        pc_cfg.type_value = 'C'
        pc_cfg.value = ''
        pc_cfg.hierarchie_menu = """RIEDL\\Codebarre\\Prefix Contenant"""
        pc_cfg.type_champ = 'reg'
        pc_cfg.ordre = 13
        pc_cfg.save()
        print('Prefix container number created')

    # Activation sortie globale
    # k_avec_riedl
    prefix = 'k_avec_riedl'
    try:
        Config.get(poste='TOUS', cle='cfg', propriete=prefix)
        print('Display unload global exists key exists')
    except DoesNotExist:
        zpl_cfg = Config()
        zpl_cfg.ref_id = '_RDLAFFI01'
        zpl_cfg.poste = 'TOUS'
        zpl_cfg.cle = 'cfg'
        zpl_cfg.propriete = prefix
        zpl_cfg.type_value = 'N'
        zpl_cfg.value = '1'
        zpl_cfg.hierarchie_menu = """RIEDL\\Affichage\\Sortie globale"""
        zpl_cfg.type_champ = 'num'
        zpl_cfg.ordre = 1
        zpl_cfg.save()
        print('Display unload global created')
