# Use by command flask fix
import logging

import bcrypt
from common.models import WebLang, WebThresholdParameter
from median.constant import EcoType
from median.database import mysql_db, decripte
from median.models import Profil, Config, Compteur
from median.models import User, CatalogType, CatalogVersion, CatalogOption, CatalogUnitStock, CatalogLines, \
    CatalogColumns, Magasin, EquipmentUnitStock, Adresse, Stock, CodeBlocage, MagasinFormat, Format, Patient, Sejour
from peewee import DoesNotExist, JOIN
from ressources.equipments.Version import EquipmentType, Version
from ressources.equipments.acced.acceds import regenerate

logger = logging.getLogger('median.webserver')


def fix_foreign_key():
    """Delete and recreate the foreign key"""
    logger.info("Recreate all foreign key")
    execute_ddl_query("ALTER TABLE web_menu_i18n DROP FOREIGN KEY web_menu_i18n_ibfk_2;")
    execute_ddl_query(
        "ALTER TABLE web_menu_i18n ADD CONSTRAINT web_menu_i18n_ibfk_2 FOREIGN KEY (lang_id) REFERENCES web_lang(id) ON DELETE CASCADE ON UPDATE CASCADE;")  # noqa
    execute_ddl_query("ALTER TABLE web_menu_i18n DROP FOREIGN KEY web_menu_i18n_ibfk_1;")
    execute_ddl_query(
        "ALTER TABLE web_menu_i18n ADD CONSTRAINT web_menu_i18n_ibfk_1 FOREIGN KEY (menu_id) REFERENCES web_menu(id) ON DELETE CASCADE;")  # noqa
    execute_ddl_query("ALTER TABLE web_menu DROP FOREIGN KEY web_menu_ibfk_1;")
    execute_ddl_query(
        "ALTER TABLE web_menu ADD CONSTRAINT web_menu_ibfk_1 FOREIGN KEY (parent_id) REFERENCES web_menu(id) ON DELETE CASCADE ON UPDATE CASCADE;")  # noqa
    execute_ddl_query("ALTER TABLE web_form_i18n DROP FOREIGN KEY web_form_i18n_ibfk_1;")
    execute_ddl_query(
        "ALTER TABLE web_form_i18n ADD CONSTRAINT web_form_i18n_ibfk_1 FOREIGN KEY (lang_id) REFERENCES web_lang(id) ON DELETE CASCADE ON UPDATE CASCADE;")  # noqa


def fix_datas():
    """Fix inconsistent datas"""
    execute_ddl_query("UPDATE f_profil SET x_visu=1 WHERE x_visu IS NULL;")
    execute_ddl_query("UPDATE f_profil SET x_edit=0 WHERE x_edit IS NULL;")
    execute_ddl_query("UPDATE f_mag SET x_libelle=x_type_mag WHERE x_libelle = '';")
    execute_ddl_query("UPDATE f_dest SET x_type_dest='SERVICE' WHERE x_type_dest = '';")
    # si vide, alors utilisation du paramèretre k_eco_tri_pil
    # execute_ddl_query("UPDATE f_dest SET x_tri='CHAMBRE' WHERE x_tri = '' AND x_type_dest='SERVICE';")
    execute_ddl_query("UPDATE f_user SET x_astup = 0 WHERE x_astup IS NULL;")
    execute_ddl_query("UPDATE f_user SET x_login = '' WHERE x_login IS NULL;")
    execute_ddl_query("UPDATE f_user SET x_url = '' WHERE x_url IS NULL;")
    execute_ddl_query("UPDATE f_user SET x_badge = '' WHERE x_badge IS NULL;")
    execute_ddl_query("UPDATE f_user SET x_inventaire = 0 WHERE x_astup IS NULL;")
    execute_ddl_query(
        "UPDATE f_user SET x_inventaire = 1, x_maintenance = 1, x_service = 'DEENOVA' WHERE x_profil = 'ECO-DEX';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_inventaire = 1, x_maintenance = 1, x_service = 'DEENOVA' WHERE x_profil = 'DEENOVA';")  # noqa
    execute_ddl_query(
        "UPDATE f_mag SET x_type_machine='RIEDL_2' WHERE x_type_machine='' AND x_eco_type='L';")  # noqa
    # Old median client failed to display datas when x_id is blank,
    execute_ddl_query(
        "UPDATE f_config SET x_id = CONCAT('_PK', LPAD(x_pk, 7, '0')) WHERE x_poste = 'TOUS' AND x_cle = 'cfg' AND x_id = '';")  # noqa
    execute_ddl_query("UPDATE f_defaut SET x_manuel=1 WHERE x_valeur in (0, 1);")  # noqa


def deenova_profil_default():
    get_or_create_profil("WEB_SETTINGS_BLOCKING_DEFECTS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_INVENTORY", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_STOCK", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # ACCED
    get_or_create_profil("WEB_ACCED", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_REAPPRO", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_DISPENSATION", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_DASHBOARD_STATS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ACCED_FEASIBILITY", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # get_or_create_profil("WEB_ACCED_HISTORY_MESSAGES", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_WITHOUT_TRESHOLD", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_SETTING_ACCED_MESSAGES", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ACCED_ZERO_CONSUMPTION", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ACCED_WAITING_MESSAGES", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ACCED_DETAIL", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_SAVED_LIST", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_THRESHOLD", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ACCED_BLOCK_CAUSES", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # TODO: active when simulator is fully merge
    # get_or_create_profil("WEB_ACCED_THRESHOLD_SIMULATOR", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # Administration
    get_or_create_profil("WEB_ADMINISTRATION", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_HISTORY_INTERFACE", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ORGANILOG", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ACTIONS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PATIENTS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_WSCLIENT_ERROR", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # ASTUS
    get_or_create_profil("WEB_ASTUS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_FEASIBILITY", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_STOCK", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_DRAWER_SEARCH", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_EVENTS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_REPLENISHMENT", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_DETAIL", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_WARD_CONFIGURATION", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_INVENTORY", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_OUTPUT", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_CONSUMPTION", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # Dashboard
    get_or_create_profil("WEB_DASHBOARD", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_DASHBOARD_COMPTEUR", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # Document
    get_or_create_profil("WEB_PICKING_REPORT", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_HISTORY", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_LOCATIONS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PRESCRIPTION", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # get_or_create_profil("WEB_BLOCKED_HISTORY", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # Reference
    get_or_create_profil("WEB_INVENTORY", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PRODUIT", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PRODUIT_BATCH", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PRODUIT_BLACKLIST", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PRODUIT_PARAM", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PRODUIT_PARAM_UCD", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PRODUIT_THRESHOLD", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_REF_ASTUS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_REF_FORMAT", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_REF_RIEDL", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_RISKY_DRUGS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # Riedl
    get_or_create_profil("WEB_RIEDL", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_RIEDL_INPUT", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_RIEDL_OUTPUT", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_RIEDL_STOCK_EXPORT", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # Settings
    get_or_create_profil("WEB_GROUP", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PARAMETER", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PEIGNES", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_SETTINGS_BLOCKING_DEFECTS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_SETTING_LANG", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_USERS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_SETTING_EQUIPMENT", ["DEENOVA", "ECO-DEX"])
    get_or_create_profil("WEB_CATALOG", ["DEENOVA", "ECO-DEX"])
    # Wards
    get_or_create_profil("WEB_SERVICES", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # External
    get_or_create_profil("WEB_EXTERNAL", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_EXTERNAL_STOCK", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_RIEDL_REPLENISH", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_LOCATIONS_GENERATOR", ["DEENOVA", "ECO-DEX", "ADMIN"])


def deenova_user_lang_default():
    """Add default lang on deenova user"""
    execute_ddl_query("UPDATE f_user SET x_lang = 'fr_FR' WHERE x_profil = 'ECO-DEX' AND x_lang='';")
    execute_ddl_query("UPDATE f_user SET x_lang = 'fr_FR' WHERE x_profil = 'DEENOVA' AND x_lang='';")
    execute_ddl_query("UPDATE f_user SET x_lang = 'en_GB' WHERE x_user = 'ecopilote';")


def deenova_user_rgpd():
    """Add email on original database user"""
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'ecopilote@deenova.com', x_login = 'ecopilote@deenova' WHERE x_user='ecopilote';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'l.lebrec@deenova.com', x_login = 'l.lebrec@deenova' WHERE x_user='Lucas LEBREC';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'a.marchand@deenova.com', x_login = 'a.marchand@deenova' WHERE x_user='Anthony MARCHAND';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'd.pinault@deenova.com', x_login = 'd.pinault@deenova', x_user = 'Dimitry PINAULT' WHERE x_user='Dimitry';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'h.tanneau@deenova.com', x_login = 'h.tanneau@deenova', x_user = 'Hugo TANNEAU' WHERE x_user='Hugo';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'g.brindeau@deenova.com', x_login = 'g.brindeau@deenova', x_user = 'Guillaume BRINDEAU' WHERE x_user='Guillaume';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'r.cousin@deenova.com', x_login = 'r.cousin@deenova', x_user = 'Romain COUSIN' WHERE x_user='Romain COUSIN';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'a.letessier@deenova.com', x_login = 'a.letessier@deenova', x_user = 'Alexandre LETESSIER1' WHERE x_user='Alexandre';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'j.cassin@deenova.com', x_login = 'j.cassin@deenova', x_user = 'Jérôme CASSIN' WHERE x_user='Jérôme CASSIN';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'jm.foucault@deenova.com', x_login = 'jm.foucault@deenova', x_user = 'Jean-Marie FOUCAULT' WHERE x_user='JM FOUCAULT';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'c.levine@deenova.com', x_login = 'c.levine@deenova', x_user = 'Cédric LEVINE' WHERE x_user='Cédric LEVINE';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'p.jeanneau@deenova.com', x_login = 'p.jeanneau@deenova', x_user = 'Pierre JEANNEAU' WHERE x_user='Pierre JEANNEAU';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'x.thoirey@deenova.com', x_login = 'x.thoirey@deenova.com', x_user = 'Xavier THOIREY' WHERE x_user='Xavier';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'd.mogne@deenova.com', x_login = 'd.mogne@deenova', x_user = 'Djalalidine MOGNE' WHERE x_user='Djalal';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'a.rebel@deenova.com', x_login = 'a.rebel@deenova', x_user = 'Alan REBEL' WHERE x_user='REBEL Alan';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'c.chauvet@deenova.com', x_login = 'c.chauvet@deenova', x_user = 'Christophe CHAUVET' WHERE x_user='Christophe CHAUVET';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'a.abdillah@deenova.com', x_login = 'a.abdillah@deenova', x_user = 'Archimède ABDILLAH' WHERE x_user='Archimède ABDILLAH';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'm.mouquet@deenova.com', x_login = 'm.mouquet@deenova.com', x_user = 'Maxime MOUQUET' WHERE x_user='Maxime';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'g.briant@deenova.com', x_login = 'g.briant@deenova', x_user = 'Guéric BRIANT' WHERE x_user='Gueric BRIANT';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 's.jegou@deenova.com', x_login = 's.jegou@deenova', x_user = 'Suzon JEGOU' WHERE x_user='Suzon';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'f.jaulneau@deenova.com', x_login = 'f.jaulneau@deenova', x_user = 'Florian Jaulneau' WHERE x_user='Florian';")  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_email = 'd.slomczynski@deenova.com', x_login = 'd.slomczynski@deenova', x_user = 'Dylan SLOMCZYNSKI' WHERE x_user='Dylan Slomczynski';")  # noqa

    # Delete old user
    execute_ddl_query("DELETE FROM f_user WHERE x_user = 'Aymeric GERARD' AND x_profil='ECO-DEX';")
    execute_ddl_query("DELETE FROM f_user WHERE x_user = 'Laurent BAGHDIKIAN' AND x_profil='ECO-DEX';")
    execute_ddl_query("DELETE FROM f_user WHERE x_user = 'Matthieu PAPOT' AND x_profil='ECO-DEX';")


def deenova_default_labels():
    """Check if blocking default is available on the database"""
    get_or_create_default(0, "Déblocage")
    get_or_create_default(1, "Blocage Median")


def deenova_default_format():
    get_or_create_format("BOITE PASS", {'nb_div': 2, 'dim_y': 2})
    get_or_create_format("BAC", {'nb_div': 1})
    get_or_create_format("SHELF", {})
    get_or_create_format("BAC 1/3 B", {'typeBac': 'bP1/3'})
    get_or_create_format("BAC 1/3 H", {'typeBac': 'hP1/3'})
    get_or_create_format("BAC 1/7 B", {'typeBac': 'bP1/7'})
    get_or_create_format("BAC 1/7 H", {'typeBac': 'hP1/7'})
    get_or_create_format("PIL 1/3 B", {'typeBac': 'bL1/3', 'nb_div': 2})
    get_or_create_format("PIL 1/3 H", {'typeBac': 'hL1/3', 'nb_div': 2})
    get_or_create_format("TIROIR 1/3", {'nb_div': 6, 'typeBac': 'tir1/3'})
    get_or_create_format("TIROIR 1/4", {'nb_div': 8, 'typeBac': 'tir1/4'})
    get_or_create_format("TIROIR 1/5", {'nb_div': 10, 'typeBac': 'tir1/5'})
    get_or_create_format("CAISSE BP", {'dim_x': 600, 'dim_y': 400, 'dim_z': 220, 'nb_div': 18})
    get_or_create_format("CAISSE EXO", {'dim_x': 600, 'dim_y': 400, 'dim_z': 220, 'nb_div': 12})


def update_password_web():
    users = User.select().where(User.passwordWeb == '')

    for usr in users:
        old_password = decripte(usr.password)
        new_password = bcrypt.hashpw(old_password.encode('UTF_8'), bcrypt.gensalt())
        User.update(passwordWeb=new_password, isTemporary=1).where(User.pk == usr.pk).execute()


def update_language():
    """
    Enable English and french language by default
    """
    WebLang.update(enable=True).where(WebLang.code == 'fr_FR').execute()
    WebLang.update(enable=True).where(WebLang.code == 'en_GB').execute()


def update_loading_ids():
    load_counters = (Compteur.select(Compteur.val, Compteur.cle)
                     .where(Compteur.cle.startswith("ID_CHARGEMENT_"))
                     .order_by(Compteur.val.asc()))

    last_counter = None
    for counter in load_counters:
        equipment = counter.cle[14:]

        try:
            config = (Config.select()
                      .where((Config.propriete == 'k_min_chargement') & (Config.cle == equipment))
                      .order_by(Config.value)).get()
        except DoesNotExist:
            config = Config()
            config.propriete = 'k_min_chargement'
            config.type_value = 'N'
            config.cle = equipment
            config.poste = 'TOUS'

        if last_counter is None:
            value = 0
            last_counter = 0
        else:
            current = str(counter.val)
            value = current[:1].ljust(len(current), '0')
            last_counter = int(value)

        config.value = value
        config.save()


def update_contract_threshold():
    _update_config(property='contract_dose_pillbox', value=8)
    _update_config(property='contract_cadence_pillbox', value=50)
    _update_config(property='contract_cadence_dose_pillbox', value=400)
    _update_config(property='contract_dose_stack', value=8)
    _update_config(property='contract_cadence_stack', value=40)
    _update_config(property='contract_cadence_dose_stack', value=320)
    _update_config(property='contract_completion_percentage', value=98)
    _update_config(property='contract_availability_percentage', value=96)
    _update_config(property='contract_cut_rejet_percentage', value=5)
    _update_config(property='contract_blocked_box_per_hour', value=10)
    _update_config(property='contract_duplicate_per_thousand', value=4)


def _update_config(property, value):
    try:
        config = (Config.select()
                  .where((Config.propriete == property))
                  .order_by(Config.value)).get()
    except DoesNotExist:
        config = Config()
        config.propriete = property
        config.type_value = 'N'
        config.poste = 'TOUS'

    config.value = value
    config.save()


def update_threshold_parameters():
    equipments = Magasin.select(Magasin.pk).where(Magasin.eco_type << [EcoType.Coupe.value, EcoType.Cueillette.value])
    WebThresholdParameter.delete().where(WebThresholdParameter.equipment_pk.not_in(equipments)).execute()
    for equipment in equipments:
        parameter = WebThresholdParameter.select(WebThresholdParameter.pk).where(
            WebThresholdParameter.equipment_pk == equipment.pk)

        if not any(parameter):
            WebThresholdParameter.create(
                equipment_pk=equipment.pk)


def fix_addresse_state_a():
    """Add block =1 on stock and state X on address"""
    addrs_query = Adresse.select(Adresse.adresse).where(Adresse.etat == 'A')
    addrs = [a.adresse for a in addrs_query]
    logger.info("We have %i addresses with bad state!" % len(addrs))
    Stock.update(bloque=1).where(Stock.adresse << addrs).execute()
    Adresse.update(etat='X', bloque=1, bloque_message='à controler').where(Adresse.etat == 'A').execute()


def execute_ddl_query(query):
    """Execute a SQL query"""
    # print(query)
    mysql_db.execute_sql(query)


def get_or_create_profil(ressource, groups=[], view=1, create=0, update=1, delete=0):
    """
    Check if profil exists, if not we create it
    """
    for grp in groups:
        try:
            Profil.get(profil=grp, ressource=ressource)
        except DoesNotExist:
            prof = Profil()
            prof.profil = grp
            prof.ressource = ressource
            prof.visu = view
            prof.cree = create
            prof.edit = update
            prof.supp = delete
            prof.save()


def get_or_create_default(code: int, label: str) -> None:
    _, created = CodeBlocage.get_or_create(valeur=code, defaults={
        'libelle': label,
    })


def get_or_create_format(code: str, def_value: dict) -> None:
    """Check if format exists or create it"""
    _, created = Format.get_or_create(format=code, defaults=def_value)


def generate_equipment_topography():
    equipments = (Magasin.select(Magasin)
                  .join(EquipmentUnitStock, JOIN.LEFT_OUTER,
                        on=EquipmentUnitStock.equipment_pk == Magasin.pk)
                  .where(EquipmentUnitStock.pk.is_null()))

    for eq in equipments:
        regenerate(eq)


def update_catalog():
    add_riedls()
    add_acceds()
    add_aide()
    add_astus()
    add_externe()


def add_externe():
    equipment = add_type(EquipmentType.EXTERNE.value, EcoType.Externe.value)
    version = add_version(Version.ETAGERE_V1.value, equipment.pk)
    add_option('LIGNE', 1, version.pk)
    add_option('COLONNE', 1, version.pk)
    add_option('PROFONDEUR', 1, version.pk)
    version = add_version(Version.CHARIOT_V1.value, equipment.pk)
    add_option('LIGNE', 1, version.pk)
    add_option('COLONNE', 1, version.pk)
    add_option('PROFONDEUR', 1, version.pk)


def astus_version(label, col, equipment_pk, sub_version):
    version = add_version(f'ASTUS_{label}', equipment_pk, sub_version=sub_version)
    add_option('PROFONDEUR', 2 if sub_version != 'XL' else 1, version.pk)
    nb_1_3 = (12 - col)

    unit_stock = add_unit_stock(total_line=23, total_column=col, order=0, version_pk=version.pk)
    lines_h = add_unit_stock_lines(start=0, end=10, unit_stock_pk=unit_stock.pk)

    end_1_7 = (2 * (6 - nb_1_3)) - 1
    if nb_1_3 > 0:
        add_unit_stock_columns(start=2 * (6 - nb_1_3), end=col - 2, line_pk=lines_h.pk, container_type='BAC 1/3 H')
    else:
        end_1_7 = end_1_7 - 1

    add_unit_stock_columns(start=0, end=end_1_7, line_pk=lines_h.pk, container_type='BAC 1/7 H')

    lines_b = add_unit_stock_lines(start=11, end=22, unit_stock_pk=unit_stock.pk)
    add_unit_stock_columns(start=0, end=(2 * (6 - nb_1_3)) - 1, line_pk=lines_b.pk, container_type='BAC 1/7 B')
    if nb_1_3 > 0:
        add_unit_stock_columns(start=2 * (6 - nb_1_3), end=col - 1, line_pk=lines_b.pk, container_type='BAC 1/3 B')


def add_astus():
    equipment = add_type(EquipmentType.ASTUS.value, EcoType.Astus.value)

    for col in range(6, 12):
        astus_version(label='V3', col=col,
                      equipment_pk=equipment.pk, sub_version=str(col))

    astus_version(label='V3', col=6,
                  equipment_pk=equipment.pk, sub_version='XL')


def add_aide():
    equipment = add_type(EquipmentType.AIDE.value, EcoType.Coupe.value)

    # Aide Cut V1
    version = add_version(Version.AIDE_CUT_V1.value, equipment.pk)
    unit_stock = add_unit_stock(total_line=3, total_column=6, order=0, version_pk=version.pk)
    lines = add_unit_stock_lines(start=0, end=2, unit_stock_pk=unit_stock.pk)
    add_unit_stock_columns(start=0, end=5, line_pk=lines.pk, container_type='BOITE PASS')

    # Aide Pick V1
    version = add_version(Version.AIDE_PICK_V1.value, equipment.pk)
    for stock in range(18):
        unit_stock = add_unit_stock(total_line=18, total_column=4, order=stock, version_pk=version.pk)
        if stock == 1:
            lines = add_unit_stock_lines(start=0, end=6, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type='BOITE PASS')
            lines = add_unit_stock_lines(start=7, end=17, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type='BOITE PASS')
        elif stock == 2 or stock == 3:
            lines = add_unit_stock_lines(start=0, end=2, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type='BOITE PASS')
            lines = add_unit_stock_lines(start=3, end=17, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type='')
        elif 4 <= stock <= 5:
            lines = add_unit_stock_lines(start=0, end=9, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type='BOITE PASS')
            lines = add_unit_stock_lines(start=10, end=17, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type='')
        elif 6 <= stock <= 8:
            lines = add_unit_stock_lines(start=0, end=3, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type='BOITE PASS')
            lines = add_unit_stock_lines(start=4, end=9, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type='')
            lines = add_unit_stock_lines(start=10, end=17, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type='BAC')
        else:
            lines = add_unit_stock_lines(start=0, end=17, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type='BOITE PASS')

    # Aide Classic
    version = add_version(Version.AIDE_V2.value, equipment.pk)
    for stock in range(10):
        unit_stock = add_unit_stock(total_line=18, total_column=4, order=stock, version_pk=version.pk)
        lines = add_unit_stock_lines(start=0, end=17, unit_stock_pk=unit_stock.pk)
        add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type='BOITE PASS')


def add_acceds():
    equipment = add_type(EquipmentType.ACCED.value, EcoType.Cueillette.value)

    # ACCED V3
    version = add_version(Version.ACCED_V3.value, equipment.pk)
    for stock in range(10):
        unit_stock = add_unit_stock(total_line=13, total_column=4, order=stock, version_pk=version.pk)
        if stock < 9:
            lines = add_unit_stock_lines(start=0, end=2, unit_stock_pk=unit_stock.pk)
        else:
            lines = add_unit_stock_lines(start=0, end=4, unit_stock_pk=unit_stock.pk)

        add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type='BOITE PASS')

        if stock <= 4:
            lines = add_unit_stock_lines(start=3, end=12, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=0, line_pk=lines.pk, container_type='TIROIR 1/4')
        elif stock <= 7:
            lines = add_unit_stock_lines(start=3, end=12, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=0, line_pk=lines.pk, container_type='TIROIR 1/5')
        elif stock == 8:
            lines = add_unit_stock_lines(start=3, end=12, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=0, line_pk=lines.pk, container_type='TIROIR 1/3')

    add_option('PROFONDEUR', 2, version.pk)

    # ACCED V2
    version = add_version(Version.ACCED_V2.value, equipment.pk)
    for stock in range(10):
        unit_stock = add_unit_stock(total_line=18, total_column=4, order=stock, version_pk=version.pk)
        lines = add_unit_stock_lines(start=0, end=17, unit_stock_pk=unit_stock.pk)
        add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type='BOITE PASS')

    add_option('PROFONDEUR', 2, version.pk)


def add_riedls():
    # RIEDL PHASYS
    equipment = add_type(EquipmentType.RIEDL.value, EcoType.Riedl.value)
    version = add_version(Version.RIEDL_V2.value, equipment.pk)
    add_option('ENTREE', 1, version.pk)
    add_option('ENTREE_AUTO', 0, version.pk)
    add_option('SORTIE', 1, version.pk)
    add_option('URGENCE', 1, version.pk)
    unit_stock = add_unit_stock(total_line=1, total_column=1, order=0, version_pk=version.pk)
    lines = add_unit_stock_lines(start=0, end=0, unit_stock_pk=unit_stock)
    add_unit_stock_columns(start=0, end=0, line_pk=lines.pk, container_type='SHELF')


def fill_mag_format_table():
    equipments = Magasin.select(Magasin.mag).where(
        Magasin.eco_type << [EcoType.Coupe.value, EcoType.Cueillette.value, EcoType.Astus.value, EcoType.Riedl.value]
    )
    for equip in equipments:
        # Retrieve all format define per equipment,
        AdrFormatList = Adresse.select(Adresse.format).distinct().where(Adresse.magasin == equip.mag)
        for CurrentFormat in AdrFormatList:
            MagasinFormat.get_or_create(magasin=equip.mag, format=CurrentFormat.format)


def add_type(label, ecoType):
    try:
        type = CatalogType.get(CatalogType.type == label)
        versions = CatalogVersion.select(CatalogVersion.pk).where(CatalogVersion.type_pk == type.pk)
        unit_stocks = CatalogUnitStock.select(CatalogUnitStock.pk).where(CatalogUnitStock.version_pk << versions)
        lines = CatalogLines.select(CatalogLines.pk).where(CatalogLines.unit_stock_pk << unit_stocks)

        CatalogColumns.delete().where(CatalogColumns.line_pk << lines).execute()
        CatalogLines.delete().where(CatalogLines.unit_stock_pk << unit_stocks).execute()
        CatalogUnitStock.delete().where(CatalogUnitStock.pk << versions).execute()

        CatalogOption.delete().where(CatalogOption.version_pk << versions).execute()
        CatalogVersion.delete().where(CatalogVersion.type_pk == type.pk).execute()
        CatalogType.delete().where(CatalogType.type == label).execute()

    except DoesNotExist:
        type = None

    equipement = CatalogType()
    equipement.type = label
    equipement.eco_type = ecoType
    equipement.save()
    return equipement


def add_version(label, type_pk, sub_version=''):
    version = CatalogVersion()
    version.numVersion = label
    version.type_pk = type_pk
    version.subVersion = sub_version
    version.save()
    return version


def add_option(label, value, version_pk):
    option = CatalogOption()
    option.version_pk = version_pk
    option.type = label
    option.value = value
    option.save()


def add_unit_stock(total_line, total_column, order, version_pk):
    unit_stock = CatalogUnitStock()
    unit_stock.version_pk = version_pk
    unit_stock.total_line = total_line
    unit_stock.total_colonne = total_column
    unit_stock.order = order
    unit_stock.save()
    return unit_stock


def add_unit_stock_lines(start, end, unit_stock_pk):
    lines = CatalogLines()
    lines.unit_stock_pk = unit_stock_pk
    lines.start_line = start
    lines.end_line = end
    lines.save()
    return lines


def add_unit_stock_columns(start, end, container_type, line_pk):
    if start <= end:
        column = CatalogColumns()
        column.line_pk = line_pk
        column.start_column = start
        column.end_Column = end
        column.container_type = container_type
        column.save()


def check_episodes_from_patients():
    query_patients = Patient.select(Patient.ipp, Patient.nom, Patient.prenom).order_by(Patient.pk.asc())
    cursor_patient = mysql_db.execute(query_patients)
    updated_count = 0
    processed_patients_count = 0

    for (patient_ipp, nom, prenom) in cursor_patient:
        processed_patients_count += 1
        print('[%+8s] %s -> %s %s' % (processed_patients_count, patient_ipp, nom, prenom))

        query_episodes = Sejour.select(Sejour.pk, Sejour.ipp, Sejour.date_entree, Sejour.date_sortie
                                       ).where(Sejour.ipp == patient_ipp).order_by(Sejour.pk.desc())

        cursor_sejours = mysql_db.execute(query_episodes)

        prec_date_entree = None
        for (sejour_pk, sejour_ipp, date_entree, date_sortie) in cursor_sejours:
            print('%+14sentry:%s |  exit:%s | pk:%i | prec_date:%s' % (
                ' ', str(date_entree), str(date_sortie), sejour_pk, str(prec_date_entree)),
                    end=" "
            )
            if prec_date_entree is None:
                # first and most recent episode, no action needed
                pass
            else:
                if date_sortie is None:
                    Sejour.update(date_sortie=prec_date_entree).where(Sejour.pk == sejour_pk).execute()
                    updated_count += 1
                    print('(updated)', end=" ")

            prec_date_entree = date_entree

            print('')

    print('%i entries updated' % updated_count)
