import base64
import json
import logging
import os
import sys
import time
import uuid
from datetime import date, timedelta
from datetime import datetime
from logging.handlers import TimedRotatingFileHandler
from pathlib import Path

import bcrypt
import click
import flask
import rsa
from flask import (
    Flask, session, redirect, request,
    send_from_directory, send_file, jsonify, abort
)
from flask_jwt_extended import JWTManager, create_refresh_token, create_access_token, jwt_required, get_jwt_identity
from flask_sock import Sock
from flask_wtf import CSRFProtect
from flask_wtf.csrf import CSRFError, generate_csrf
from median.models import User, HistoriqueIdentification

from common.command import register_commands
from common.database import sync_profil
from common.status import HTTP_404_NOT_FOUND, HTTP_401_UNAUTHORIZED, HTTP_202_ACCEPTED, HTTP_400_BAD_REQUEST, \
    HTTP_200_OK, HTTP_204_NO_CONTENT, HTTP_301_MOVED_PERMANENTLY
from common.translations import generate_resources_files, generate_translations_files, generate_translations_lang
from ressources.acced.acced_batch_blueprint import acced_batch_blueprint
from ressources.acced.acced_blueprint import acced_blueprint
from ressources.acced.acced_consumption_blueprint import acced_consumption_blueprint
from ressources.acced.acced_message_blueprint import acced_message_blueprint
from ressources.acced.acced_output_blueprint import acced_output_blueprint
from ressources.acced.acced_reappro_blueprint import acced_reappro_blueprint
from ressources.acced.acced_replenish_blueprint import acced_replenish_blueprint
from ressources.acced.acced_saved_blueprint import acced_saved_blueprint
from ressources.acced.acced_setting_message_blueprint import acced_setting_message_blueprint
from ressources.acced.acced_without_threshold_blueprint import acced_without_threshold_blueprint
from ressources.acced.block_cause.block_cause_blueprint import block_causes_blueprint
from ressources.acced.production.acced_production_blueprint import acced_production_blueprint
from ressources.acced.ward_blueprint import ward_blueprint
from ressources.astus.astus_blueprint import astus_blueprint
from ressources.astus.consumption.astus_consumption_blueprint import astus_consumption_blueprint
from ressources.astus.astus_drawer_blueprint import astus_drawer_blueprint
from ressources.astus.astus_filled_blueprint import astus_filled_blueprint
from ressources.astus.astus_logs_blueprint import astus_logs_blueprint
from ressources.astus.astus_nurse_stock_blueprint import astus_nurse_stock_blueprint
from ressources.astus.astus_output_blueprint import astus_output_blueprint
from ressources.astus.astus_stat_blueprint import astus_stats_blueprint
from ressources.astus.astus_stock_blueprint import astus_stock_blueprint
from ressources.astus.astus_ward_blueprint import astus_ward_blueprint
from ressources.blueprint.account_blueprint import account_blueprint
from ressources.blueprint.actions_blueprint import actions_blueprint
from ressources.blueprint.adress_blueprint import address_blueprint
from ressources.blueprint.available_adresses_blueprint import available_adresses_blueprint
from ressources.blueprint.batch_blueprint import batch_blueprint
from ressources.blueprint.blocked_history.blocked_history_blueprint import blocked_history_blueprint
from ressources.blueprint.blocking_defects_blueprint import blocking_defects_blueprint
from ressources.blueprint.catalog_blueprint import catalog_blueprint
from ressources.blueprint.cip_blueprint import cip_blueprint
from ressources.blueprint.config.config_blueprint import config_blueprint
from ressources.blueprint.container_format_blueprint import container_format_blueprint
from ressources.blueprint.external.external_blueprint import external_blueprint
from ressources.blueprint.external.external_replenish_blueprint import external_replenish_blueprint
from ressources.blueprint.external.external_stock_blueprint import external_stock_blueprint
from ressources.blueprint.feasibility_blueprint import feasibility_blueprint
from ressources.blueprint.form_blueprint import form_blueprint
from ressources.blueprint.format_blueprint import format_blueprint
from ressources.blueprint.global_blueprint import global_blueprint
from ressources.blueprint.global_output_blueprint import global_output_blueprint
from ressources.blueprint.gtin_blueprint import gtin_blueprint
from ressources.blueprint.gtin_cip_blueprint import gtin_cip_blueprint
from ressources.blueprint.historic_blueprint import historic_blueprint
from ressources.blueprint.importref.import_ref_blueprint import importref_blueprint
from ressources.blueprint.input_lists_blueprint import input_lists_blueprint
from ressources.blueprint.interface_blueprint import interface_blueprint
from ressources.blueprint.inventory_blueprint import inventory_blueprint
from ressources.blueprint.languages_blueprint import languages_blueprint
from ressources.blueprint.locations.locations_blueprint import locations_blueprint
from ressources.blueprint.move_container_blueprint import move_container_blueprint
from ressources.blueprint.organilog_ticket_blueprint import organilog_ticket_blueprint
from ressources.blueprint.peigne_blueprint import peignes_blueprint
from ressources.blueprint.prescriptions_blueprint import prescriptions_blueprint
from ressources.blueprint.product_blueprint import product_blueprint
from ressources.blueprint.profil_blueprint import profils_blueprint
from ressources.blueprint.reference_blueprint import reference_blueprint
from ressources.blueprint.reporting_blueprint import reporting_blueprint
from ressources.blueprint.riedl_blueprint import riedl_blueprint, get_riedls, get_diff
from ressources.blueprint.rights_blueprint import rights_blueprint
from ressources.blueprint.stock.ref_stock_blueprint import ref_stock_blueprint
from ressources.blueprint.stock_blueprint import stock_blueprint
from ressources.blueprint.store_blueprint import store_blueprint
from ressources.blueprint.stores.stores_blueprint import stores_blueprint
from ressources.blueprint.suggestions_blueprint import suggestions_blueprint
from ressources.blueprint.threshold.threshold_blueprint import threshold_blueprint
from ressources.blueprint.threshold_simulator.threshold_sim_blueprint import threshold_sim_blueprint
from ressources.blueprint.traceability_blueprint import traceability_blueprint
from ressources.blueprint.trends_blueprint import trends_blueprint
from ressources.blueprint.unit_blueprint import unit_blueprint
from ressources.blueprint.user_blueprint import user_blueprint
from ressources.blueprint.users_blueprint import users_blueprint
from ressources.blueprint.wards_blueprint import wards_blueprint
from ressources.blueprint.withdrawnbatch_blueprint import withdrawnbatch_blueprint
from ressources.blueprint.patients_blueprint import patients_blueprint
from ressources.riedls.riedl_input_blueprint import riedl_input_blueprint
from ressources.riedls.riedl_output_blueprint import riedl_output_blueprint
from ressources.riedls.riedl_output_list_blueprint import riedl_output_list_blueprint
from ressources.riedls.riedl_stock_blueprint import riedl_stock_blueprint
from ressources.riedls.riedl_taking_mode_blueprint import riedl_taking_mode_blueprint
from security import decryptRSA
from ressources.blueprint.transfert.transfert_blueprint import transferts_blueprint


class MedianJSONEncoder(json.JSONEncoder):
    def default(self, o):
        if isinstance(o, date):
            return o.isoformat()

        return super().default(o)


csrf = CSRFProtect()
# Les fichiers HTML sont fournis en statique.
html_dir = os.environ.get('MEDIAN_HTML_DIR', os.path.join('..', '..', 'frontend_v2', 'dist'))
static_dir = os.environ.get('MEDIAN_STATIC_DIR', os.path.join('..', '..', 'frontend_v2', 'dist', 'assets'))

# Le fichier de log se configure dans IIS

# Ajouter Varaible LOG_CONSOLE si on veut le log en console
if os.environ.get("LOG_CONSOLE", False):
    logging.basicConfig(level=logging.INFO)

# The log file must be rotate automatically at midninght
log_file = os.environ.get('MEDIAN_LOG_FILE', 'medianweb.log')
tfh = TimedRotatingFileHandler(log_file, when="midnight", encoding='utf-8')

logging.basicConfig(
    handlers=[tfh],
    level=logging.DEBUG,
    format='%(asctime)s  %(name)-20s  %(levelname)-6s: %(message)s'
)
logger = logging.getLogger('median.webserver')

# Key use to crypt session cookie
if os.environ.get('MEDIAN_SECRET_KEY'):
    s_key = bytes(os.environ['MEDIAN_SECRET_KEY'], encoding='utf8')
else:
    s_key = os.urandom(16)

timeoutSession = 'TIMEOUT_SESSION' in os.environ and int(os.environ['TIMEOUT_SESSION']) or 15

app = Flask(__name__)
# scheduler = APScheduler()
app.config['SOCK_SERVER_OPTIONS'] = {'ping_interval': 25}
sock = Sock(app)

app.config["JWT_SECRET_KEY"] = os.environ.get(
    'MEDIAN_JWT_SECRET_KEY', base64.b64encode(str(uuid.uuid4()).encode('ascii')))
app.config["JWT_ACCESS_TOKEN_EXPIRES"] = timedelta(minutes=30)
app.config["JWT_REFRESH_TOKEN_EXPIRES"] = timedelta(days=1)
app.config['SOCK_SERVER_OPTIONS'] = {'ping_interval': 25}

jwt = JWTManager(app)

# Generic Blueprint
app.register_blueprint(account_blueprint, url_prefix='/api/account')
app.register_blueprint(ward_blueprint, url_prefix='/api/ward')
app.register_blueprint(wards_blueprint, url_prefix='/api/services')
app.register_blueprint(product_blueprint, url_prefix='/api/product')
app.register_blueprint(prescriptions_blueprint, url_prefix='/api/prescriptions')
app.register_blueprint(organilog_ticket_blueprint, url_prefix='/api/tickets')
app.register_blueprint(historic_blueprint, url_prefix='/api/historic')
app.register_blueprint(interface_blueprint, url_prefix='/api/interface')
app.register_blueprint(withdrawnbatch_blueprint, url_prefix='/api/withdrawnbatch')
app.register_blueprint(cip_blueprint, url_prefix='/api/cip')
app.register_blueprint(rights_blueprint, url_prefix='/api/rights')
app.register_blueprint(reference_blueprint, url_prefix='/api/reference')
app.register_blueprint(unit_blueprint, url_prefix='/api/unit')
app.register_blueprint(form_blueprint, url_prefix='/api/forme')
app.register_blueprint(format_blueprint, url_prefix='/api/format')
app.register_blueprint(languages_blueprint, url_prefix='/api/languages')
app.register_blueprint(traceability_blueprint, url_prefix='/api/traceability')
app.register_blueprint(inventory_blueprint, url_prefix='/api/inventory')
app.register_blueprint(user_blueprint, url_prefix='/api/user')
app.register_blueprint(profils_blueprint, url_prefix='/api/profils')
app.register_blueprint(suggestions_blueprint, url_prefix='/api/suggestions')
app.register_blueprint(stock_blueprint, url_prefix='/api/stock')
app.register_blueprint(container_format_blueprint, url_prefix='/api/container_formats')
app.register_blueprint(available_adresses_blueprint, url_prefix='/api/available_adresses')
app.register_blueprint(address_blueprint, url_prefix='/api/adresse')
app.register_blueprint(move_container_blueprint, url_prefix='/api/move_container')
app.register_blueprint(threshold_blueprint, url_prefix='/api/seuils')
app.register_blueprint(input_lists_blueprint, url_prefix='/api/liste_entrees')
app.register_blueprint(gtin_blueprint, url_prefix='/api/ucd')
app.register_blueprint(gtin_cip_blueprint, url_prefix='/api/ucd_cip')
app.register_blueprint(reporting_blueprint, url_prefix='/api/reporting')
app.register_blueprint(global_output_blueprint, url_prefix='/api/sortie_globale')
app.register_blueprint(global_blueprint, url_prefix='/api/globale')
app.register_blueprint(actions_blueprint, url_prefix='/api/actions')
app.register_blueprint(threshold_sim_blueprint, url_prefix='/api/threshold_sim')
app.register_blueprint(feasibility_blueprint, url_prefix='/api/feasibility')
app.register_blueprint(importref_blueprint, url_prefix='/api/importref')
app.register_blueprint(blocking_defects_blueprint, url_prefix='/api/blockingdefects')
app.register_blueprint(catalog_blueprint, url_prefix='/api/catalog')
app.register_blueprint(batch_blueprint, url_prefix='/api/batch')
app.register_blueprint(blocked_history_blueprint, url_prefix='/api/blockedhistoric')
app.register_blueprint(ref_stock_blueprint, url_prefix='/api/reference/stock')
app.register_blueprint(patients_blueprint, url_prefix='/api/patients')
app.register_blueprint(transferts_blueprint, url_prefix='/api/transfert')

# Parameters Blueprint
app.register_blueprint(users_blueprint, url_prefix='/api/users')
app.register_blueprint(peignes_blueprint, url_prefix='/api/peignes')
app.register_blueprint(config_blueprint, url_prefix='/api/config')

# Acced Blueprint
app.register_blueprint(acced_production_blueprint, url_prefix='/api/acced/prod')
app.register_blueprint(locations_blueprint, url_prefix='/api/emplacements')
app.register_blueprint(acced_output_blueprint, url_prefix='/api/acced/output')
app.register_blueprint(acced_reappro_blueprint, url_prefix='/api/acced/reappro')
app.register_blueprint(acced_consumption_blueprint, url_prefix='/api/acced/consumption')
app.register_blueprint(acced_batch_blueprint, url_prefix='/api/acced/batch')
app.register_blueprint(acced_message_blueprint, url_prefix='/api/acced/message')
app.register_blueprint(acced_setting_message_blueprint, url_prefix='/api/acced/settings/messages')
app.register_blueprint(acced_blueprint, url_prefix='/api/acced')
app.register_blueprint(acced_without_threshold_blueprint, url_prefix='/api/acced/cwothreshold')
app.register_blueprint(acced_replenish_blueprint, url_prefix='/api/listes_reappro')
app.register_blueprint(trends_blueprint, url_prefix='/api/trends')
app.register_blueprint(acced_saved_blueprint, url_prefix='/api/acced/savedlist')
app.register_blueprint(block_causes_blueprint, url_prefix='/api/acced/block')


# Blueprint for ASTUS
app.register_blueprint(astus_blueprint, url_prefix='/api/astus')
app.register_blueprint(astus_ward_blueprint, url_prefix='/api/astus/param/ward')
app.register_blueprint(astus_logs_blueprint, url_prefix='/api/astus/logs')
app.register_blueprint(astus_stock_blueprint, url_prefix='/api/astus/stock')
app.register_blueprint(astus_filled_blueprint, url_prefix='/api/astus/replenish')
app.register_blueprint(astus_drawer_blueprint, url_prefix='/api/astus/drawer')
app.register_blueprint(astus_nurse_stock_blueprint, url_prefix='/api/astus/nursestock')
app.register_blueprint(astus_output_blueprint, url_prefix='/api/astus/output')
app.register_blueprint(astus_consumption_blueprint, url_prefix='/api/astus/consumption')
app.register_blueprint(astus_stats_blueprint, url_prefix='/api/astus/stats')

# Blueprint for Riedl
app.register_blueprint(riedl_blueprint, url_prefix='/api/riedl')
app.register_blueprint(riedl_taking_mode_blueprint, url_prefix='/api/riedl/takingmode')
app.register_blueprint(riedl_output_blueprint, url_prefix='/api/riedl/output')
app.register_blueprint(riedl_output_list_blueprint, url_prefix='/api/riedl/output/list')
app.register_blueprint(riedl_input_blueprint, url_prefix='/api/riedl/input')
app.register_blueprint(riedl_stock_blueprint, url_prefix='/api/riedl/stock')

# Blueprint for external
app.register_blueprint(external_blueprint, url_prefix='/api/external')
app.register_blueprint(external_stock_blueprint, url_prefix='/api/external/stock')
app.register_blueprint(external_replenish_blueprint, url_prefix='/api/external/replenish')

# Blueprint for store
app.register_blueprint(store_blueprint, url_prefix='/api/magasin')
app.register_blueprint(stores_blueprint, url_prefix='/api/magasins')
app.secret_key = s_key
csrf = CSRFProtect(app)

register_commands(app)

# scheduler.init_app(app)
# scheduler.start()


@app.before_request
def before_request():
    """Use for reverse proxy with https enabled"""
    if request.url.endswith('ws/riedl'):
        return

    try:
        if request.url.endswith('/') or '/assets/' in request.url:
            pass
        elif (not request.url.endswith('crsftoken') and
              not request.url.endswith('login') and
              not request.url.endswith('publickey') and
              not request.url.endswith('refresh-token') and
              not request.url.endswith('manifest.json')):
            session['username']
    except Exception:
        return {'message': 'connection.failed'}, HTTP_401_UNAUTHORIZED

    scheme = request.headers.get('X-Forwarded-Proto')
    if scheme and scheme == 'http' and request.url.startswith('http://'):
        url = request.url.replace('http://', 'https://', 1)
        code = HTTP_301_MOVED_PERMANENTLY
        return redirect(url, code=code)


@sock.route('/ws/riedl')
def echo(ws):
    try:
        old_rields = get_riedls()
        while 1:
            new_rields = get_riedls()
            changes = []

            for new in new_rields:
                old = list(filter(lambda x: (x['pk'] == new['pk']), old_rields))
                obj = {}
                if len(old) == 1:
                    obj = get_diff(new, old[0])

                if len(obj.keys()) > 0:
                    changes.append(obj)

                if len(changes) > 0:
                    ws.send(data=json.dumps({'data': changes}))
                    old_rields = new_rields

            time.sleep(30)
    except Exception:
        pass
    return ''


@app.errorhandler(404)
def page_not_found(e):
    """
    When we use F5 to refresh browser, it return 404
    we need to got to the root application
    """
    return redirect('/')


@app.route('/')
def home():
    """Redirect to the index.html page"""
    return send_from_directory(html_dir, 'index.html')
    # return redirect('/index.html')


@app.route('/home')
def homepage():
    """Launch the homepage,
    TODO: Can be defined per user
    """
    if not session.get('username', False):
        return redirect('/login')
    return redirect('/reference.html')


@app.errorhandler(CSRFError)
def handle_csrf_error(e):
    return abort(HTTP_400_BAD_REQUEST, 'Bad request')


@app.route('/crsftoken', methods=['GET'])
def inject_csrf_token():
    resp = flask.Response()
    resp.headers['X-CSRF-Token'] = generate_csrf()
    return resp


@app.route('/inituserpassword', methods=['POST'])
def inituserpassword():
    if request.method == 'POST':
        user_id = request.form['userid']
        password = "D33nov@" + datetime.now().strftime("%H%M%S")

        try:
            usr = User.select().where((User.isEnabled == 1) & (User.pk == user_id)).get()

            User.update(
                passwordWeb=bcrypt.hashpw(password.encode('UTF_8'), bcrypt.gensalt()),
                isTemporary=1,
                retry=5
            ).where(User.pk == usr.pk).execute()

            return {'password': password}, HTTP_200_OK

        except Exception:
            return {'message': 'connection.failed'}, HTTP_400_BAD_REQUEST


@app.route('/renewpassword', methods=['POST'])
def renewpassword():
    if request.method == 'POST':

        password = request.form['password']
        confirm = request.form['password_confirmation']

        if password == confirm:
            user_id = session['user_id']
            usr = User.select().where((User.isEnabled == 1) & (User.pk == user_id)).get()
            User.update(
                passwordWeb=bcrypt.hashpw(password.encode('UTF_8'), bcrypt.gensalt()),
                isTemporary=0,
                retry=5
            ).where(User.pk == usr.pk).execute()

            return {'route': '/reference.html'}, HTTP_202_ACCEPTED
        else:
            return {'message': 'renewpassword.notequal'}, HTTP_400_BAD_REQUEST


@app.route("/publickey", methods=["POST"])
def public_key_generate():
    (publicKey, privateKey) = rsa.newkeys(1024)
    session['privateKey'] = privateKey.save_pkcs1('PEM')
    return {
        'key': publicKey.save_pkcs1('PEM').decode('utf8'),
    }, HTTP_200_OK


@app.route("/refresh-token", methods=["POST"])
@jwt_required(refresh=True)
def refresh():
    identity = get_jwt_identity()

    usr = User.get(User.pk == identity)

    access_token = create_access_token(
        identity=identity,
        additional_claims={
            'username': usr.username,
            'email': usr.email,
            'user_lang': usr.lang or 'en_US'
        }
    )
    refresh_token = create_refresh_token(identity=identity)

    session['username'] = usr.username
    session['email'] = usr.email
    session['user_id'] = usr.pk
    session['user_lang'] = usr.lang or 'en_US'
    session.permanent = True
    app.permanent_session_lifetime = timedelta(minutes=timeoutSession)

    return {
        'refreshToken': refresh_token,
        'token': access_token
    }, HTTP_200_OK


@app.route('/login', methods=['POST', 'GET'])
def login():
    """Send user to login form"""
    if request.method == 'POST':

        try:
            user = decryptRSA(request.form['username'], session['privateKey'])
            logger.info("User: %s" % user)
            usr = User.select().where(
                ((User.username == user) | (User.email == user) | (User.login == user))).get()
            logger.debug("User %s line found %i" % (usr.username, usr.pk))
        except Exception as e:
            logger.error(str(e.args))
            return {'message': 'connection.notexist'}, HTTP_401_UNAUTHORIZED

        user_id = usr.pk

        if not usr.isEnabled:
            logger.warning(f"User: {usr.username} ({usr.pk}) is disable")
            return {'message': 'connection.disabled'}, HTTP_401_UNAUTHORIZED

        if usr.retry <= 0:
            logger.warning(f"User: {usr.username} ({usr.pk}) Maximum de tentatives atteintes")
            return {'message': 'connection.max_retry'}, HTTP_401_UNAUTHORIZED

        if (user_id is not None) & (usr.passwordWeb is not None):
            password = decryptRSA(request.form['password'], session['privateKey'])

            if bcrypt.checkpw(password.encode('UTF_8'), usr.passwordWeb.encode('UTF_8')):
                logger.info("Check password success!")
                User.update(retry=5, date_last_login=datetime.now()).where(User.pk == user_id).execute()
                # Register the login, in the history database
                usrHistory = HistoriqueIdentification()
                usrHistory.user = usr.username
                usrHistory.demande = 'MWEB: Login'
                usrHistory.save()

                refresh_token = create_refresh_token(identity=user_id)
                access_token = create_access_token(
                    identity=user_id,
                    additional_claims={
                        'username': usr.username,
                        'email': usr.email,
                        'user_lang': usr.lang or 'en_US'
                    }
                )
                session['username'] = usr.username
                session['email'] = usr.email
                session['user_id'] = user_id
                session['user_lang'] = usr.lang or 'en_US'
                session.permanent = True
                app.permanent_session_lifetime = timedelta(minutes=timeoutSession)

                if usr.isTemporary:
                    logger.info("Password is temporary, ask for changes")
                    return {
                        'refreshToken': refresh_token,
                        'token': access_token,
                        'route': '/renewpassword.html'
                    }, HTTP_202_ACCEPTED
                else:
                    logger.info("User: Authentification is OK")
                    return {
                        'refreshToken': refresh_token,
                        'token': access_token,
                        'route': '/reference.html'
                    }, HTTP_202_ACCEPTED
            else:
                logger.info("Password mismatch from form and database")
                retry = usr.retry - 1
                User.update(retry=retry).where(User.pk == user_id).execute()
                return {'message': 'connection.failed'}, HTTP_401_UNAUTHORIZED
        else:
            logger.warning("User: Failed to retrieve user ou passwordweb in database")
            return {'message': 'connection.failed'}, HTTP_401_UNAUTHORIZED

    return redirect('/login.html')


@app.route('/logout')
def logout():
    """On deconnecte l'utilisateur et nettoie sa session"""
    try:
        username = session['username']
    except Exception:
        username = '-'

    session.clear()
    usrHistory = HistoriqueIdentification()
    usrHistory.user = username
    usrHistory.demande = 'MWEB: Logout'
    usrHistory.save()
    logger.info('User: logout [%s]' % username)
    return {}, HTTP_204_NO_CONTENT


@app.route('/manifest.json', methods=['GET'])
def manifest():
    icons_size = [(48, 48), (72, 72), (96, 96), (144, 144), (168, 168), (192, 192)]
    icons = []
    for i in icons_size:
        icons.append({
            "src": "/assets/logo_mweb-%ix%i.png" % (i[0], i[1]),
            "sizes": "%ix%i" % (i[0], i[1]),
            "type": "image/png"
        })

    res = {
        "name": "MedianWeb 2",
        "short_name": "MedianWeb",
        "start_url": "/?utm_source=web_app_manifest",
        "display": "standalone",
        "background_color": "#2e86c1",
        "icons": icons
    }
    return res, HTTP_200_OK


@app.route('/static/<path:chemin>', methods=['GET'])
def fichier_static(chemin):
    """On recupere et renvoi les fichiers statiques tel quel"""
    return send_from_directory(static_dir, chemin)


@app.route('/assets/<path:chemin>', methods=['GET'])
def fichier_assets(chemin):
    """On recupere et renvoi les fichiers statiques tel quel"""
    mimetype = 'text/plain'
    if request.url.endswith('.js'):
        mimetype = 'application/javascript'
    elif request.url.endswith('.css'):
        mimetype = 'text/css'
    elif request.url.endswith('.woff2'):
        mimetype = 'font/woff2'
    elif request.url.endswith('.png'):
        mimetype = 'image/png'
    elif request.url.endswith('.jpg'):
        mimetype = 'image/jpeg'
    elif request.url.endswith('.svg'):
        mimetype = 'image/svg+xml'
    return send_from_directory(static_dir, chemin, mimetype=mimetype)


@app.route('/<path:chemin>', methods=['GET'])
def fichier_html(chemin):
    """On recupere et renvoi les page HTML telle quelle"""
    mimetype = 'text/plain'
    if request.url.endswith('.js'):
        mimetype = 'application/javascript'
    elif request.url.endswith('.css'):
        mimetype = 'text/css'
    elif request.url.endswith('.woff2'):
        mimetype = 'font/woff2'
    elif request.url.endswith('.png'):
        mimetype = 'image/png'
    elif request.url.endswith('.jpg'):
        mimetype = 'image/jpeg'
    elif request.url.endswith('.svg'):
        mimetype = 'image/svg+xml'
    return send_from_directory(html_dir, chemin, mimetype=mimetype)


@app.route('/status', methods=['GET'])
@jwt_required()
def app_status():
    """retourne les informations utiles au debuggage"""
    if not session.get('username', False):
        return redirect('/login')
    dbinfo = {
        'name': os.environ.get('MEDIAN_DB_NAME', 'unknown'),
        'server': os.environ.get('MEDIAN_DB_HOST', 'unknown'),
        'user': os.environ.get('MEDIAN_DB_USER', 'unknown'),
        'port': os.environ.get('MEDIAN_DB_PORT', 'unknown'),
    }
    pyinfo = {
        'version': sys.version,
        'major': sys.version_info.major,
        'minor': sys.version_info.minor,
        'micro': sys.version_info.micro,
        'release': sys.version_info.releaselevel,
        'path': os.environ.get('PYTHONPATH', 'N/A'),
        'wsgi': os.environ.get('WSGI_HANDLER', 'N/A'),
    }
    version = {
        "Major": 0,
        "Minor": 0,
        "Patch": 0,
        "MajorMinorPatch": "0.0.0",
    }
    version_file = os.path.join(os.path.dirname(os.path.realpath(__file__)), 'version.json')
    logger.debug(version_file)
    if os.path.exists(version_file):
        with open(version_file, 'r') as f:
            logger.debug(f.read())
            f.seek(0)
            version = json.loads(f.read())
    else:
        logger.error("version file not found")

    return jsonify(database=dbinfo, python=pyinfo, version=version)


@app.route('/download/<string:filename>')
def download_file(filename):
    args = request.args
    val = args.get("attachment")
    attachment = (val is not None)
    cr_folder = os.environ.get('REPORTING_FOLDER', '.')
    fp = Path(cr_folder).joinpath(filename)
    if not fp.exists():
        logger.error('Download: file %s not found!' % filename)
        return {'message': 'File not found', 'filename': filename}, HTTP_404_NOT_FOUND
    return send_file(fp, as_attachment=attachment)


@app.cli.command("migrate")
@click.option('--force', is_flag=True, help="Delete web_tables, before recreate")
def migrate_app(force=False):
    """Update database catalog"""
    from common.database import (check_model, delete_model, sync_menu, median_upgrade, medianweb_upgrade,
                                 median_information)
    from common.models import (WebLang, WebMenu, WebMenuTranslation, WebForm, WebLogActions,
                               WebThresholdSheet, WebThresholdParameter, WebImportRefHeader,
                               WebImportRefLine)
    from common.migration import add_default_rights
    dbi = median_information()
    print('Database information\nHost: %(server)s\nPort: %(port)s\nName: %(name)s\nUser: %(user)s\n-----------' % dbi)
    if force:
        print("Force mode detected, delete tables before recreate\n-----------")
        delete_model('web_form_i18n', WebForm)
        delete_model('web_menu_i18n', WebMenuTranslation)
        delete_model('web_menu', WebMenu)
        delete_model('web_lang', WebLang)

    check_model('web_lang', WebLang, 'web_lang.json')
    check_model('web_menu', WebMenu, 'web_menu.json')
    sync_menu('web_menu', WebMenu, 'web_menu.json')
    check_model('web_menu_i18n', WebMenuTranslation)
    check_model('web_form_i18n', WebForm)
    check_model('web_log_actions', WebLogActions)
    check_model('web_threshold_sheet', WebThresholdSheet)
    check_model('web_threshold_parameter', WebThresholdParameter)
    check_model('web_importref_header', WebImportRefHeader)
    check_model('web_importref_line', WebImportRefLine)
    median_upgrade(force)
    medianweb_upgrade(force)
    sync_profil()
    add_default_rights()
    print('Migration done')


@app.cli.command("init")
def init_data():
    """Add initiate data"""
    from common.database import median_information
    from common.migration import add_default_libelle, add_default_rights, add_default_interfaces, add_default_counter
    print("Initilize datas")
    dbi = median_information()
    print('Database information\nHost: %(server)s\nPort: %(port)s\nName: %(name)s\nUser: %(user)s\n-----------' % dbi)
    add_default_libelle()
    add_default_rights()
    add_default_interfaces()
    add_default_counter()
    print("End initialize data")


@app.cli.command("fix")
def fix_datas():
    from common.fix import (
        fix_foreign_key, fix_datas, deenova_user_lang_default, deenova_user_rgpd,
        update_password_web, update_language, deenova_profil_default, update_catalog,
        generate_equipment_topography, fix_addresse_state_a, deenova_default_labels,
        fill_mag_format_table, deenova_default_format, update_loading_ids, update_contract_threshold
    )
    print("Recompute Foreign Key on Web_ table")
    fix_foreign_key()
    print("Fix table datas f_profil, f_mag, f_user")
    fix_datas()
    print("Set french lang on Deenova user")
    deenova_user_lang_default()
    print("Fix RGPD on deenova User")
    deenova_user_rgpd()
    print("Fix password web")
    update_password_web()
    print("Fix language available")
    update_language()
    print("Fix default rights for DEENOVA, ECO-DEX profil")
    deenova_profil_default()
    print("Catalog configuration")
    update_catalog()
    print("Generation equipment topography")
    generate_equipment_topography()
    print('Block address with state = A and bloque = 1')
    fix_addresse_state_a()
    print('Check if default codes are available in the database')
    deenova_default_labels()
    print('Check if container format are available in the database')
    deenova_default_format()
    print('Compute the format per warehouse')
    fill_mag_format_table()
    print("Fix min loading Id")
    update_loading_ids()
    print('Fix contract threshold')
    update_contract_threshold()


@app.cli.command('fix-patients')
def fix_patients():
    from common.fix import check_episodes_from_patients
    print('Starting fixing patients')
    check_episodes_from_patients()
    print('Fixing patients done')


@app.cli.command("riedl")
def riedl_management():
    from common.database import median_information
    from common.migration import riedl_migration
    dbi = median_information()
    print('Database information\nHost: %(server)s\nPort: %(port)s\nName: %(name)s\nUser: %(user)s\n-----------' % dbi)
    riedl_migration()
    print("End of Riedl migration")


@app.cli.command("translation")
@click.option('--resource', is_flag=True, help="Generate resource files from translation files")
@click.option('--json', is_flag=True, help="Generate translation files from resource files")
@click.option('--language',  help="Generate translation files from resource files")
def translationPo(resource=False, json=False, language=None):
    if resource:
        print("Generate resource files")
        generate_resources_files()
    if json:
        print("Generate translation files")
        generate_translations_files()
    if language is not None:
        generate_translations_lang(language)


@app.cli.command("demo")
@click.option('--force', is_flag=True, help="Force is mandatory to generate demo data")
@click.option('--clean', is_flag=True, help="Clean all riedl demo data")
@click.option('--riedl', is_flag=True, help="Generate RIEDL demo data")
@click.option('--acced', is_flag=True, help="Generate ACCED demo data")
@click.option('--astus', is_flag=True, help="Generate ASTUS demo data")
def demo_generator(force=False, clean=False, riedl=False, acced=False, astus=False):
    from common.demo import demo_riedl, demo_acced_globale, demo_acced_nominatif, \
        demo_riedl_storage, demo_riedl_taking_mode, demo_astus_nominatif
    if clean:
        demo_riedl(clean)
        demo_riedl_storage(clean)
        demo_riedl_taking_mode(clean)
        demo_acced_globale(clean)
        demo_acced_nominatif(clean)
        demo_astus_nominatif(clean)

    if force and riedl:
        print("Generate demo data for RIEDL")
        demo_riedl()
        demo_riedl_storage()
        demo_riedl_taking_mode()
    elif force and acced:
        print("Generate demo data for ACCED")
        demo_acced_globale()
        demo_acced_nominatif()
    elif force and astus:
        print("Generate demo data for ASTUS")
        demo_astus_nominatif()
    else:
        print("Nothing append, use force Luck")


# @scheduler.task('cron', id='notifications', second='*/10')
# def notifications():
#     notification_search()
