from median.models import (Magasin, Adresse, EquipmentUnitStock, EquipmentLines, EquipmentColumns, Stock, MagasinFormat,
                           Product, CodeBlocage, CatalogUnitStock, CatalogLines, CatalogColumns, CatalogVersion)
from median.constant import EquipmentType, PickingMachineType, CuttingMachineType
from peewee import JOIN, fn

from ressources.equipments.acced.acced_v3 import topo_acced_v3
from ressources.equipments.acced.acced_v2 import topo_acced_v2

from ressources.equipments.astus import topo_astus

ACCED_BP_FAMILY = [
    PickingMachineType.Acced_110.value,
    PickingMachineType.Acced_120.value,
    PickingMachineType.Acced_210.value,
    PickingMachineType.Acced_220.value,
    PickingMachineType.Acced_230.value,
    CuttingMachineType.Acced_320.value,
    PickingMachineType.Acced_440.value,
    PickingMachineType.Aide_V2.value,
]

AIDE_CUT_FAMILY = [
    CuttingMachineType.Aide_Cut.value,
    CuttingMachineType.Module_Coupe_500.value,
    CuttingMachineType.Module_Coupe_1000.value
]


def generate_equipment_topo(type_mag):
    equipment = Magasin.get(Magasin.type_mag == type_mag)
    res = _generate_topo(equipment)
    return res


def generator_topo_catalog(catalog_version_pk):
    version = CatalogVersion.get(CatalogVersion.pk == catalog_version_pk)

    detailCatalog = CatalogUnitStock.select(
        CatalogUnitStock.total_line.alias('total_line'),
        CatalogUnitStock.total_colonne.alias('total_colonne'),
        CatalogLines.start_line.alias('start_line'),
        CatalogLines.end_line.alias('end_line'),
        CatalogColumns.start_column.alias('start_column'),
        CatalogColumns.end_Column.alias('end_Column'),
        CatalogUnitStock.order.alias('order'),
        CatalogColumns.container_type.alias('container_type')
    ).switch(
        CatalogUnitStock
    ).join(
        CatalogLines, on=CatalogUnitStock.pk == CatalogLines.unit_stock_pk
    ).switch(
        CatalogUnitStock
    ).join(
        CatalogColumns, on=CatalogColumns.line_pk == CatalogLines.pk
    ).where(
        (CatalogUnitStock.version_pk == catalog_version_pk)
    ).order_by(CatalogUnitStock.order, CatalogLines.start_line, CatalogColumns.start_column)

    return _get_topo(detailCatalog=detailCatalog, version=version.numVersion, sub_version=version.subVersion)


def _generate_topo(equipment):
    detailCatalog = (EquipmentUnitStock.select(EquipmentUnitStock.total_line.alias('total_line'),
                                               EquipmentUnitStock.total_colonne.alias('total_colonne'),
                                               EquipmentLines.start_line.alias('start_line'),
                                               EquipmentLines.end_line.alias('end_line'),
                                               EquipmentColumns.start_column.alias('start_column'),
                                               EquipmentColumns.end_Column.alias('end_Column'),
                                               EquipmentUnitStock.order.alias('order'),
                                               EquipmentColumns.container_type.alias('container_type'))
                     .join(EquipmentLines, on=EquipmentUnitStock.pk == EquipmentLines.unit_stock_pk)
                     .switch(EquipmentUnitStock)
                     .join(EquipmentColumns, on=EquipmentColumns.line_pk == EquipmentLines.pk)
                     .switch(EquipmentUnitStock)
                     .join(Magasin, on=Magasin.pk == EquipmentUnitStock.equipment_pk)
                     .where((Magasin.pk == equipment.pk))
                     .order_by(EquipmentUnitStock.order, EquipmentLines.start_line, EquipmentColumns.start_column))

    adresses = list(Adresse.select(Adresse.pk, Adresse.etat,
                                   Adresse.bloque,
                                   Adresse.adresse,
                                   Adresse.contenant,
                                   Adresse.format,
                                   Product.designation,
                                   Stock.reference,
                                   fn.SUM(Stock.quantite).alias('qte_total'),
                                   Adresse.bloque_message.alias('custom_msg'),
                                   CodeBlocage.libelle.alias('msg')
                                   )
                    .join(Stock, JOIN.LEFT_OUTER, on=(Adresse.adresse == Stock.adresse) &
                                                     (Adresse.magasin == Stock.magasin))
                    .switch(Adresse)
                    .join(Product, JOIN.LEFT_OUTER, on=(Product.reference == Stock.reference))
                    .switch(Adresse)
                    .join(CodeBlocage, JOIN.LEFT_OUTER, on=(CodeBlocage.valeur == Adresse.bloque) &
                                                           (CodeBlocage.valeur != 0))
                    .where(Adresse.magasin == equipment.mag)
                    .group_by(Adresse.pk, Adresse.etat,
                              Adresse.bloque, Adresse.adresse,
                              Adresse.format))
    return _get_topo(detailCatalog=detailCatalog, equipment=equipment, adresses=adresses)


def _get_topo(detailCatalog, equipment=None, adresses=[], version=None, sub_version=''):
    topo = []
    unit_adresses = []
    # _add_magasin_format(equipment)
    for catalog in detailCatalog.objects():

        catalog_unit_stock = catalog
        unit = next(filter(lambda u: u['value'] == catalog_unit_stock.order, topo), None)
        if unit is None:
            unit = {
                'value': catalog_unit_stock.order,
                'lines': []
            }

            if equipment is not None:
                drawer_adr = (f'{equipment.mag}.TIR.'
                              f'{"{: >3}".format(str(catalog_unit_stock.order + 1))}.')
                bac_adr = (f'{equipment.mag}.BAC.'
                           f'{"{: >3}".format(str(catalog_unit_stock.order + 1))}.')

                if equipment.type_machine == PickingMachineType.Aide_Pick.value:
                    idx = 1 if unit['value'] < 9 else 2
                    if unit['value'] < 9:
                        offset = catalog_unit_stock.total_colonne * catalog_unit_stock.order
                    else:
                        offset = catalog_unit_stock.total_colonne * (catalog_unit_stock.order - 9)

                    bp_adrs = [(f'{equipment.mag}.'
                                f'{"{: >3}".format("BP")}.'
                                f'{"{: >3}".format(str(idx))}.'
                                f'{"{: >3}".format(str(offset + i + 1))}.')
                               for i in range(catalog_unit_stock.total_colonne)]
                    b_adrs = [(f'{equipment.mag}.'
                               f'{"{: >3}".format("TIR")}.'
                               f'{"{: >3}".format(str(idx))}.'
                               f'{"{: >3}".format(str(offset + i + 1))}.')
                              for i in range(catalog_unit_stock.total_colonne)]
                    unit_adresses = list(a for a in adresses if a.adresse[:16] in bp_adrs or a.adresse[:16] in b_adrs)

                elif equipment.type_machine != PickingMachineType.Acced_Boite_Pass.value:
                    offset = catalog_unit_stock.total_colonne * catalog_unit_stock.order
                    bp_adrs = [(f'{equipment.mag}. BP.'
                                f'{"{: >3}".format(str(offset + 1 + i))}.')
                               for i in range(catalog_unit_stock.total_colonne)]
                    unit_adresses = list(a for a in adresses if a.adresse.startswith(drawer_adr) or
                                         a.adresse.startswith(bac_adr) or
                                         a.adresse[:12] in bp_adrs)
                else:
                    offset = (catalog_unit_stock.total_colonne *
                              catalog_unit_stock.total_line *
                              catalog_unit_stock.order)
                    bp_adrs = [(f'{equipment.mag}.'
                                f'{"{: >3}".format(str(offset + 1 + i))}.')
                               for i in range(catalog_unit_stock.total_colonne * catalog_unit_stock.total_line)]
                    unit_adresses = list(a for a in adresses if a.adresse.startswith(drawer_adr) or
                                         a.adresse.startswith(bac_adr) or
                                         a.adresse[:8] in bp_adrs)

            topo.append(unit)

        for line in range(catalog.start_line, catalog.end_line + 1):

            line_unit = next(filter(lambda u: u['value'] == line, unit['lines']), None)
            if line_unit is None:
                line_unit = {
                    'value': line,
                    'total_col': catalog.total_colonne,
                    'type': '',
                    'columns': []}
                unit['lines'].append(line_unit)

            for col in range(catalog.start_column, catalog.end_Column + 1):
                col_adresses = get_adresses(unit_stock=catalog_unit_stock,
                                            version=equipment.type_machine if equipment is not None else version,
                                            column=catalog, col=col, line=line, sub_version=sub_version)

                line_unit['type'] = catalog.container_type
                column_unit = {
                    'value': col,
                    'label': '',
                    'type': '',
                    'depths': []}

                for col_adr in col_adresses:
                    adr = None
                    if equipment is not None:
                        if col_adr.startswith('TIR'):
                            drawer_adr = f'{equipment.mag}.{col_adr}'
                            bac_adr = f'{equipment.mag}.BAC.{col_adr[4:]}'
                            adr = next((a for a in unit_adresses if a.adresse == drawer_adr or a.adresse == bac_adr),
                                       None)

                        else:
                            adr = next((x for x in unit_adresses if x.adresse == f'{equipment.mag}.{col_adr}'), None)

                    if adr is not None:

                        column_depth = {
                            'pk': adr.pk,
                            'adress': col_adr,
                            'state': adr.etat,
                            'format': adr.format,
                            'locked': adr.bloque,
                            "qte": adr.qte_total,
                            "container": adr.contenant,
                            "reference": adr.stock.reference if hasattr(adr, 'stock') else '',
                            "designation": adr.product.designation if hasattr(adr, 'product') else '',
                            'lock_msg': adr.codeblocage.msg if hasattr(adr, 'codeblocage') else '',
                            'lock_custom_msg': adr.custom_msg if hasattr(adr, 'adresse') else '',
                        }
                    else:
                        column_depth = {
                            'pk': None,
                            'adress': col_adr,
                            'format': catalog.container_type,
                            'state': 'U',
                            'locked': 0
                        }

                    column_unit['label'] = column_depth['adress'][0:len(column_depth['adress']) - 4]
                    column_unit['type'] = column_depth['format']
                    column_unit['depths'].append(column_depth)

                line_unit['columns'].append(column_unit)

    return topo


def _add_magasin_format(equipement: Magasin):
    """Associate format to an equipement"""
    if equipement.type_machine in ACCED_BP_FAMILY:
        MagasinFormat.get_or_create(magasin=equipement.mag, format='BOITE PASS')
        MagasinFormat.get_or_create(magasin=equipement.mag, format='BAC')
    elif equipement.type_machine == CuttingMachineType.Aide_V3.value:
        MagasinFormat.get_or_create(magasin=equipement.mag, format='BOITE PASS')
        MagasinFormat.get_or_create(magasin=equipement.mag, format='TIROIR 1/3')
        MagasinFormat.get_or_create(magasin=equipement.mag, format='TIROIR 1/4')
        MagasinFormat.get_or_create(magasin=equipement.mag, format='TIROIR 1/5')
    elif equipement.type_machine == [
        CuttingMachineType.Aide_Cut.value,
        CuttingMachineType.Module_Coupe_500.value,
        CuttingMachineType.Module_Coupe_1000.value
    ]:
        MagasinFormat.get_or_create(magasin=equipement.mag, format='BOITE PASS')


def get_adresses(unit_stock, column, version, line, col, sub_version=''):
    adresses = []
    if version == PickingMachineType.Acced_Dose_Unit.value:
        adresses = topo_acced_v3(unit_stock=unit_stock, column=column,
                                 col=col, line=line)
    elif version == PickingMachineType.Acced_Boite_Pass.value:
        adresses = topo_acced_v2(unit_stock=unit_stock,
                                 col=col, line=line)
    elif version.startswith(EquipmentType.ASTUS.value):
        adresses = topo_astus(unit_stock=unit_stock, column=column,
                              col=col, line=line, sub_version=sub_version)
    elif version == CuttingMachineType.Aide_Cut.value or version == PickingMachineType.Aide_V2.value:
        adresses = topo_aide_cut_v1(col=col, line=line)
    elif version == PickingMachineType.Aide_Pick.value:
        adresses = topo_aide_pick_v1(unit_stock=unit_stock, column=column,
                                     col=col, line=line)
    return adresses


def topo_aide_cut_v1(line, col):
    return [
        f'{"{: >3}".format("BP")}.'
        f'{"{: >3}".format(str(col + 1))}.'
        f'{"{: >3}".format(str(line + 1))}']


def topo_aide_pick_v1(unit_stock, column, line, col):
    adresses = []
    if column.container_type == 'BOITE PASS' or column.container_type == 'BAC':
        prefix = 'BP' if column.container_type == 'BOITE PASS' else 'TIR'
        if unit_stock.order < 9:
            idx = 1
            offset = unit_stock.total_colonne * unit_stock.order
        else:
            idx = 2
            offset = unit_stock.total_colonne * (unit_stock.order - 9)

        adresses.append(
            f'{"{: >3}".format(prefix)}.'
            f'{"{: >3}".format(str(idx))}.'
            f'{"{: >3}".format(str(offset + col + 1))}.'
            f'{"{: >3}".format(str(line + 1))}')

    return adresses
