import json
import operator
import os
import uuid
import logging
from functools import reduce

from datetime import datetime, timedelta
from flask import Blueprint, request, send_file, session
from flask_jwt_extended import jwt_required
from median.constant import TypeServiListe
from common.status import HTTP_200_OK, HTTP_500_INTERNAL_SERVER_ERROR, HTTP_201_CREATED, HTTP_409_CONFLICT
from common.models import WebLogActions
from median.models import (Magasin, LotRetire, Ucd, Stock, Product, Dispensation, DispensationItem,
                           Historique, Patient, Service)
from median.views import RawConfig
from peewee import fn, JOIN

from ressources.astus.utils import generate_excel_file

withdrawnbatch_blueprint = Blueprint('withdrawnbatch', __name__)

logger = logging.getLogger('median.webserver')


def get(ref, args):
    v_search_list = args['search']

    andexpr = (ref is None) | (Product.pk == ref)

    if len(v_search_list) > 0:

        lst = list(map(lambda s: (
            ((ref is None) & (Product.reference.contains(s.strip()))) |
            ((ref is None) & (Product.designation.contains(s.strip()))) |
            (LotRetire.lot.contains(s.strip())) |
            (LotRetire.ucd.contains(s.strip()))
        ), v_search_list))
        search = reduce(operator.and_, lst)

        expr = reduce(operator.and_, [andexpr, search])
    else:
        expr = andexpr

    filtered_lots_retires_query = (
        LotRetire.select(LotRetire.pk, LotRetire.ucd, LotRetire.lot,
                         Product.reference, Product.designation,
                         fn.SUM(Stock.quantite).alias('qty'))
        .join(Ucd, on=(Ucd.ucd == LotRetire.ucd))
        .switch(LotRetire)
        .join(Product, on=(Product.reference == Ucd.reference))
        .switch(LotRetire)
        .join(Stock, JOIN.LEFT_OUTER, on=(Stock.ucd == LotRetire.ucd) &
                                         (Stock.lot == LotRetire.lot) &
                                         (Ucd.reference == Product.reference))
        .where(expr)
        .group_by(LotRetire.pk, LotRetire.ucd, LotRetire.lot,
                  Product.reference, Product.designation))
    return filtered_lots_retires_query


def get_obj_batch(batch):
    return {
        'pk': batch.pk,
        'productReference': batch.product.reference,
        'productLabel': batch.product.designation,
        'ucd': batch.ucd,
        'lot': batch.lot,
        'quantite': batch.qty
    }


def get_obj_batch_excel(batch):
    return {
        'productReference': batch.product.reference,
        'productLabel': batch.product.designation,
        'ucd': batch.ucd,
        'lot': batch.lot,
        'quantite': batch.qty
    }


def get_obj_historic(i):
    return {
        'pk': i.pk,
        'chrono': str(i.chrono),
        'type_mouvement': i.type_mouvement,
        'adresse': i.adresse,
        'adresse_from': i.adresse_from,
        'quantite_mouvement': i.quantite_mouvement,
        'quantite_prescrite': (i.quantite_demande if (i.type_mouvement == 'CPE') or (i.type_mouvement == 'ENT')
                               else i.quantite_prescrite),
        'quantite_totale': i.quantite_totale,
        'info': i.info,
        'liste': i.liste,
        'service': i.service,
        'patient': (i.patient.nom + " " + i.patient.prenom if hasattr(i, 'patient') else ''),
        'pilulier': i.id_pilulier,
        'utilisateur': i.utilisateur,
        'poste': i.poste,
        'date_peremption': str(i.date_peremption or ''),
        'serial': i.serial,
        'contenant': i.contenant
    }


def _get_batch_history_request(batch_pk, data):
    _criterias = data['criterias']

    fullname = Patient.prenom + ' ' + Patient.nom
    end = datetime.now().date()
    end = end + timedelta(days=1)
    start = end - timedelta(days=15)

    andexpr = (
        (LotRetire.pk == batch_pk) & (Historique.chrono < end) & (Historique.chrono > start)
    )

    if len(_criterias) > 0:

        lst = list(map(lambda s: (
                Historique.info.contains(s["value"].strip()) |
                Historique.utilisateur.contains(s["value"].strip()) |
                Historique.liste.contains(s["value"].strip()) |
                Historique.adresse.contains(s["value"].strip()) |
                fullname.contains(s["value"].strip()) |
                Historique.serial.contains(s["value"].strip()) |
                Magasin.libelle.contains(s["value"].strip()) |
                Historique.poste.contains(s["value"].strip()) |
                Service.libelle.contains(s["value"].strip()) |
                Historique.type_mouvement << s['mvts'])
            ), _criterias)

        search = reduce(operator.and_, lst)

        expr = reduce(operator.and_, [andexpr, search])
    else:
        expr = andexpr

    return (
        Historique.select(
            Historique.pk, Historique.chrono,
            Historique.type_mouvement, Historique.adresse,
            Historique.adresse_from, Historique.quantite_mouvement,
            Historique.quantite_prescrite, Historique.quantite_totale,
            Historique.info, Historique.quantite_demande,
            Historique.liste, Historique.service, Historique.lot,
            Historique.utilisateur, Historique.poste,
            Historique.date_peremption, Historique.serial,
            Historique.contenant, Patient.nom, Patient.prenom,
            fullname.alias('NomPatient'),
            Historique.id_pilulier, Magasin
        ).join(
            LotRetire, on=(Historique.lot == LotRetire.lot) & (Historique.ucd == LotRetire.ucd)
        ).join(
            Patient, JOIN.LEFT_OUTER, on=(Patient.ipp == Historique.ipp)
        ).switch(Historique).join(
            Magasin, JOIN.LEFT_OUTER, on=(Magasin.type_mag == Historique.poste)
        ).switch(Historique).join(
            Service, JOIN.LEFT_OUTER, on=(Service.code == Historique.service)
        ).where(expr).order_by(
            Historique.chrono.desc()
        )
    )


@withdrawnbatch_blueprint.route('<string:batch_pk>/history/export', methods=['POST'])
@jwt_required()
def get_batch_history_export(batch_pk):
    data = json.loads(request.data)
    headers = data['translations']
    req = _get_batch_history_request(data=data, batch_pk=batch_pk)

    name = os.sep.join(
        [os.getcwd(), "tmp_export", "%s.xlsx" % uuid.uuid4()])
    generate_excel_file(name=name, headers=headers, items=req, transform_function=get_obj_historic)

    return send_file(name, as_attachment=True)


@withdrawnbatch_blueprint.route('<string:batch_pk>/history', methods=['POST'])
@jwt_required()
def get_batch_history(batch_pk):
    data = json.loads(request.data)
    req = _get_batch_history_request(data=data, batch_pk=batch_pk)

    return {'list': [
        get_obj_historic(item) for item in req
    ]}, HTTP_200_OK


@withdrawnbatch_blueprint.route('<string:ref_id>/outputlist', methods=['POST'])
@jwt_required()
def generate_output_list(ref_id):
    data = json.loads(request.data)
    logger.info("Création d'une liste de sortie de lots retirés...")

    _ucd = data['ucd']
    _lot = data['lot']

    try:
        # récupérer les produits en stock ayant ce numéro de lot
        refs_in_stock = (
            Stock.select(
                Magasin.mag,
                Magasin.type_mag,
                Magasin.eco_type,
                Stock.reference,
                Stock.fraction,
                fn.SUM(Stock.quantite).alias('quantite'))
            .join(Magasin, on=(Stock.magasin == Magasin.mag))
            .where(
                ((Stock.reference == ref_id) | (Stock.ucd == _ucd)) &
                (Stock.lot == _lot))
            .group_by(Stock.magasin, Stock.reference))

        if refs_in_stock.count() == 0:
            return {'message': "withdraw.batch.del.error.empty"}, 503

        # vérifier que le lot en question est bien un lot à retirer
        lo = (
            LotRetire.select()
            .where((LotRetire.ucd == _ucd) & (LotRetire.lot == _lot))
        )
        if lo.count() == 0:
            return {'message': "withdraw.batch.del.error.marked"}, 503

        # chercher l'id dans f_dest dont le libellé = "SORTIE DE LOTS"
        # peut faire mieux, dans la version finale
        try:
            service = RawConfig().read(param='k_ua_pui').value
        except Exception as error:
            logger.error(error.args)
            return {'message': "k_ua_pui n'est pas défini."}, 503

        # Liste de sortie créé uniquement pour les acceds et riedl
        liste_stock = list(filter(lambda s: (Magasin.eco_type == 'C') | (Magasin.eco_type == 'L'), refs_in_stock))

        for r in liste_stock:

            liste_label = ("{store}-{ward} LOT A SORTIR {batch}"
                           .format(ward=service,
                                   store=r.magasin.mag,
                                   batch=_lot))

            _existing_disp = (
                Dispensation.select()
                .where(Dispensation.liste == liste_label)
            )
            if _existing_disp.count() == 0:
                Dispensation.create(
                    fusion="SORTIE DE LOTS",
                    liste=liste_label,
                    mode="S",
                    service=service,
                    type_servi=TypeServiListe.GlobaleBoite.value,
                    id_servi=2,
                    nb_item=refs_in_stock.count(),
                    zone_deb='RIELD' if r.magasin.eco_type == 'L' else '',
                    zone_fin=r.magasin.type_mag if r.magasin.eco_type == 'L' else '',
                    sous_secteur=r.magasin.type_mag if r.magasin.eco_type == 'C' else '',
                    etat='V',
                    date_creation=datetime.now(),
                    num_ipp='GLOBAL',
                    num_sej='GLOBAL',
                )

                DispensationItem.create(
                    liste=liste_label,
                    item='000001',
                    mode="S",
                    reference=r.reference,
                    qte_dem=r.quantite,
                    fraction=r.fraction,
                    lot=_lot,
                    dest=service,
                    type_servi=TypeServiListe.GlobaleBoite.value,
                    id_servi=2,
                    etat='V'
                )
                logger.info('Création d\'une liste de sortie de lots à retirer... REUSSI')

        return 'Success'

    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, 503


@withdrawnbatch_blueprint.route('/all', methods=['POST'])
@jwt_required()
def get_all_batchs():
    data = json.loads(request.data)
    req = get(ref=None, args=data)
    logger.debug('Lines : %s.' % len(req))
    return {'data': [get_obj_batch(s) for s in req]
            }, 200


@withdrawnbatch_blueprint.route('export', methods=['POST'])
@jwt_required()
def export_all():
    data = json.loads(request.data)
    headers = data['translations']
    name = os.sep.join(
        [os.getcwd(), "tmp_export", "%s.xlsx" % uuid.uuid4()])
    lines = get(ref=None, args=data)
    generate_excel_file(name=name, headers=headers, items=lines, transform_function=get_obj_batch_excel)

    return send_file(name, as_attachment=True)


@withdrawnbatch_blueprint.route('<string:ref_id>/export', methods=['POST'])
@jwt_required()
def export_by_ref(ref_id):
    data = json.loads(request.data)
    headers = data['translations']
    name = os.sep.join(
        [os.getcwd(), "tmp_export", "%s.xlsx" % uuid.uuid4()])
    lines = get(ref=ref_id, args=data)
    generate_excel_file(name=name, headers=headers, items=lines, transform_function=get_obj_batch_excel)

    return send_file(name, as_attachment=True)


@withdrawnbatch_blueprint.route('<string:ref_id>', methods=['POST'])
@jwt_required()
def get_batchs_by_ref(ref_id):
    data = json.loads(request.data)
    req = get(ref=ref_id, args=data)
    logger.debug('Lines : %s.' % len(req))
    return {
        'data': [get_obj_batch(s) for s in req]
    }, 200


@withdrawnbatch_blueprint.route('<string:ref_id>/create', methods=['POST'])
@jwt_required()
def create(ref_id):
    args = json.loads(request.data)
    _ucd = args['ucd']
    _lot = args['lot']

    logger.info('Create a blacklist batch lot: "%s"' % (_lot))

    try:
        if LotRetire.select().where((LotRetire.ucd == _ucd) & (LotRetire.lot == _lot)).exists():
            return {"message": "withdrawnbatch.add.error.duplicate"}, HTTP_409_CONFLICT

        LotRetire.create(ucd=_ucd, lot=_lot)
        log_withdraw(
            session['username'], 'withdraw_add', 'Added withdrawn batch ucd=%s batch=%s' % (_ucd, _lot)
        )

    except Exception as error:
        logger.error(error.args)
        return {"message": "withdrawnbatch.add.error.generic"}, HTTP_500_INTERNAL_SERVER_ERROR

    logger.info('Création réussie d\'une ligne Lot Retiré, lot: "%s"' % (_lot))
    return {'message': 'withdrawnbatch.add.success'}, HTTP_201_CREATED


@withdrawnbatch_blueprint.route('', methods=['DELETE'])
@jwt_required()
def delete():
    args = request.args
    _ucd = args['ucd']
    _lot = args['lot']

    logger.info('Suppression d\'une ligne Lot Retiré, lot: "%s"' % (_lot))

    try:
        LotRetire.delete().where((LotRetire.ucd == _ucd) & (LotRetire.lot == _lot)).execute()
        log_withdraw(
            session['username'], 'withdraw_remove', 'Deleted withdrawn batch ucd=%s batch=%s' % (_ucd, _lot)
        )

    except Exception as error:
        logger.error(error.args)
        return error.args, 503

    logger.info('Suppression réussie d\'une ligne Lot Retiré, lot: "%s"' % (_lot))
    return 'Success'


def log_withdraw(username: str, action: str, message: str):
    """
    Add new log for the withdrawn batches

    :param username: User made the action to log
    :param action:
    :param message: message to log
    """
    logger.info('WITHDRAW[%s](%s)): %s' % (action, username, message))
    wlog = WebLogActions()
    wlog.chrono = datetime.now()
    wlog.username = username
    wlog.equipement_type = ''
    wlog.action = action
    wlog.message = message
    wlog.save()
