import csv
import json
import logging
import operator
import os
import uuid
from datetime import datetime
from functools import reduce

from flask import Blueprint, request, send_file

from common.status import HTTP_200_OK, HTTP_400_BAD_REQUEST
from flask_jwt_extended import jwt_required
from median.constant import HistoryType
from median.models import Historique, Product, Service, Patient, Magasin
from median.models.log import DefDefaut

from ressources.astus.utils import generate_excel_file

traceability_blueprint = Blueprint('traceability', __name__)

logger = logging.getLogger('median')


@traceability_blueprint.route('importdefault', methods=['POST'])
@jwt_required()
def import_default():
    file = request.files['file']
    type = request.form['type']
    in_memory_file = file.read()
    text = in_memory_file.decode("ANSI")
    spamreader = csv.reader(text.splitlines(), delimiter=';', quotechar='|')
    (DefDefaut
     .delete()
     .where(DefDefaut.type_machine == type)
     .execute())

    line = 0
    for row in spamreader:
        if (line != 0) & (len(row) >= 7):
            station = row[0].zfill(2)
            index = row[1].zfill(2)

            obj = DefDefaut(
                num_defaut=station + index,
                niveau=row[6],
                defaut=row[4],
                type_machine=type)
            obj.save()

        line = line + 1

    return 'success'


@traceability_blueprint.route('count', methods=['POST'])
@jwt_required()
def getCount():
    data = json.loads(request.data)

    try:
        return {"count": request_traceability(data).count()}, HTTP_200_OK

    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_400_BAD_REQUEST


@traceability_blueprint.route('wards', methods=['POST'])
@jwt_required()
def get_wards():
    data = json.loads(request.data)
    v_search_list = data['criterias']
    v_date_debut = data['date_debut']
    v_date_fin = data['date_fin']
    products = data.get('products', [])

    expr = _get_where(start=v_date_debut, end=v_date_fin, criterias=v_search_list,
                      products=products)

    req = (Historique.select(
        Historique.service.alias('code'), Service.libelle.alias('label')
    ).distinct()
           .join(Product, on=Historique.reference == Product.reference)
           .join(Magasin, on=Historique.poste == Magasin.type_mag)
           .join(Service, on=Historique.service == Service.code)
           .join(Patient, on=(Patient.ipp == Historique.ipp))
           .where(expr)

           )

    try:
        return {"list": [
            {
                "code": item.code,
                "label": item.label,

            } for item in req.objects()
        ]}, HTTP_200_OK

    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_400_BAD_REQUEST


@traceability_blueprint.route('drugs', methods=['POST'])
@jwt_required()
def get_drugs():
    data = json.loads(request.data)
    v_search_list = data['criterias']
    v_date_debut = data['date_debut']
    v_date_fin = data['date_fin']
    wards = data.get('wards', [])
    expr = _get_where(start=v_date_debut, end=v_date_fin, criterias=v_search_list, wards=wards)

    req = (Historique.select(
        Historique.reference.alias('reference'),
        Product.designation.alias('designation'),
        Product.risque.alias('isRisky'),
        Product.stup.alias('isDrug')
    ).distinct()
           .join(Product, on=Historique.reference == Product.reference)
           .join(Magasin, on=Historique.poste == Magasin.type_mag)
           .join(Service, on=Historique.service == Service.code)
           .join(Patient, on=(Patient.ipp == Historique.ipp))
           .where(expr)

           )

    try:
        return {"list": [
            {
                "reference": item.reference,
                "designation": item.designation,
                "isRisky": item.isRisky,
                "isDrug": item.isDrug,
            } for item in req.objects()
        ]}, HTTP_200_OK

    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_400_BAD_REQUEST


def _get_where(start, end, criterias, products=[], wards=[]):
    expr = ((Product.risque | Product.stup) &
            (Historique.type_mouvement == HistoryType.Sortie.value) &
            (Historique.quantite_mouvement > 0) &
            (Historique.chrono >= start) &
            (Historique.chrono <= end))
    if len(criterias) > 0:
        fullname = Patient.prenom + ' ' + Patient.nom
        lst = list(map(lambda s: (
            (Product.designation.contains(s.strip())) |
            (Historique.reference.contains(s.strip())) |
            (Historique.utilisateur.contains(s.strip())) |
            (Magasin.libelle.contains(s.strip())) |
            (fullname.contains(s.strip())) |
            (Historique.poste.contains(s.strip())) |
            (Historique.service.contains(s.strip())) |
            (Service.libelle.contains(s.strip()))
        ), criterias))
        search = reduce(operator.and_, lst)
        expr = reduce(operator.and_, [expr, search])

    if any(products):
        expr = expr & (Historique.reference << products)

    if any(wards):
        expr = expr & (Historique.service << wards)

    return expr


@traceability_blueprint.route('export', methods=['PATCH'])
@jwt_required()
def export():
    data = json.loads(request.data)
    headers = data['translations']

    name = os.sep.join(
        [os.getcwd(), "tmp_export", "%s.xlsx" % uuid.uuid4()])
    req = request_traceability(data)
    generate_excel_file(name, headers, req.objects(), get_obj)

    return send_file(name, as_attachment=True)


def get_obj(item):
    return {
        'chrono': item.start.strftime('%Y-%m-%d %H:%M'),
        'reference': item.reference,
        'designation': item.designation,
        'quantite_mouvement': item.quantity,
        'serviceCode': item.wardCode,
        'service': item.wardLabel,
        'patient': f'{item.firstName} {item.lastName}',
        'poste': item.equipmentLabel,
        'utilisateur': item.user,

    }


def request_traceability(data):
    v_search_list = data['criterias']
    v_date_debut = data['date_debut']
    v_date_fin = data['date_fin']
    products = data.get('products', [])
    wards = data.get('wards', [])

    expr = _get_where(start=v_date_debut, end=v_date_fin, criterias=v_search_list,
                      products=products, wards=wards)

    return (Historique.select(
        Historique.type_mouvement.alias('type'),
        Historique.chrono.alias('start'),
        Historique.utilisateur.alias('user'),
        Historique.reference.alias('reference'),
        Product.designation.alias('designation'),
        Patient.nom.alias('lastName'),
        Patient.prenom.alias('firstName'),
        Historique.quantite_mouvement.alias('quantity'),
        Service.code.alias('wardCode'),
        Service.libelle.alias('wardLabel'),
        Magasin.libelle.alias('equipmentLabel'),
        Historique.poste.alias('equipmentCode'),
        Product.risque.alias('isRisky'),
        Product.stup.alias('isDrug')
    )
            .join(Product, on=Historique.reference == Product.reference)
            .join(Magasin, on=Historique.poste == Magasin.type_mag)
            .join(Service, on=Historique.service == Service.code)
            .join(Patient, on=(Patient.ipp == Historique.ipp))
            .where(expr)
            .order_by(Historique.chrono.desc()))


@traceability_blueprint.route('', methods=['POST'])
@jwt_required()
def getAll():
    data = json.loads(request.data)

    try:
        v_start = data['pagination']['page']
        v_length = data['pagination']['rowsPerPage']

        req = request_traceability(data) \
            .limit(v_length).offset(v_start * v_length)

        return {"list": [{
            "pk": item.pk,
            "movmentType": item.type,
            "start": datetime.utcfromtimestamp(datetime.timestamp(item.start)),
            "user": item.user,
            "quantity": item.quantity,
            "product": {
                "reference": item.reference,
                "designation": item.designation,
                "isRisky": item.isRisky,
                "isDrug": item.isDrug
            },
            "patient": {
                "firstName": item.firstName,
                "lastName": item.lastName,

            },
            "ward": {
                "code": item.wardCode,
                "label": item.wardLabel
            },
            "equipment": {
                "code": item.equipmentCode,
                "label": item.equipmentLabel
            }
        } for item in req.objects()]}, HTTP_200_OK

    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_400_BAD_REQUEST
