import json
import logging
from common.status import HTTP_200_OK, HTTP_400_BAD_REQUEST
from flask import Blueprint, request, session
from flask_jwt_extended import jwt_required
from peewee import DoesNotExist

from median.models import Magasin
from median.views import ReplenishmentView

replenishment_blueprint = Blueprint("replenishment", __name__)

logger = logging.getLogger("median")


@replenishment_blueprint.route("generate", methods=["POST"])
@jwt_required()
def generate():
    """Generate a replenishment"""
    try:
        res = json.loads(request.data)
        type_magasin = res["type_magasin"]
    except Exception as e:
        logger.error(str(e))
        return {"message": "arguments missing or corrupt", "error": str(e)}, HTTP_400_BAD_REQUEST

    logger.info(f"Generate replenishment for {type_magasin}")
    try:
        mag = Magasin.get(type_mag=type_magasin)
        rep = ReplenishmentView(mag.mag, session["username"])
        for line in res.get("lines", []):
            logger.info(line)
            rep.add_reference(line["ref"], int(line["qte"]), int(line["frac"]))
    except DoesNotExist as e:
        logger.error(str(e))
        return {"message": f'{type_magasin} does not exist'}, HTTP_400_BAD_REQUEST
    except Exception as e:
        logger.error(str(e))
        return {"message": "Error when generating replenishment", "error": str(e)}, HTTP_400_BAD_REQUEST

    return {
        "output_ext": [
            {
                "pk": rep.liste_out[lst].pk,
                "zone_deb": rep.liste_out[lst].zone_deb,
                "zone_fin": rep.liste_out[lst].zone_fin,
            }
            for lst in rep.liste_out.keys()
        ],
        "replenish_pick": {"pk": rep.liste and rep.liste.pk or None},
    }, HTTP_200_OK
