import logging

from datetime import datetime
from flask import Blueprint, jsonify, session
from flask_jwt_extended import jwt_required
from median.models import Profil
from median.constant import DEFAULT_PROFILS, CUSTOM_PROFILS
from common.util import mustHaveRights
from common.exception import WebPermissionException
from common.models import WebLogActions
from common.status import HTTP_404_NOT_FOUND, HTTP_403_FORBIDDEN, HTTP_204_NO_CONTENT, HTTP_500_INTERNAL_SERVER_ERROR

profils_blueprint = Blueprint('profils', __name__)

logger = logging.getLogger('median')

RESSOURCE_NAME = 'WEB_GROUP'


@profils_blueprint.errorhandler(WebPermissionException)
def handle_error(err):
    return jsonify(err.to_dict()), err.status_code


@profils_blueprint.route('', methods=['GET'])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canView=True)
def get_all():
    logger.info("Récupérer les profils utilisateurs")
    # TODO: Utiliser f_resrc qui est utilisé pour lister les droits, au lieu du distinct qui peut en oublier
    paged_profils = (Profil.select(Profil.profil).distinct())

    return ([{
        'profil': i.profil
    } for i in paged_profils])


@profils_blueprint.route('<string:profile_name>', methods=['DELETE'])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canEdit=True)
def delete_group(profile_name):
    profil = Profil.get_or_none(Profil.profil == profile_name)
    if profil:
        if profile_name not in DEFAULT_PROFILS + CUSTOM_PROFILS:
            query = Profil.delete().where(Profil.profil == profile_name)
            if query.execute() > 0:
                log_groups(session['username'], 'delete_group',
                           f'Deleted group (pk: {profil.pk} - name: {profil.profil})')
                return {}, HTTP_204_NO_CONTENT
            else:
                return {'message': 'group.delete.error.failed'}, HTTP_500_INTERNAL_SERVER_ERROR
        else:
            log_groups(session['username'], 'delete_group',
                       f'Tried to delete default group (pk: {profil.pk} - name: {profil.profil})')
            logger.error('Profile deletion canceled, group is default. name: ' + profile_name)
            return {'message': 'group.delete.error.protected'}, HTTP_403_FORBIDDEN
    else:
        logger.error('Profile deletion failed, none found with name: ' + profile_name)
        return {'message': 'group.delete.error.not_found'}, HTTP_404_NOT_FOUND


def log_groups(username: str, action: str, message: str):
    """
    Add new log for Groupe

    :param username: User made the action to log
    :param action:
    :param message: message to log
    """
    logger.info('Groups[%s](%s)): %s' % (action, username, message))
    wlog = WebLogActions()
    wlog.chrono = datetime.now()
    wlog.username = username
    wlog.equipement_type = ''
    wlog.action = action
    wlog.message = message
    wlog.save()
