import logging
import os
import base64
from io import BytesIO
from common.status import HTTP_204_NO_CONTENT, HTTP_400_BAD_REQUEST, HTTP_200_OK, HTTP_500_INTERNAL_SERVER_ERROR
from flask import Blueprint, request
from flask_jwt_extended import jwt_required
from median.models import Ucd, Product, Gtin
from median.constant import DpmType
from median.views import RawConfig
from PIL import Image
from peewee import DoesNotExist
from playhouse.shortcuts import model_to_dict

from ressources.blueprint.config.config_service import get_drug_standard, DrugStandard

gtin_cip_blueprint = Blueprint("gtin_cip", __name__)

logger = logging.getLogger("median")


@gtin_cip_blueprint.route("<string:ref_pk>", methods=["DELETE"])
@jwt_required()
def delete(ref_pk):
    args = request.args
    _ucd = args["ucd"]

    p = Product.select().where(Product.pk == ref_pk).get()
    ref = p.reference
    logger.info('Suppression d\'un UCD, ref: "%s", UCD: "%s"' % (ref, _ucd))

    try:
        Ucd.delete().where((Ucd.reference == ref) & (Ucd.ucd == _ucd)).execute()
        logger.info('Suppression réussie d\'un UCD, ref: "%s", UCD: "%s"' % (ref, _ucd))
        return "Success", HTTP_204_NO_CONTENT

    except Exception as error:
        logger.error(error.args)
        return error.args, HTTP_400_BAD_REQUEST


@gtin_cip_blueprint.route("<string:ref_pk>", methods=["POST"])
@jwt_required()
def post(ref_pk):
    logger.info("Add new UCD value, or equivalent on other country")

    args = request.json
    v_ucd = args["ucd"]

    drugs_standard = get_drug_standard()

    valid_lengths = {DrugStandard.Pzn: [7, 8], DrugStandard.Cnk: [7, 13]}.get(
        drugs_standard, [7, 13]
    )  # Default for UcdCip and others

    if len(v_ucd) not in valid_lengths:
        message = (
            f"threshold.{drugs_standard.value}.error.length"
            if drugs_standard in [DrugStandard.Pzn, DrugStandard.Cnk]
            else "threshold.ucdcip.error.length"
        )
        return {"message": message}, HTTP_400_BAD_REQUEST

    if not v_ucd.isdigit() and drugs_standard != DrugStandard.Cnk:
        return {"message": f"threshold.{drugs_standard.value}.error.digit"}, HTTP_400_BAD_REQUEST

    try:
        p = Product.select().where(Product.pk == ref_pk).get()
        ref = p.reference

        # regarder si on n'a pas déjà une réf associée à cet UCD
        existing_ucd_count = Ucd.select(Ucd.reference).where((Ucd.ucd == v_ucd))

        if existing_ucd_count.count() > 0:
            return {"message": f"threshold.{drugs_standard.value}.association.error.exist"}, HTTP_400_BAD_REQUEST

        Ucd.create(reference=ref, ucd=v_ucd)

        logger.info("Add new UCD with success")
        return "Success", HTTP_204_NO_CONTENT

    except Exception as error:
        logger.error(error.args)
        return {"message": error.args}, HTTP_400_BAD_REQUEST


@gtin_cip_blueprint.route("dpm/<string:dpm_type>/<string:gtin>", methods=["GET"])
@jwt_required()
def get_all_gtin_versions(dpm_type, gtin):
    # Check if type included in constant enum
    if dpm_type not in [e.value for e in DpmType]:
        return {"message": "Invalid DPM type"}, HTTP_400_BAD_REQUEST

    # NOTE: Using a placeholder GTIN to make the following work, waiting for the real files
    # gtin = "05099151924766"

    casted_gtin = str(int(gtin))

    for g in [gtin, casted_gtin]:
        query = (
            Gtin.select()
            .where(
                (
                    (Gtin.cip.contains(g))
                    # TODO: Implement the following line when the database has all the
                    # correct x_type_dpm values inf_ucd_cip
                    # & (Gtin.type_dpm == dpm_type)
                    & (Gtin.dossier != 0)
                )
            )
            .order_by(Gtin.dossier)
        )

        if query.count() > 0:
            break

    dpm_data = []

    gtin: Gtin
    for gtin in query:
        # Try to get the right folder path

        cfg = RawConfig("MEDIANWEB").read("k_eco_dir_fiche_track")
        destination_folder = os.path.join(cfg.value, gtin.cip, gtin.dossier)
        # Try to open the folder
        if destination_folder:
            if os.path.exists(destination_folder):
                new_data = {"gtin": gtin.cip, "index": gtin.dossier, "files": []}
                # Get a list of files of type JPG, PNG, JPEG
                image_files = [
                    f
                    for f in os.listdir(destination_folder)
                    if f.lower().endswith((".jpg", ".jpeg", ".png")) and f.lower().startswith(("image_1", "image_2"))
                ]

                for image_file in image_files:
                    image_path = os.path.join(destination_folder, image_file)

                    try:
                        with open(image_path, "rb") as f:
                            img_buffer = f.read()

                        # Convert image to WebP format
                        buf = BytesIO()
                        img_byte_io = BytesIO(img_buffer)
                        img = Image.open(img_byte_io).convert("RGB")

                        # Resize to web-friendly size (max 800x600 while maintaining aspect ratio)
                        img.thumbnail((800, 600), Image.Resampling.LANCZOS)
                        img.save(buf, format="webp", quality=85, optimize=True)
                        buf.seek(0)

                        # Convert to base64
                        img_base64 = base64.b64encode(buf.getvalue()).decode("utf-8")

                        # Add image data to files array
                        new_data["files"].append(
                            {
                                "filename": f"{os.path.splitext(image_file)[0]}.webp",
                                "data": img_base64,
                                "mimetype": "image/webp",
                            }
                        )

                    except (IOError, OSError) as e:
                        logger.error(f"Error processing image {image_file}: {str(e)}")
                        return {"message": "dpm.not_image.error"}, HTTP_500_INTERNAL_SERVER_ERROR

                # Get also the first file being an xml
                xml_file = next(
                    (f for f in os.listdir(destination_folder) if f.lower().endswith(".xml")),
                    None,
                )

                if xml_file:
                    xml_path = os.path.join(destination_folder, xml_file)
                    try:
                        import xml.etree.ElementTree as ET

                        tree = ET.parse(xml_path)
                        root = tree.getroot()
                        # Assuming the structure matches the provided XML
                        type_posage = (
                            root.find(".//Type_Posage").text if root.find(".//Type_Posage") is not None else None
                        )
                        nbr_case_bac = (
                            root.find(".//Nbr_Case_Bac").text if root.find(".//Nbr_Case_Bac") is not None else None
                        )
                        # Add to new_data if needed
                        new_data["type_posage"] = type_posage
                        new_data["nbr_case_bac"] = nbr_case_bac
                    except ET.ParseError as e:
                        logger.error(f"Error parsing XML {xml_file}: {str(e)}")
                        # Optionally handle error, e.g., skip or return error
                else:
                    logger.warning(f"No XML file found in {destination_folder}")
                    return {"message": "dpm.no_xml.error"}, HTTP_500_INTERNAL_SERVER_ERROR

                # Only add to dpm_data if we found and processed images
                if new_data["files"]:
                    dpm_data.append(new_data)

    return {"data": dpm_data}, HTTP_200_OK


@gtin_cip_blueprint.route("fields/<string:ucd>", methods=["GET"])
@jwt_required()
def get_fields(ucd):
    try:
        ucd_obj: Ucd = Ucd.get(Ucd.ucd == ucd)
        data = model_to_dict(ucd_obj)

        keys_to_remove = []
        for key, _ in data.items():
            if "alpha" not in key:
                continue

            try:
                ucd_label = _get_label_value("ucd_" + key)

                if ucd_label and ucd_label.readonly == 2:
                    # This is hidden, don't pass it in the return
                    keys_to_remove.append(key)

            except Exception:
                continue

        for key in keys_to_remove:
            data.pop(key, None)

        return data, HTTP_200_OK

    except DoesNotExist:
        logger.error('UCD "%s" not found' % ucd)
        return {"message": "UCD not found"}, HTTP_400_BAD_REQUEST
    except Exception as e:
        logger.error("Error getting UCD fields: %s" % str(e))
        return {"message": "An error has occured"}, HTTP_500_INTERNAL_SERVER_ERROR


@gtin_cip_blueprint.route("fields", methods=["GET"])
@jwt_required()
def use_fields():
    from median.constant import MEDIANWEB_POSTE, CONFIG_WEB_CLE

    ucd_fields_data = RawConfig(MEDIANWEB_POSTE, CONFIG_WEB_CLE).read(param="k_web_ucd_fields")

    can_use_fields = ucd_fields_data is not None and ucd_fields_data.value

    return {"can_use_fields": can_use_fields}, HTTP_200_OK


@gtin_cip_blueprint.route("fields/<string:ucd>", methods=["PUT"])
@jwt_required()
def update_fields(ucd):
    try:
        ucd_obj: Ucd = Ucd.get(Ucd.ucd == ucd)
        data = request.json

        # Update only alpha_ fields that exist on the model
        for key, value in data.items():
            label = _get_label_value("ucd_" + key)

            if label and label.readonly == 0 and key.startswith("alpha_") and hasattr(Ucd, key):
                setattr(ucd_obj, key, value)

        ucd_obj.save()
        logger.info('Updated UCD fields for UCD: "%s"' % ucd)
        return model_to_dict(ucd_obj), HTTP_200_OK

    except DoesNotExist:
        return {"message": "UCD not found"}, HTTP_400_BAD_REQUEST
    except Exception as e:
        logger.error("Error updating UCD fields: %s" % str(e))
        return {"message": "An error occurred while updating fields"}, HTTP_500_INTERNAL_SERVER_ERROR


def _get_label_value(key):
    from median.models import Label

    return Label.get_or_none(Label.code == key)
