import json
import logging
from datetime import datetime
from flask import Blueprint, request, session
from flask_jwt_extended import jwt_required

from median.models.log import LogScan
from common.status import HTTP_200_OK, HTTP_400_BAD_REQUEST

logging_blueprint = Blueprint('logging', __name__)
logger = logging.getLogger('median.logging')


@logging_blueprint.route('/scan', methods=['POST'])
@jwt_required()
def log_scan():
    try:
        data = json.loads(request.data)

        if data.get('value') is None:
            logger.error("Missing 'value' in request data")
            return {"message": "Missing required field: value"}, HTTP_400_BAD_REQUEST

        username = session['username']

        # Extract scan data
        scan_value = data.get('value')
        scan_data = data.get('data')
        scan_gtin = scan_data.get('gtin', None)
        scan_serial = scan_data.get('customerPartNumber') or scan_data.get('serial', None)
        scan_date = scan_data.get('expiry', None)

        # Try to parse the scan date as YYMMDD and convert for mysql
        try:
            if scan_date:
                scan_date = datetime.strptime(scan_date, '%y%m%d')
                scan_date = scan_date.strftime('%Y-%m-%d %H:%M:%S')
        except ValueError:
            scan_date = None

        # Convert to MySQL datetime format

        logScan: LogScan = LogScan()
        logScan.chrono = datetime.now()
        logScan.user = username
        logScan.value = scan_value
        logScan.scan_gtin = scan_gtin or ''
        logScan.scan_serial = scan_serial or ''
        if scan_date:
            logScan.scan_date = scan_date
        logScan.save()

        return {
            "message": "Scan logged successfully",
            "log_id": logScan.pk
        }, HTTP_200_OK

    except Exception as e:
        logger.error(f"Error logging scan: {str(e)}")
        return {"message": f"Failed to log scan: {str(e)}"}, HTTP_400_BAD_REQUEST
