import json
import logging

from datetime import datetime
from flask import Blueprint, request, session
from flask_jwt_extended import jwt_required
from median.models import Service, Product, ReferencePerService
from common.models import WebLogActions
from common.status import HTTP_200_OK, HTTP_500_INTERNAL_SERVER_ERROR, HTTP_409_CONFLICT, HTTP_400_BAD_REQUEST
from median.constant import ReferenceDistribution, ReferenceDistributionType
from peewee import JOIN, DoesNotExist

completion_blueprint = Blueprint("completion", __name__)

logger = logging.getLogger("median")


@completion_blueprint.route("", methods=["POST"])
@jwt_required()
def get_all():
    data = json.loads(request.data)
    limit = data.get("limit", 20)
    offset = data.get("offset", 0)
    descending = data.get("descending", False)
    page = data.get("page", 1)
    sortBy = data.get("sortBy", "pk")
    filter_data = data.get("filter", {})

    # Calculate actual offset based on page number
    actual_offset = offset + (page - 1) * limit

    # Start building the query
    ref_dest_query = (
        ReferencePerService.select(ReferencePerService, Service, Product)
        .join(Service, JOIN.INNER, on=(Service.pk == ReferencePerService.dest_pk))
        .switch(ReferencePerService)
        .join(Product, JOIN.INNER, on=(Product.pk == ReferencePerService.ref_pk))
    )

    # Apply filters
    if "service_name" in filter_data and filter_data["service_name"]:
        ref_dest_query = ref_dest_query.where(Service.libelle.contains(filter_data["service_name"]))

    if "reference" in filter_data and filter_data["reference"]:
        ref_dest_query = ref_dest_query.where(Product.reference.contains(filter_data["reference"]))

    if "reference_designation" in filter_data and filter_data["reference_designation"]:
        ref_dest_query = ref_dest_query.where(Product.designation.contains(filter_data["reference_designation"]))

    if "distribution" in filter_data and filter_data["distribution"]:
        ref_dest_query = ref_dest_query.where(ReferencePerService.distribution.contains(filter_data["distribution"]))

    if "distribution_type" in filter_data and filter_data["distribution_type"]:
        ref_dest_query = ref_dest_query.where(
            ReferencePerService.distribution_type.contains(filter_data["distribution_type"])
        )

    # Get total count before pagination
    total_count = ref_dest_query.count()

    # Apply sorting
    order_field = None
    if sortBy == "pk":
        order_field = ReferencePerService.pk
    elif sortBy == "ward":
        order_field = Service.libelle
    elif sortBy == "reference":
        order_field = Product.reference
    elif sortBy == "designation":
        order_field = Product.designation
    elif sortBy == "distribution":
        order_field = ReferencePerService.distribution
    elif sortBy == "distribution_type":
        order_field = ReferencePerService.distribution_type

    if order_field:
        ref_dest_query = ref_dest_query.order_by(order_field.desc() if descending else order_field.asc())

    # Apply pagination
    ref_dest_data = ref_dest_query.limit(limit).offset(actual_offset)

    return {
        "table": [
            {
                "pk": refdest.pk,
                "distribution": refdest.distribution,
                "distribution_type": refdest.distribution_type,
                "service_pk": refdest.dest_pk.pk,
                "service_name": refdest.dest_pk.libelle,
                "reference_pk": refdest.ref_pk.pk,
                "reference_designation": refdest.ref_pk.designation,
                "reference_ref": refdest.ref_pk.reference,
            }
            for refdest in ref_dest_data
        ],
        "total": total_count,
    }, HTTP_200_OK


@completion_blueprint.route("get_wards", methods=["GET"])
@jwt_required()
def get_wards():
    # For the ward configuration dialog
    wards = Service.select(Service.pk, Service.code, Service.libelle).order_by(+Service.libelle)

    return {"wards": [{"pk": ward.pk, "libelle": f"{ward.libelle} [{ward.code}]"} for ward in wards]}, HTTP_200_OK


@completion_blueprint.route("get_products", methods=["POST"])
@jwt_required()
def get_products():
    # For the ward configuration dialog
    data = json.loads(request.data)
    desig_filter = data.get("filter", "")

    query = Product.select(Product.pk, Product.reference, Product.designation)

    if desig_filter:
        filter_words = desig_filter.strip().split()
        for word in filter_words:
            query = query.where(Product.designation.contains(word) | (Product.reference.contains(word)))

    products = query.order_by(+Product.designation)

    return {
        "products": [
            {"pk": product.pk, "designation": f"{product.designation} [{product.reference}]"} for product in products
        ]
    }, HTTP_200_OK


@completion_blueprint.route("add_ward_config", methods=["PUT"])
@jwt_required()
def add_ward_config():
    data = json.loads(request.data)

    ward_pk = data.get("ward_pk", None)
    product_pk = data.get("product_pk", None)
    distribution = data.get("distribution", ReferenceDistribution.Unitaire.value)  # Default to 'UN' if not provided
    distribution_type = data.get(
        "distribution_type", ReferenceDistributionType.Global.value
    )  # Default to 'UN' if not provided

    if not ward_pk or not product_pk:
        return {"message": "Missing required parameters"}, HTTP_400_BAD_REQUEST

    # Check if the association already exists
    try:
        ReferencePerService.get(
            (ReferencePerService.dest_pk == ward_pk)
            & (ReferencePerService.ref_pk == product_pk)
            & (ReferencePerService.distribution_type == distribution_type)
        )
        return {
            "message": "This product is already associated with this ward "
            f"[{'Global' if distribution_type == 'G' else 'Nominative'}]"
        }, HTTP_409_CONFLICT
    except DoesNotExist:
        # Association doesn't exist, proceed with creation
        try:
            new_association: ReferencePerService = ReferencePerService()
            new_association.dest_pk = ward_pk
            new_association.ref_pk = product_pk
            new_association.distribution = distribution
            new_association.distribution_type = distribution_type
            new_association.save()

            username = session.get("username", "Unknown")
            ward = Service.get_by_id(ward_pk)
            product = Product.get_by_id(product_pk)
            log_message = (
                f"Associated product '{product.reference}' with ward '{ward.libelle}' "
                f"using distribution type '{distribution}'"
            )
            log_completion(username, "ADD_WARD_CONFIG", log_message)

            return {"message": "Association created successfully", "pk": new_association.pk}, HTTP_200_OK
        except Exception as e:
            return {"message": f"Error creating association: {str(e)}"}, HTTP_500_INTERNAL_SERVER_ERROR


@completion_blueprint.route("delete_ward_configs", methods=["post"])
@jwt_required()
def delete_ward_configs():
    data = request.get_json()
    pks = data.get("pks", [])

    delete_query = ReferencePerService.delete().where(ReferencePerService.pk << pks)
    deleted_count = delete_query.execute()

    return {"deleted_count": deleted_count}, HTTP_200_OK


def log_completion(username: str, action: str, message: str):
    """
    Add new log for completion

    :param username: User made the action to log
    :param action:
    :param message: message to log
    """
    logger.info("Completion[%s](%s)): %s" % (action, username, message))
    wlog = WebLogActions()
    wlog.chrono = datetime.now()
    wlog.username = username
    wlog.equipement_type = ""
    wlog.action = action
    wlog.message = message
    wlog.save()
