import json
import logging
import operator
import os
import uuid
from functools import reduce

from common.status import HTTP_200_OK, HTTP_400_BAD_REQUEST, HTTP_204_NO_CONTENT
from flask import Blueprint, request, send_file
from flask_jwt_extended import jwt_required
from median.models import CodeBlocage
from peewee import DoesNotExist
from ressources.astus.utils import generate_excel_file

blocking_defects_blueprint = Blueprint('blocking_defects', __name__)

logger = logging.getLogger('median')


def get_all():
    return (CodeBlocage
            .select(CodeBlocage.pk, CodeBlocage.libelle, CodeBlocage.valeur, CodeBlocage.manuel)
            .order_by(CodeBlocage.valeur.asc()))


def get_obj(i):
    return {
        'pk': i.pk,
        'label': i.libelle,
        'value': i.valeur,
        'manual': i.manuel
    }


@blocking_defects_blueprint.route('', methods=['POST'])
@jwt_required()
def get_list():
    andexpr = True
    data = json.loads(request.data)

    v_search_list = data.get('criterias', [])
    pagination = data.get('pagination', None)

    if len(v_search_list) > 0:
        lst = list(map(lambda s: (
            (CodeBlocage.libelle.contains(s.strip())) |
            (CodeBlocage.valeur.contains(s.strip()))
        ), v_search_list))
        search = reduce(operator.and_, lst)
        expr = reduce(operator.and_, [andexpr, search])
    else:
        expr = andexpr

    req = (CodeBlocage
           .select(CodeBlocage.pk, CodeBlocage.libelle, CodeBlocage.valeur, CodeBlocage.manuel)
           .where(expr)
           .order_by(CodeBlocage.valeur.asc()))
    count = req.count()
    if pagination is not None:
        v_start = pagination['page']
        v_length = pagination['rowsPerPage']
        req = req.limit(v_length).offset(v_start * v_length)

    return {
        'count': count,
        'list': [get_obj(item) for item in req]
    }, HTTP_200_OK


@blocking_defects_blueprint.route('', methods=['PUT'])
@jwt_required()
def update():
    args = json.loads(request.data)
    pk = args['pk']
    code = args['value']
    label = args['label']
    manual = args['manual']

    v = (CodeBlocage.select(CodeBlocage.pk)
         .where((CodeBlocage.valeur == code) & (CodeBlocage.pk != pk))).count()

    if v > 0:
        return {
            'message': 'blocking_defects.code.exists',
        }, HTTP_400_BAD_REQUEST

    try:
        block_defect = (CodeBlocage.select(CodeBlocage.pk)
                        .where((CodeBlocage.pk == pk))).get()
    except DoesNotExist:
        return {
            'message': 'blocking_defects.notexist',
        }, HTTP_400_BAD_REQUEST

    block_defect.valeur = code
    block_defect.libelle = label
    block_defect.manuel = manual
    block_defect.save()

    return get_obj(block_defect), HTTP_200_OK


@blocking_defects_blueprint.route('<string:pk>', methods=['DELETE'])
@jwt_required()
def delete(pk):
    try:
        v = (CodeBlocage.select().where((CodeBlocage.pk == pk))).get()

    except DoesNotExist:
        return {
            'message': 'blocking_defects.notexist',
        }, HTTP_400_BAD_REQUEST

    v.delete_instance()

    return {}, HTTP_204_NO_CONTENT


@blocking_defects_blueprint.route('create', methods=['POST'])
@jwt_required()
def create():
    args = json.loads(request.data)
    code = args['value']
    label = args['label']
    manual = args['manual']

    v = (CodeBlocage.select(CodeBlocage.pk).where((CodeBlocage.valeur == code))).count()

    if v > 0:
        return {
            'message': 'blocking_defects.code.exists',
        }, HTTP_400_BAD_REQUEST

    block_defect = CodeBlocage()
    block_defect.valeur = code
    block_defect.libelle = label
    block_defect.manuel = manual
    block_defect.save()

    return get_obj(block_defect), HTTP_200_OK


@blocking_defects_blueprint.route('export', methods=['PATCH'])
@jwt_required()
def export():
    data = json.loads(request.data)
    headers = data['headers']

    name = os.sep.join(
        [os.getcwd(), "tmp_export", "%s.xlsx" % uuid.uuid4()])
    stocks = get_all()
    generate_excel_file(name, headers, stocks, get_obj)

    return send_file(name, as_attachment=True)
