import json
import operator
from datetime import datetime
from functools import reduce

from flask import Blueprint, request
from flask_jwt_extended import jwt_required

from median.models import User
from peewee import JOIN
from common.status import HTTP_200_OK

from common.models import WebLogActions

actions_blueprint = Blueprint('actions', __name__)


@actions_blueprint.route('', methods=['POST'])
@jwt_required()
def get_all():
    data = json.loads(request.data)
    search_list = data.get('criterias', [])
    page_number = data.get('pageNumber', 0)
    start = data['start']
    end = data['end']
    page_size = data.get('pageSize', -1)
    andexpr = (WebLogActions.chrono >= start) & (WebLogActions.chrono <= end)

    if len(search_list) > 0:
        lst = list(map(lambda s: (
            (WebLogActions.username.contains(s.strip())) |
            (WebLogActions.action.contains(s.strip())) |
            (WebLogActions.message.contains(s.strip())) |
            (WebLogActions.equipement_type.contains(s.strip()))
        ),
                       search_list))
        search = reduce(operator.and_, lst)
        expr = reduce(operator.and_, [andexpr, search])
    else:
        expr = andexpr

    req = (WebLogActions.select(WebLogActions.pk.alias('pk'),
                                WebLogActions.chrono.alias('chrono'),
                                WebLogActions.action.alias('action'),
                                WebLogActions.message.alias('message'),
                                WebLogActions.equipement_type.alias('type'),
                                WebLogActions.username.alias('user'),
                                User.avatar.alias('avatar'))
           .join(User, JOIN.LEFT_OUTER, User.username == WebLogActions.username)
           .where(expr)
           .order_by(WebLogActions.chrono.desc()))

    total = req.count()

    req = req.limit(page_size).offset(page_number * page_size)

    return {
        'total': total,
        'list': [
            {
                'pk': item.pk,
                'date': datetime.utcfromtimestamp(item.chrono.timestamp()),
                'action': item.action,
                'message': item.message,
                'user': {
                    'name': item.user,
                    'avatar': item.avatar
                },
                'equipment': {
                    'type': item.type,
                }
            }
            for item in req.objects()]
    }, HTTP_200_OK
