import logging
import io

from datetime import datetime
from median.models import ItemValide, ListeValide, Stock, Magasin, Service, Product, ListeModel, ListeItemModel
from median.models import Espace, UnitDose, Seuil, Ucd, Cip, Printer, Patient
from median.views import RawConfig
from common.util import mustHaveRights
from common.status import (
    HTTP_200_OK,
    HTTP_204_NO_CONTENT,
    HTTP_500_INTERNAL_SERVER_ERROR,
    HTTP_400_BAD_REQUEST,
)
from flask import Blueprint, request
from flask import make_response, send_file
from flask_jwt_extended import jwt_required
from peewee import DoesNotExist, JOIN, fn
from median.database import mysql_db
from median.constant import EcoType, TypeMouvementGpao, TypeServiListe, MEDIANWEB_POSTE, CONFIG_WEB_CLE
from common.pdf import TablePDF
from fpdf.errors import FPDFException

from .services.tray_service import TrayService
from .services.completion_service import CompletionService
from .services.gpao_service import GPAOService
from .services.stock_service import StockService
from .services.label_service import LabelService
from .services.ward_service import WardService

from ..blueprint.product_blueprint import find_product_with_gtin_no_CIP

logger = logging.getLogger("median.completion")
loggerPrint = logging.getLogger("median.printer")
aideplus_completion_blueprint = Blueprint("aideplus_completion", __name__)

# Initialize services
tray_service = TrayService()
completion_service = CompletionService()
stock_service = StockService()
gpao_service = GPAOService(stock_service)
label_service = LabelService()
ward_service = WardService()

RESSOURCE_NAME = "WEB_AIDEPLUS_COMPLETION"
EXTERNAL_MAG = "P00"

serial_management: RawConfig = RawConfig(MEDIANWEB_POSTE, CONFIG_WEB_CLE).read("k_web_serial_management")
INCORRECT_SERIAL_MANAGEMENT = "incorrect"


@aideplus_completion_blueprint.route("tray/<string:code>", methods=["PUT"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canView=True)
def get_tray_of_pillboxes(code):
    """Main router function for tray retrieval"""
    data = request.get_json()
    chrono = data.get("chrono", None)
    id_chargement = data.get("idChargement", None)
    return tray_service.get_tray_of_pillboxes(code, chrono, id_chargement)


@aideplus_completion_blueprint.route("complete/<string:container_id>", methods=["PUT"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canEdit=True)
def complete_item(container_id):
    """Main router function for item completion"""

    try:
        request_data = request.get_json()
        data = request_data.get("data")
        products = data["scannedProducts"]
        loading_id = request_data.get("loading_id")
        nominative_mode = request_data.get("nominative", False)
        ward = request_data.get("ward", None)
        is_empty = request_data.get("isEmpty", False)

        if not is_empty:
            # Validate common parameters
            if not products or len(products) == 0:
                return {"error": "completion.plus.api.errors.no_products"}, HTTP_400_BAD_REQUEST

            reference = products[0]["reference"]
            total_quantity = sum(float(product.get("quantity", 0)) for product in products)

            if total_quantity <= 0:
                logger.error("No valid quantity provided")
                return {"error": "completion.plus.api.errors.no_valid_quantity"}, HTTP_400_BAD_REQUEST

            try:
                return completion_service.complete_item(
                    container_id, loading_id, reference, products, total_quantity, nominative_mode, ward
                )
            except Exception as err:
                logger.error(f"Completion error: {str(err)}")
                return {"message": str(err)}, HTTP_500_INTERNAL_SERVER_ERROR
        else:
            try:
                reference = request_data.get("reference")
                return completion_service.complete_item(
                    container_id, loading_id, reference, products, 0, nominative_mode, ward
                )
            except Exception as err:
                logger.error(f"Empty completion error: {str(err)}")
                return {"message": str(err)}, HTTP_500_INTERNAL_SERVER_ERROR

    except Exception as e:
        logger.error(str(e))
        return {"message": str(e)}, HTTP_500_INTERNAL_SERVER_ERROR


@aideplus_completion_blueprint.route("gpao", methods=["POST"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canView=True)
def fetch_gpao():
    """Fetch GPAO data with optional ward filtering"""
    try:
        data = request.get_json()
        ward_filter = data.get("ward")
        id_chargement = data.get("idChargement")
        gpao_type = data.get("gpaoType")
        chrono = data.get("chrono")

        return gpao_service.fetch_gpao_data(ward_filter, id_chargement, chrono, gpao_type)

    except Exception as e:
        logger.error(f"Error fetching GPAO data: {str(e)}")
        return {
            "error": "completion.plus.api.errors.fetch_gpao_failed",
            "details": str(e),
        }, HTTP_500_INTERNAL_SERVER_ERROR


@aideplus_completion_blueprint.route("global_wards", methods=["GET"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canView=True)
def fetch_global_wards():
    """Fetch list of wards for global completion filtering"""
    try:
        return ward_service.fetch_global_wards()

    except Exception as e:
        logger.error(f"Error fetching wards: {str(e)}")
        return {"error": f"Failed to fetch wards: {str(e)}"}, HTTP_500_INTERNAL_SERVER_ERROR


@aideplus_completion_blueprint.route("wards", methods=["POST"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canView=True)
def fetch_wards():
    """Fetch list of wards for filtering"""
    try:
        data = request.get_json()
        gpaoType = data.get("gpaoType", TypeMouvementGpao.COMPLEMENT.value)

        return ward_service.fetch_wards(gpaoType)

    except Exception as e:
        logger.error(f"Error fetching wards: {str(e)}")
        return {"error": f"Failed to fetch wards: {str(e)}"}, HTTP_500_INTERNAL_SERVER_ERROR


@aideplus_completion_blueprint.route("external_mag", methods=["GET"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canView=True)
def get_external_mag_api():
    mag = _get_external_mag()

    return mag, HTTP_200_OK


@aideplus_completion_blueprint.route("check", methods=["POST"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canView=True)
def check_product():
    # This is a check on scan of a product to add to the cart
    data = request.get_json()

    try:
        try:
            reference = data.get("reference", "")
            batch = data.get("batch", "")
            gtin = data.get("gtin", "")
            serial_code = data.get("serial", "")
            shortest_cip = str(int(gtin)) if gtin else ""
        except ValueError as e:
            logger.error(f"Invalid value: {str(e)}")
            return {"error": "completion.plus.api.errors.invalid_value", "details": str(e)}, HTTP_400_BAD_REQUEST

        try:
            Product.get(Product.reference == reference)
            ucd_cip: Cip = Cip.get((Cip.cip.contains(shortest_cip)))
            ref_ucd: Ucd = Ucd.get((Ucd.ucd == ucd_cip.ucd))
        except DoesNotExist:
            logger.info(f"Product not found for GTIN: {gtin}")
            return {"error": "completion.plus.api.errors.product_not_found_gtin", "gtin": gtin}, HTTP_200_OK

        seuils: Seuil = Seuil.get_or_none(Seuil.reference == ref_ucd.reference)

        if not seuils:
            return {}, HTTP_204_NO_CONTENT

        # We continue if there is a threshold
        try:
            serial: UnitDose = UnitDose.select().where(UnitDose.serial == serial_code).order_by(-UnitDose.pk).get()
        except DoesNotExist:
            logger.info(f"Serial not found : {serial_code}")
            return {"error": "completion.plus.api.errors.serial_not_found", "serial_code": serial_code}, HTTP_200_OK

        stock: Stock = (
            Stock.select(Stock)
            .join(Magasin, JOIN.LEFT_OUTER, on=(Stock.magasin == Magasin.mag))
            .where(
                (Stock.reference == ref_ucd.reference)
                & (Stock.lot == batch)
                & (Magasin.eco_type == EcoType.Externe.value)
                & (Stock.contenant == serial.contenant)
            )
        )
        if stock.count() == 0:
            if serial_management.value == INCORRECT_SERIAL_MANAGEMENT:
                lostStock = get_stock_with_fix(serial_code)

                if lostStock:
                    return {"message": "stock found with fix"}, HTTP_200_OK

            logger.info(f"No stock for reference and batch: {reference} / {batch}")
            return {
                "error": "completion.plus.api.errors.no_stock",
                "container": serial.contenant,
                "reference": reference,
                "batch": batch,
            }, HTTP_200_OK
        if stock.count() > 1:
            logger.error(f"Multiple stocks found for serial: {serial_code}, reference: {reference}, batch: {batch}")
            return {
                "error": "completion.plus.api.errors.multiple_stocks",
                "container": serial.contenant,
                "reference": reference,
                "batch": batch,
                "serial_code": serial_code,
            }, HTTP_200_OK

        return {"stock_qte": stock[0].quantite if stock else None}, HTTP_200_OK

    except Exception as e:
        logger.error(f"Error checking product: {str(e)}")
        return {
            "error": "completion.plus.api.errors.check_product_error",
            "details": str(e),
        }, HTTP_500_INTERNAL_SERVER_ERROR


def get_stock_with_fix(serial_code):
    """Get stock with fix - delegated to StockService"""
    return stock_service.get_stock_with_fix(serial_code)


@aideplus_completion_blueprint.route("print_box_label", methods=["POST"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canEdit=True)
def print_box_label():
    data = request.get_json()
    ipp = data.get("ipp")
    num_sej = data.get("num_sej")
    id_pilulier = data.get("id_pilulier")
    id_chargement = data.get("id_chargement")
    id_plateau = data.get("id_plateau")
    pk_plateau = data.get("pk_plateau")
    product_data = data.get("product")
    printer_pk = data.get("printer", None)
    ward_code = data.get("ward", None)

    global_mode = True if ipp == "GLOBAL" else False

    if not all([ipp, num_sej, id_pilulier, pk_plateau, id_plateau, product_data]):
        return {"error": "completion.plus.api.errors.missing_parameters"}, HTTP_400_BAD_REQUEST

    print(f"printing box label in {'global' if global_mode else 'nominative'} mode!")

    try:
        # Get all needed objects
        printer: Printer = Printer.get_or_none(Printer.pk == printer_pk)
        magasin: Magasin = Magasin.get_or_none(Magasin.eco_type == EcoType.Externe.value)

        try:
            shortest_cip = str(int(product_data["gtin"]))

            # Check if it exists, recreate ucd/cip if able
            test_cip = find_product_with_gtin_no_CIP(shortest_cip)
            if test_cip[1] != HTTP_200_OK:
                return test_cip[0], test_cip[1]

            ucd_cip: Cip = Cip.get_or_none((Cip.cip.endswith(shortest_cip)))
            ref_ucd: Ucd = Ucd.get_or_none((Ucd.ucd == ucd_cip.ucd))
        except Exception as e:
            logger.error(f"Product not found for GTIN: {product_data['gtin']}")
            logger.error(str(e))
            return {
                "error": "completion.plus.api.errors.product_not_found_gtin",
                "gtin": product_data["gtin"],
            }, HTTP_400_BAD_REQUEST

        if not global_mode:
            item: ItemValide = ItemValide.get_or_none(
                (ItemValide.id_pilulier == id_pilulier)
                & (ItemValide.reference == ref_ucd.reference)
                & (ItemValide.id_plateau == id_plateau)  # TODO: to be changed
            )
            liste: ListeValide = ListeValide.get_or_none(
                (ListeValide.pk == item.liste_pk) & (ListeValide.pk_pilulier == pk_plateau)
            )

            # Build patient data
            patient_info: Patient = Patient.get_or_none((Patient.ipp == liste.num_ipp))
            patient_data = {
                "ipp": liste.num_ipp,
                "first_name": liste.prenom,
                "last_name": liste.nom,
                "maiden_name": liste.nom_jf,
                "birthdate": liste.date_naissance,
                "stay": liste.num_sej,
            }

            # Compare with database data :
            if (
                patient_data["first_name"] != patient_info.prenom
                or patient_data["last_name"] != patient_info.nom
                or patient_data["maiden_name"] != patient_info.nom_jeune_fille
            ):
                logger.error(f"Patient data mismatch: {patient_data} != {patient_info}")
                # We'll trust the list but this should be reported
        else:
            # global mode
            try:
                ward: Service = Service.get(Service.code == ward_code["code"])
                ward_data = {
                    "ward_name": ward.libelle,
                    "ward_code": ward.code,
                }
            except DoesNotExist:
                logger.error(f"Ward not found for code: {ward_code}")
                return {"error": f"Ward not found for code: {ward_code}"}, HTTP_400_BAD_REQUEST
            except Exception as err:
                logger.error(f"Error fetching ward data: {str(err)}")
                return {
                    "error": "completion.plus.api.errors.fetch_wards_failed",
                    "details": str(err),
                }, HTTP_500_INTERNAL_SERVER_ERROR

        if not printer or not magasin or not ref_ucd or not ucd_cip:
            logger.error(
                f"Missing required object - Printer: {printer_pk}, Store: {magasin}, UCD: {ref_ucd}, CIP: {ucd_cip}"
            )
            return {"error": "completion.plus.api.errors.required_config_not_found"}, HTTP_400_BAD_REQUEST

        # Set only one entry datetime
        entry_datetime = datetime.now()

    except Exception as err:
        logger.error("Print Labels : Error while preparing data")
        logger.error(str(err))
        return {
            "message": "completion.print.error_data_preparation",
            "error": str(err),
        }, HTTP_500_INTERNAL_SERVER_ERROR

    with mysql_db.atomic() as transaction:
        try:
            current_date = datetime.now().strftime("%y%m%d")

            # Warn if the mag letter is empty
            if not magasin.lettre:
                logger.error(f"Magazine letter is empty for magazine {magasin.mag} (ID: {magasin.pk})")

            existing_serials = [
                i.serial
                for i in UnitDose.select(UnitDose.serial).where(
                    (UnitDose.serial.startswith(f"{current_date}{magasin.lettre}"))
                )
            ]

            # Check if the id_chargement is 3 chars long
            if len(str(id_chargement)) > 3:
                id_chargement = str(id_chargement[:3])
            elif len(str(id_chargement)) < 3:
                id_chargement = str(id_chargement).rjust(3, "0")

            for n in range(9999):
                serial = f"{current_date}{magasin.lettre}{str(id_chargement).rjust(3, '0')}{str(n).rjust(4, '0')}"

                # Testing if the serial is free, else increment the counter until we find a free one.
                if serial not in existing_serials:
                    break

            # f_sachet
            dose = UnitDose()
            dose.serial = serial
            dose.contenant = id_pilulier
            dose.chrono = entry_datetime
            dose.save()

            if not global_mode:
                label_service.print_label_nominative(
                    printer,
                    {
                        "serial": serial,
                        "product": product_data,
                        "ref": ref_ucd.reference,
                    },
                    patient_data,
                )
            else:
                label_service.print_label_global(
                    printer,
                    {
                        "serial": serial,
                        "product": product_data,
                        "ref": ref_ucd.reference,
                    },
                    ward_data,
                )
        except Exception as err:
            transaction.rollback()
            # print(str(err))
            logger.error(str(err))
            return {"error": str(err)}, HTTP_400_BAD_REQUEST

    return {"message": "completion.plus.api.success.printing_finished", "serial": serial}, HTTP_200_OK


@aideplus_completion_blueprint.route("updateGpao", methods=["POST"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canEdit=True)
def update_gpao():
    """Update GPAO data - delegated to GPAOService"""
    try:
        data = request.get_json()
        products = data.get("products")
        mvt_type = data.get("gpaoType")

        return gpao_service.update_gpao_status(products, mvt_type)

    except Exception as e:
        logger.error(f"Error updating GPAO data: {str(e)}")
        return {
            "error": "completion.plus.api.errors.update_gpao_failed",
            "details": str(e),
        }, HTTP_500_INTERNAL_SERVER_ERROR


def _get_external_mag():
    mag: Espace = Espace.get_or_none((Espace.alloc == 0) & (Espace.service == 1) & (Espace.mag == EXTERNAL_MAG))

    if mag:
        return {"mag": mag.mag, "type": mag.poste}
    else:
        logger.info(f"Searched for the store [{EXTERNAL_MAG}] but didn't find any")
        return {"mag": None, "type": None}


@aideplus_completion_blueprint.route("print", methods=["POST"])
@jwt_required()
def print_list():
    data = request.get_json()
    headerTranslations = data.get("headerTranslations", {})
    ward_code = data.get("code", None)
    chrono = data.get("chrono", None)
    id_chargement = data.get("idChargement", None)
    pillboxNumber = data.get("pillboxNumber", None)

    res = []
    try:
        head = [
            ("reference", 80, "L"),
            ("address", 20, "L"),
            ("quantite", 10, "R"),
            ("date", 15, "C"),
            ("fraction", 10, "C"),
            ("risque", 8, "C"),
        ]

        for i, (name, width, align) in enumerate(head):
            if name in headerTranslations:
                head[i] = (headerTranslations[name], width, align)

        if ward_code:
            lines = _get_global_items_for_pdf(ward_code, chrono, id_chargement)
        elif pillboxNumber:
            lines = _get_nominative_items_for_pdf(pillboxNumber)
            ward_code = lines[0].listevalide.service if lines else "?"
            chrono = lines[0].date if lines else "?"

        if len(lines) == 0:
            logger.info(f"No items to complete found for ward {ward_code} on date {chrono}, or pillbox {pillboxNumber}")
            return {"message": "completion.plus.api.errors.no_items_to_complete"}, HTTP_400_BAD_REQUEST

        for li in lines:
            adr_data = "ARX" if not hasattr(li, "has_seuil") else li.product.adr_pref if li.product.adr_pref else ""

            data = [
                f"[{li.reference}] {li.product.designation}",
                adr_data,
                f"{int(li.qte_serv)}/{int(li.qte_dem)}",
                li.date.strftime("%d/%m/%Y") if li.date else "",
                li.fraction,
                "R" if li.product.risque else "S" if li.product.stup else "",
            ]

            if len(head) != len(data):
                logger.error("Header and row length mismatch: %s != %s" % (len(head), len(res[0])))
                return {"error": "completion.plus.api.errors.header_row_mismatch"}, HTTP_500_INTERNAL_SERVER_ERROR

            # Add a last column to condition the line style, this should not have a header.
            # Only one last column accepted

            # 25-07-18 : Don't add the ARX products in this pdf
            # 25-08-07 : Add the ARX products in this pdf, heh
            # if li.product.adr_pref and li.product.adr_pref != "ARX":
            res.append(data)
            res[-1].append("N" if li.product.risque else "W" if li.product.stup else "")

        filename = f"completion-{ward_code}-{chrono}"
        logger.info(f"Trying to generate PDF for unload {filename} for list {ward_code}")
        file_io = io.BytesIO()

        try:
            pdf_file = TablePDF()
            pdf_file.doc_font(size=7)
            pdf_file.grid_header(head)
            pdf_file.grid_rows(res)

            pdf_file.doc_title("List of incomplete items for ward %s" % ward_code)
            if pillboxNumber:
                pdf_file.doc_title(f"List of incomplete items based on production found by pillbox n°{pillboxNumber}")

            pdf_file.doc_save(file_io)

        except FPDFException as ex:
            logger.error("Error when generating a PDF for replenishment %s" % ward_code)
            logger.error(str(ex))
            error_message = {"error": "replenishment.header.pdf.error", "param": [str(ex)]}
            response = make_response(error_message, HTTP_500_INTERNAL_SERVER_ERROR)
            response.headers["Content-Type"] = "application/json"
            return response

        file_io.seek(0)
        response = make_response(
            send_file(
                file_io, download_name="%s.pdf" % filename, mimetype="application/pdf", as_attachment=True, max_age=0
            )
        )
        response.headers.set("file-name", "%s.pdf" % filename)
        return response

    except DoesNotExist:
        logger.error("Replenishment %s not found" % ward_code)
        error_message = {"alertMessage": "replenishment.header.print.error", "param": [ward_code]}
        response = make_response(error_message, HTTP_500_INTERNAL_SERVER_ERROR)
        response.headers["Content-Type"] = "application/json"
        return response
    except Exception as error:
        logger.error("An error occurred while generating the PDF for replenishment %s: %s" % (ward_code, error))
        error_message = {"alertMessage": "replenishment.header.print.error", "param": [str(error)]}
        response = make_response(error_message, HTTP_500_INTERNAL_SERVER_ERROR)
        response.headers["Content-Type"] = "application/json"
        return response


def _get_global_items_for_pdf(ward_code, chrono, id_chargement):
    seuil_subquery_valid = Seuil.select(Seuil.pk).where(Seuil.reference == ItemValide.reference).limit(1)
    seuil_subquery_base = Seuil.select(Seuil.pk).where(Seuil.reference == ListeItemModel.reference).limit(1)

    list_items = (
        ListeItemModel.select(
            ListeItemModel.reference,
            fn.SUM(ListeItemModel.qte_dem).alias("qte_dem"),
            fn.SUM(ListeItemModel.qte_serv).alias("qte_serv"),
            ListeItemModel.fraction,
            fn.DATE(ListeModel.ddeb).alias("date"),
            Product,
            seuil_subquery_base.alias("has_seuil"),
        )
        .join(ListeModel, JOIN.INNER, on=(ListeItemModel.liste == ListeModel.liste))
        .join_from(ListeItemModel, Product, JOIN.INNER, on=(ListeItemModel.reference == Product.reference))
        .where(
            (ListeModel.service == ward_code)
            & (fn.DATE(ListeModel.ddeb) == chrono)
            & (ListeItemModel.type_servi << [TypeServiListe.GlobalePilulier.value, TypeServiListe.GlobaleBoite.value])
        )
        .group_by(ListeItemModel.reference, ListeItemModel.fraction, ListeModel.service, ListeModel.ddeb)
        .having(fn.SUM(ListeItemModel.qte_serv) < fn.SUM(ListeItemModel.qte_dem))
    )

    item_valides = (
        ItemValide.select(
            ItemValide.reference,
            fn.SUM(ItemValide.quantite_dem).alias("qte_dem"),
            fn.SUM(ItemValide.quantite_serv).alias("qte_serv"),
            ItemValide.fraction,
            fn.DATE(ListeValide.ddeb).alias("date"),
            Product,
            seuil_subquery_valid.alias("has_seuil"),
        )
        .join(ListeValide, JOIN.INNER, on=(ItemValide.liste_pk == ListeValide.pk))
        .join_from(ItemValide, Product, JOIN.INNER, on=(ItemValide.reference == Product.reference))
        .where(
            (ListeValide.service == ward_code)
            & (fn.DATE(ListeValide.ddeb) == chrono)
            & (ItemValide.type_servi << [TypeServiListe.GlobalePilulier.value, TypeServiListe.GlobaleBoite.value])
        )
        .group_by(ItemValide.reference, ItemValide.fraction, ListeValide.service, ListeValide.ddeb)
        .having(fn.SUM(ItemValide.quantite_serv) < fn.SUM(ItemValide.quantite_dem))
    )

    items = list_items.union(item_valides).order_by(-Product.risque, -Product.stup)

    return items


def _get_nominative_items_for_pdf(pillboxNumber):
    last_item = (
        ItemValide.select(ItemValide, ListeValide, Product)
        .join(ListeValide, JOIN.INNER, on=(ItemValide.liste_pk == ListeValide.pk))
        .join_from(ItemValide, Product, JOIN.INNER, on=(ItemValide.reference == Product.reference))
        .where((ItemValide.id_pilulier == pillboxNumber))
        .order_by(-ItemValide.pk)
        .get_or_none()
    )

    if last_item:
        # Create subquery for has_seuil
        seuil_subquery = Seuil.select(Seuil.pk).where(Seuil.reference == ItemValide.reference).limit(1)

        items = (
            ItemValide.select(
                ItemValide,
                ListeValide,
                fn.SUM(ItemValide.quantite_dem).alias("qte_dem"),
                fn.SUM(ItemValide.quantite_serv).alias("qte_serv"),
                fn.DATE(ListeValide.ddeb).alias("date"),
                Product,
                seuil_subquery.alias("has_seuil"),
            )
            .join(ListeValide, JOIN.INNER, on=(ItemValide.liste_pk == ListeValide.pk))
            .join_from(ItemValide, Product, JOIN.INNER, on=(ItemValide.reference == Product.reference))
            .where((ListeValide.id_chargement == last_item.listevalide.id_chargement))
            .group_by(ItemValide.reference, ItemValide.fraction)
            .having(fn.SUM(ItemValide.quantite_serv) < fn.SUM(ItemValide.quantite_dem))
        )
        return items.order_by(-Product.risque, -Product.stup)
    else:
        return []
