import json
import logging
import operator
from functools import reduce

from flask import Blueprint, request

from common.status import HTTP_400_BAD_REQUEST, HTTP_200_OK
from flask_jwt_extended import jwt_required
from median.models import Seuil, Product, Stock, Magasin

from peewee import fn, JOIN

acced_without_threshold_blueprint = Blueprint('acced_without_threshold', __name__)

logger = logging.getLogger('median')


@acced_without_threshold_blueprint.route('<string:store_pk>', methods=['POST'])
@jwt_required()
def get_stock(store_pk):
    try:
        data = json.loads(request.data)
        _criterias = data.get('criterias', [])
        andexpr = (Magasin.type_mag == store_pk) & (Seuil.pk.is_null())

        if len(_criterias) > 0:
            lst = list(map(lambda s: (
                (Product.designation.contains(s.strip())) |
                (Product.reference.contains(s.strip()))
            ), _criterias))
            search = reduce(operator.and_, lst)

            expr = reduce(operator.and_, [andexpr, search])
        else:
            expr = andexpr

        req = (Stock.select(Product.designation, Product.reference, Stock.fraction, Product.pk,
                            fn.SUM(Stock.quantite).alias('qty'), Magasin.type_mag)
               .join(Product, on=Product.reference == Stock.reference)
               .switch(Stock)
               .join(Magasin, on=Magasin.mag == Stock.magasin)
               .join(Seuil, JOIN.LEFT_OUTER, on=(Seuil.fraction == Stock.fraction) &
                                                (Seuil.reference == Stock.reference) &
                                                (Seuil.zone == Magasin.type_mag))
               .where(expr)
               .group_by(Product.designation, Product.reference, Stock.fraction)
               )

        return {'list': [
            {
                'reference_pk': item.product.pk,
                'designation': item.product.designation,
                'reference': item.product.reference,
                'fraction': item.fraction,
                'qty': item.qty,
                'type_mag': item.magasin.type_mag
            } for item in req
        ]}, HTTP_200_OK

    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_400_BAD_REQUEST
