"""
Replenish Blueprint for ACCED / AIDE *AND* Externals
"""

import logging
import time
import io
from flask import Blueprint, make_response, send_file, request
from flask_jwt_extended import jwt_required

# from marshmallow import Schema, fields
from median.models import ListeModel, ListeItemModel, Product
from peewee import DoesNotExist, JOIN

# from ressources.astus.utils import generate_excel_file
from common.pdf import TablePDF
from common.status import HTTP_204_NO_CONTENT, HTTP_500_INTERNAL_SERVER_ERROR, HTTP_404_NOT_FOUND
from fpdf.errors import FPDFException

acced_reappro_blueprint = Blueprint("reappro", __name__)

logger = logging.getLogger("median")


@acced_reappro_blueprint.route("line/<string:item_pk>", methods=["DELETE"])
@jwt_required()
def delete_lines(item_pk):
    try:
        item: ListeItemModel = ListeItemModel.get_or_none(ListeItemModel.pk == item_pk)

        if not item:
            errormsg = f"Reappro - Delete : Item non trouvée: {item_pk}"
            logger.error(errormsg)
            return {"message": errormsg}, HTTP_404_NOT_FOUND

        item.delete_instance()
        logger.info("Reappro - Delete : Item {item_pk} supprimée.")
    except Exception as error:
        logger.error(error.args)
        return {"alertMessage": "replenishment.line.delete.error", "param": [item_pk]}, HTTP_500_INTERNAL_SERVER_ERROR
    return {}, HTTP_204_NO_CONTENT


@acced_reappro_blueprint.route("print/<string:list_pk>", methods=["POST"])
@jwt_required()
def printing(list_pk):
    data = request.get_json()
    headerTranslations = data.get("headerTranslations", {})
    res = []
    try:
        reappro = ListeModel.get_by_id(list_pk)

        head = [
            ("reference", 80, "L"),
            ("quantite", 10, "R"),
            ("fraction", 10, "C"),
            ("forme", 10, "C"),
            ("risque", 10, "C"),
            ("stup", 10, "C"),
        ]

        for i, (name, width, align) in enumerate(head):
            if name in headerTranslations:
                head[i] = (headerTranslations[name], width, align)

        lines = (
            ListeItemModel.select(
                ListeItemModel.pk,
                ListeItemModel.reference,
                ListeItemModel.qte_dem,
                ListeItemModel.qte_serv,
                ListeItemModel.fraction,
                Product.designation,
                Product.risque,
                Product.stup,
                Product.forme,
            )
            .join(Product, JOIN.INNER, on=Product.reference == ListeItemModel.reference)
            .where(ListeItemModel.mode == reappro.mode, ListeItemModel.liste == reappro.liste)
            .order_by(-Product.risque, -Product.stup, -Product.forme, +Product.designation)
        )
        # print(lines)
        for li in lines:
            data = [
                "[%s] %s" % (li.reference, li.product.designation),
                li.qte_dem,
                li.fraction,
                "X" if li.product.forme else "",
                "X" if li.product.risque else "",
                "X" if li.product.stup else "",
            ]

            if len(head) != len(data):
                logger.error("Header and row length mismatch: %s != %s" % (len(head), len(res[0])))
                return {"error": "Header and row length mismatch"}, HTTP_500_INTERNAL_SERVER_ERROR

            # Add a last column to condition the line style, this should not have a header.
            # Only one last column accepted
            res.append(data)
            res[-1].append("N" if li.product.risque else "W" if li.product.stup else "")

        filename = "reappro-%s-%s" % (reappro.zone_fin, time.strftime("%Y%m%d_%H%M%S"))
        logger.info(f"Trying to generate PDF for replenishment {filename} for list {list_pk}")
        file_io = io.BytesIO()

        try:
            pdf_file = TablePDF()
            pdf_file.doc_font(size=7)
            pdf_file.doc_title(reappro.liste)
            pdf_file.grid_header(head)
            pdf_file.grid_rows(res)
            pdf_file.doc_save(file_io)

        except FPDFException as ex:
            logger.error("Error when generating a PDF for replenishment %s" % list_pk)
            logger.error(str(ex))
            error_message = {"error": "replenishment.header.pdf.error", "param": [str(ex)]}
            response = make_response(error_message, HTTP_500_INTERNAL_SERVER_ERROR)
            response.headers["Content-Type"] = "application/json"
            return response

        file_io.seek(0)
        response = make_response(
            send_file(
                file_io, download_name="%s.pdf" % filename, mimetype="application/pdf", as_attachment=True, max_age=0
            )
        )
        response.headers.set("file-name", "%s.pdf" % filename)
        return response

    except DoesNotExist:
        logger.error("Replenishment %s not found" % list_pk)
        error_message = {"alertMessage": "replenishment.header.print.error", "param": [list_pk]}
        response = make_response(error_message, HTTP_500_INTERNAL_SERVER_ERROR)
        response.headers["Content-Type"] = "application/json"
        return response
    except Exception as error:
        logger.error("An error occurred while generating the PDF for replenishment %s: %s" % (list_pk, error))
        error_message = {"alertMessage": "replenishment.header.print.error", "param": [str(error)]}
        response = make_response(error_message, HTTP_500_INTERNAL_SERVER_ERROR)
        response.headers["Content-Type"] = "application/json"
        return response


@acced_reappro_blueprint.route("selectbox", methods=["GET", "POST"])
@jwt_required()
def get_selectbox_values():
    return {
        "data": [
            {"value": "T", "label": "AideCut"},
            {"value": "K", "label": "AideTrack"},
            {"value": "0", "label": "Manual"},
        ]
    }, 200
