import json
import os
import uuid
from datetime import datetime

from flask import Blueprint, request, send_file

from common.status import HTTP_400_BAD_REQUEST
from flask_jwt_extended import jwt_required
from median.constant import EcoType
from peewee import fn, JOIN, SQL
from median.models import Magasin, Adresse, Seuil, Product, Stock, Config
from median.models.log import LogDefaut
from median.utils import logger
from ressources.astus.utils import generate_excel_file
from median.constant import ACCED_TAQUIN

acced_blueprint = Blueprint('acced', __name__)


@acced_blueprint.route('/count/waiting', methods=['GET'])
@jwt_required()
def get_count_messages():
    try:
        return {'data': get_count_messages_request()
                }, 200
    except Exception as error:
        logger.error('erreur recuperation magasin')
        return {'message': error.args}, 400


def get_count_messages_request():
    subquery = LogDefaut.select(LogDefaut.poste.alias('module'),
                                LogDefaut.num_defaut.alias('num_defaut'),
                                fn.MAX(LogDefaut.pk).alias('pk')) \
        .group_by(LogDefaut.poste, LogDefaut.num_defaut, LogDefaut.poste)

    req = (LogDefaut
           .select(LogDefaut.poste, fn.Count(1).alias('waiting_messages'))
           .join(subquery,
                 on=(
                     (subquery.c.pk == LogDefaut.pk) &
                     (subquery.c.module == LogDefaut.poste)))
           .where(LogDefaut.valeur == 1)
           .group_by(LogDefaut.poste))

    return ([{
        'type_mag': m.poste,
        'waiting_messages': m.waiting_messages,
    } for m in req])


def get_all():
    """
    Retrieve all ACCED informations
    """
    mag_list = []
    now = datetime.now()
    subquery = Stock.select(fn.COUNT(Stock.reference.distinct())).where(
        (Stock.magasin == Magasin.mag) & (Stock.date_peremption < now)
    )

    ms = Magasin.select(
        Magasin, subquery.alias('expired_counter')
    ).where((Magasin.eco_type == 'C') | (Magasin.eco_type == 'T')).order_by(Magasin.type_mag)

    for m in ms:
        res = {
            'pk': m.pk,
            'mag': m.mag,
            'type_mag': m.type_mag,
            'eco_type': m.eco_type,
            'libelle': m.libelle,
            'type_machine': m.type_machine,
            'nb_bloque': 0,
            'nb_taquin': 0,
            'nb_demi_taquin': 0,
            'nb_boite_pass': 0,
            'imprimante1': 1,
            'imprimante2': 1,
            'imprimante_active': 1,
            'avatar': m.avatar,
            'expired_counter': m.expired_counter,
            'label_2': m.lib_2,
            'label_3': m.lib_3,
            'label_4': m.lib_4,
            'label_5': m.lib_5,
            'value_2': m.dim_2,
            'value_3': m.dim_3,
            'value_4': m.dim_4,
            'value_5': m.dim_5,
        }

        if Magasin.eco_type != EcoType.Coupe.value:
            adr_block = Adresse.select(Adresse.adresse).where((Adresse.magasin == m.mag) & (Adresse.bloque > 0))
            res['nb_bloque'] = len(adr_block)

        if m.type_machine not in ACCED_TAQUIN:
            res['nb_taquin'] = None
            res['nb_demi_taquin'] = None
        else:
            adr_taquin = Adresse.select(fn.substr(Adresse.adresse, 1, fn.CHAR_LENGTH(Adresse.adresse) - 3)).where(
                (Adresse.magasin == m.mag) & (Adresse.format == 'BOITE PASS') &
                (Adresse.etat == 'L') & (Adresse.contenant == '') & (Adresse.bloque == 0)
            ).group_by(
                fn.substr(Adresse.adresse, 1, fn.CHAR_LENGTH(Adresse.adresse) - 3)
            ).having(
                fn.COUNT(SQL('*')) == 2
            )

            res['nb_taquin'] = len(adr_taquin)

            AdresseFront = Adresse.alias()
            AdresseBack = Adresse.alias()

            adr_demi_taquin = AdresseFront.select(AdresseFront.adresse).join(
                AdresseBack, on=(
                    (AdresseFront.magasin == AdresseBack.magasin) &
                    (fn.substr(AdresseFront.adresse, 1, fn.CHAR_LENGTH(AdresseFront.adresse) - 3) ==
                     fn.substr(AdresseBack.adresse, 1, fn.CHAR_LENGTH(AdresseFront.adresse) - 3))
                )
            ).where(
                (AdresseFront.magasin == m.mag) &
                (AdresseFront.format == 'BOITE PASS') &
                (AdresseFront.etat != 'X') & (AdresseFront.bloque == 0) &
                (AdresseBack.format == 'BOITE PASS') &
                (AdresseBack.etat == 'L') & (AdresseBack.bloque == 0) &
                (AdresseFront.adresse.endswith(' 1')) &
                (AdresseBack.adresse.endswith(' 3')) &
                (AdresseFront.contenant != '') &
                (AdresseBack.contenant == '')
            )

            res['nb_demi_taquin'] = len(adr_demi_taquin)

            nb_boite_pass = Adresse.select(Adresse.adresse).where(
                (Adresse.magasin == m.mag) & (Adresse.format == 'BOITE PASS')
            )
            res['nb_boite_pass'] = len(nb_boite_pass)

        prt = Config.select(Config.cle, Config.value).where(
            Config.poste == m.mag, Config.cle << ['ECO_IMPRIMANTE1', 'ECO_IMPRIMANTE2', 'ECO_IMPRIMANTE_ACTIVE']
        )
        for cfg in prt:
            key = cfg.cle.replace('ECO_', '').lower()
            res[key] = int(cfg.value)

        mag_list.append(res)

    return mag_list


@acced_blueprint.route('/all', methods=['GET'])
@jwt_required()
def get_all_acced():
    try:
        return {'data': get_all()}, 200
    except Exception as error:
        logger.error('erreur recuperation magasin')
        return {'message': error.args}, 400


@acced_blueprint.route('<string:store_pk>/threshold/export', methods=['POST'])
@jwt_required()
def exportThreshold(store_pk):
    try:
        data = json.loads(request.data)
        headers = data['headers']

        name = os.sep.join(
            [os.getcwd(), "tmp_export", "%s.xlsx" % uuid.uuid4()])

        req = Seuil.select(
            Seuil.reference, Product.designation, Seuil.fraction,
            Seuil.stock_mini, Seuil.stock_maxi,
            fn.ifnull(fn.SUM(Stock.quantite), 0).alias('qty')
        ).join(
            Product, on=Seuil.reference == Product.reference
        ).switch(Seuil).join(
            Magasin, on=(Seuil.zone == Magasin.type_mag)
        ).join(
            Stock, JOIN.LEFT_OUTER,
            on=(Stock.reference == Seuil.reference) &
               (Stock.fraction == Seuil.fraction) &
               (Stock.magasin == Magasin.mag)
        ).where(
            Magasin.pk == store_pk
        ).group_by(
            Seuil.reference, Seuil.fraction,
            Seuil.stock_mini, Seuil.stock_maxi,
            Product.designation
        ).order_by(Product.designation)

        generate_excel_file(name=name, headers=headers, items=req, transform_function=transform)
        return send_file(name, as_attachment=True)

    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_400_BAD_REQUEST


def transform(obj):
    return {
        'reference': obj.reference,
        'designation': obj.product.designation,
        'fraction': obj.fraction,
        'stock_mini': obj.stock_mini,
        'stock_maxi': obj.stock_maxi,
        'quantity': obj.qty
    }
