# Use by command flask fix
"""
Database fix and migration script for MedianWeb.

This module contains various database fixes and data corrections that are applied
to maintain data consistency and set up default configurations in the MedianWeb
application. The main apply() function runs all available fixes in sequence:

- Recompute foreign keys on web tables (fix_foreign_key)
- Fix inconsistent data in profile, warehouse, and user tables (fix_datas)
- Set default French language for Deenova users (deenova_user_lang_default)
- Fix web user password issues (update_password_web)
- Fix available languages (update_language)
- Set default rights for DEENOVA and ECO-DEX profiles (deenova_profil_default)
- Update catalog configuration (update_catalog)
- Generate equipment topography (generate_equipment_topography)
- Block addresses with specific states (fix_addresse_state_a)
- Check and create drug country system entries (get_or_create_drug_county_system)
- Ensure default codes are available in database (deenova_default_labels)
- Check and create container format entries (deenova_default_format)
- Compute format per warehouse (fill_mag_format_table)
- Fix missing warehouse groups (fix_equipement_group)
- Update minimum loading IDs (update_loading_ids)
- Fix contract threshold parameters (update_contract_threshold)
- Set default configuration parameters (deenova_default_config)
- Update DPM (Demande Parametrage Medicament / Medicine Settings Request) properties (update_dpm_config)
- Fix equipment types after constant updates (fix_equipment_types)
- Set default printer settings (set_default_printer_values)
- Add default ACCED Zebra printer configuration (set_default_acced_printer)
- Add default RIEDL Zebra printer configuration (set_default_riedl_printer)
- Fix external loading counters (fix_external_loading_counters)
- Fix list notes (fix_list_notes)
- Add Twincat configuration for equipment (fix_config_twincat_amsid)
- Add missing free fields (fix_freefields)
- Fix zone/zonemag for new equipments (fix_zonemag)
"""

import logging
import os

import bcrypt
from common.models import WebLang, WebThresholdParameter
from median.constant import (
    EcoType,
    PickingMachineType,
    CuttingMachineType,
    EquipmentType,
    ExternalMachineType,
    RiedlMachineType,
    DPM_PINCE,
    DPM_VENTOUSE,
    DpmType,
    MEDIANWEB_POSTE,
    TOUS_POSTE,
    CONFIG_CFG_CLE,
    CONFIG_WEB_CLE
)
from median.database import mysql_db, decripte
from median.models import Profil, Config, Compteur, Poste
from median.models import (
    User,
    CatalogType,
    CatalogVersion,
    CatalogOption,
    CatalogUnitStock,
    CatalogLines,
    CatalogColumns,
    Magasin,
    EquipmentUnitStock,
    Adresse,
    Stock,
    CodeBlocage,
    MagasinFormat,
    Format,
    Patient,
    Sejour,
    MagasinGroupe,
    Gtin,
    PrinterType,
    PrinterLabel,
    Printer,
    PrinterMag,
    PrinterCommand,
    PrinterCommandLabel,
    ListeErrorModel,
    ListeModel,
    Label,
    Zone,
    ZoneMag,
)
from peewee import DoesNotExist, JOIN
from ressources.equipments.acced.acceds import regenerate

logger = logging.getLogger("median.fix")


def apply():
    print("Recompute Foreign Key on Web_ table")
    fix_foreign_key()
    print("Fix table datas f_profil, f_mag, f_user")
    fix_datas()
    print("Set french lang on Deenova user")
    deenova_user_lang_default()
    print("Fix password web")
    update_password_web()
    print("Fix language available")
    update_language()
    print("Fix default rights for DEENOVA, ECO-DEX profil")
    deenova_profil_default()
    print("Catalog configuration")
    update_catalog()
    print("Generation equipment topography")
    generate_equipment_topography()
    print("Block address with state = A and bloque = 1")
    fix_addresse_state_a()
    print("Check drug country system")
    get_or_create_drug_county_system()
    print("Check if default codes are available in the database")
    deenova_default_labels()
    print("Check if container format are available in the database")
    deenova_default_format()
    print("Compute the format per warehouse")
    fill_mag_format_table()
    print("Fix missing warehouse groups")
    fix_equipement_group()
    print("Fix min loading Id")
    update_loading_ids()
    print("Fix contract threshold")
    update_contract_threshold()
    print("Fix config parameters")
    deenova_default_config()
    print("Fix dpm properties")
    update_dpm_config()
    print("Fix store types after use of constants")
    fix_equipment_types()
    print("Fix printer default settings")
    set_default_printer_values()
    print("Add ACCED Zebra printer with default value")
    set_default_acced_printer()
    print("Add RIEDL Zebra printer with default value")
    set_default_riedl_printer()
    print("Add external loading counters")
    fix_external_loading_counters()
    print("Fix list notes")
    fix_list_notes()
    print("Add Twincat Configuration for equipment")
    fix_config_twincat_amsid()
    print("Add missing free fields")
    fix_freefields()
    print("Fix the zone/zonemag tables")
    fix_zonemag()


def fix_foreign_key():
    """Delete and recreate the foreign key"""
    logger.info("Recreate all foreign key")
    execute_ddl_query("ALTER TABLE web_menu_i18n DROP FOREIGN KEY web_menu_i18n_ibfk_2;")
    execute_ddl_query(
        "ALTER TABLE web_menu_i18n ADD CONSTRAINT web_menu_i18n_ibfk_2 "
        "FOREIGN KEY (lang_id) REFERENCES web_lang(id) "
        "ON DELETE CASCADE ON UPDATE CASCADE;"
    )
    execute_ddl_query("ALTER TABLE web_menu_i18n DROP FOREIGN KEY web_menu_i18n_ibfk_1;")
    execute_ddl_query(
        "ALTER TABLE web_menu_i18n ADD CONSTRAINT web_menu_i18n_ibfk_1 "
        "FOREIGN KEY (menu_id) REFERENCES web_menu(id) "
        "ON DELETE CASCADE;"
    )
    execute_ddl_query("ALTER TABLE web_menu DROP FOREIGN KEY web_menu_ibfk_1;")
    execute_ddl_query(
        "ALTER TABLE web_menu ADD CONSTRAINT web_menu_ibfk_1 "
        "FOREIGN KEY (parent_id) REFERENCES web_menu(id) "
        "ON DELETE CASCADE ON UPDATE CASCADE;"
    )
    execute_ddl_query("ALTER TABLE web_form_i18n DROP FOREIGN KEY web_form_i18n_ibfk_1;")
    execute_ddl_query(
        "ALTER TABLE web_form_i18n ADD CONSTRAINT web_form_i18n_ibfk_1 "
        "FOREIGN KEY (lang_id) REFERENCES web_lang(id) "
        "ON DELETE CASCADE ON UPDATE CASCADE;"
    )


def fix_datas():
    """Fix inconsistent datas"""
    execute_ddl_query("UPDATE f_profil SET x_visu=1 WHERE x_visu IS NULL;")
    execute_ddl_query("UPDATE f_profil SET x_edit=0 WHERE x_edit IS NULL;")
    execute_ddl_query("UPDATE f_mag SET x_libelle=x_type_mag WHERE x_libelle = '';")
    execute_ddl_query("UPDATE f_dest SET x_type_dest='SERVICE' WHERE x_type_dest = '';")
    # si vide, alors utilisation du paramèretre k_eco_tri_pil
    # execute_ddl_query("UPDATE f_dest SET x_tri='CHAMBRE' WHERE x_tri = '' AND x_type_dest='SERVICE';")
    execute_ddl_query("UPDATE f_user SET x_astup = 0 WHERE x_astup IS NULL;")
    execute_ddl_query("UPDATE f_user SET x_login = '' WHERE x_login IS NULL;")
    execute_ddl_query("UPDATE f_user SET x_url = '' WHERE x_url IS NULL;")
    execute_ddl_query("UPDATE f_user SET x_badge = '' WHERE x_badge IS NULL;")
    execute_ddl_query("UPDATE f_user SET x_inventaire = 0 WHERE x_astup IS NULL;")
    execute_ddl_query(
        "UPDATE f_user SET x_inventaire = 1, x_maintenance = 1, x_service = 'DEENOVA' WHERE x_profil = 'ECO-DEX';"
    )  # noqa
    execute_ddl_query(
        "UPDATE f_user SET x_inventaire = 1, x_maintenance = 1, x_service = 'DEENOVA' WHERE x_profil = 'DEENOVA';"
    )  # noqa
    execute_ddl_query("UPDATE f_mag SET x_type_machine='RIEDL_2' WHERE x_type_machine='' AND x_eco_type='L';")  # noqa
    # Old median client failed to display datas when x_id is blank,
    execute_ddl_query(
        "UPDATE f_config SET x_id = CONCAT('_PK', LPAD(x_pk, 7, '0')) "
        "WHERE x_poste = 'TOUS' AND x_cle = 'cfg' AND x_id = '';"
    )  # noqa
    execute_ddl_query("UPDATE f_defaut SET x_manuel=1 WHERE x_valeur in (0, 1);")  # noqa
    execute_ddl_query("UPDATE f_resrc SET x_group = 'FOXPRO' WHERE x_group = '';")  # noqa
    get_or_create_poste("MEDIANWEB", {"pc": ""})


def deenova_profil_default():
    get_or_create_profil("WEB_SETTINGS_BLOCKING_DEFECTS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_INVENTORY", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_STOCK", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # ACCED
    get_or_create_profil("WEB_ACCED", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_REAPPRO", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_DISPENSATION", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_DASHBOARD_STATS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ACCED_FEASIBILITY", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # get_or_create_profil("WEB_ACCED_HISTORY_MESSAGES", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_WITHOUT_TRESHOLD", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_SETTING_ACCED_MESSAGES", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ACCED_ZERO_CONSUMPTION", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ACCED_WAITING_MESSAGES", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ACCED_DETAIL", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_SAVED_LIST", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_THRESHOLD", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ACCED_BLOCK_CAUSES", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PATIENTS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # TODO: active when simulator is fully merge
    # get_or_create_profil("WEB_ACCED_THRESHOLD_SIMULATOR", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # Administration
    get_or_create_profil("WEB_ADMINISTRATION", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_HISTORY_INTERFACE", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ORGANILOG", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ACTIONS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PATIENTS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_WSCLIENT_ERROR", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_DPM", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_MEDICATION_COMPLETION", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # ASTUS
    get_or_create_profil("WEB_ASTUS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_FEASIBILITY", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_STOCK", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_DRAWER_SEARCH", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_EVENTS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_REPLENISHMENT", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_DETAIL", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_WARD_CONFIGURATION", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_INVENTORY", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_OUTPUT", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_CONSUMPTION", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_ASTUS_MULTI_DOSES", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # Dashboard
    get_or_create_profil("WEB_DASHBOARD", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_DASHBOARD_COMPTEUR", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # Document
    get_or_create_profil("WEB_PICKING_REPORT", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_HISTORY", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_LOCATIONS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PRESCRIPTION", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # get_or_create_profil("WEB_BLOCKED_HISTORY", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # Reference
    get_or_create_profil("WEB_INVENTORY", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PRODUIT", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PRODUIT_BATCH", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PRODUIT_BLACKLIST", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PRODUIT_PARAM", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PRODUIT_PARAM_UCD", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PRODUIT_THRESHOLD", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_REF_ASTUS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_REF_FORMAT", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_REF_RIEDL", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_RISKY_DRUGS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_IDENTIFY", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # Riedl
    get_or_create_profil("WEB_RIEDL", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_RIEDL_INPUT", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_RIEDL_OUTPUT", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_RIEDL_EXCHANGE", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_RIEDL_STOCK_EXPORT", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # Settings
    get_or_create_profil("WEB_GROUP", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PARAMETER", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PEIGNES", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_SETTINGS_BLOCKING_DEFECTS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_SETTING_LANG", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_USERS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_SETTING_EQUIPMENT", ["DEENOVA", "ECO-DEX"])
    get_or_create_profil("WEB_CATALOG", ["DEENOVA", "ECO-DEX"])
    get_or_create_profil("WEB_MAGGROUPS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_PRINTERS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # Wards
    get_or_create_profil("WEB_SERVICES", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # External
    get_or_create_profil("WEB_EXTERNAL", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_EXTERNAL_STOCK", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_EXTERNAL_UNLOAD", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_EXTERNAL_RETURNS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_RIEDL_REPLENISH", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_LOCATIONS_GENERATOR", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # Aide-Plus
    get_or_create_profil("WEB_AIDEPLUS", ["DEENOVA", "ECO-DEX", "ADMIN"])
    get_or_create_profil("WEB_AIDEPLUS_COMPLETION", ["DEENOVA", "ECO-DEX", "ADMIN"])
    # SPECIFIC TOOLS
    get_or_create_profil("WEB_DATA_FETCHER", ["DEENOVA", "ECO-DEX", "ADMIN"])  # See bamberg


def deenova_user_lang_default():
    """Add default lang on deenova user"""
    execute_ddl_query("UPDATE f_user SET x_lang = 'fr_FR' WHERE x_profil = 'ECO-DEX' AND x_lang='';")
    execute_ddl_query("UPDATE f_user SET x_lang = 'fr_FR' WHERE x_profil = 'DEENOVA' AND x_lang='';")
    execute_ddl_query("UPDATE f_user SET x_lang = 'en_GB' WHERE x_user = 'ecopilote';")


def deenova_default_labels():
    """Check if blocking default is available on the database"""
    get_or_create_default(0, "Déblocage")
    get_or_create_default(1, "Blocage Median")
    get_or_create_default(999, "En inventaire")


def deenova_default_format():
    get_or_create_format("BOITE PASS", {"nb_div": 2, "dim_y": 2})
    get_or_create_format("BAC", {"nb_div": 1})
    get_or_create_format("SHELF", {})
    get_or_create_format("BAC 1/3 B", {"typeBac": "bP1/3"})
    get_or_create_format("BAC 1/3 H", {"typeBac": "hP1/3"})
    get_or_create_format("BAC 1/7 B", {"typeBac": "bP1/7"})
    get_or_create_format("BAC 1/7 H", {"typeBac": "hP1/7"})
    get_or_create_format("PIL 1/3 B", {"typeBac": "bL1/3", "nb_div": 2})
    get_or_create_format("PIL 1/3 H", {"typeBac": "hL1/3", "nb_div": 2})
    get_or_create_format("TIROIR 1/3", {"nb_div": 6, "typeBac": "tir1/3"})
    get_or_create_format("TIROIR 1/4", {"nb_div": 8, "typeBac": "tir1/4"})
    get_or_create_format("TIROIR 1/5", {"nb_div": 10, "typeBac": "tir1/5"})
    get_or_create_format("CAISSE BP", {"dim_x": 600, "dim_y": 400, "dim_z": 220, "nb_div": 18})
    get_or_create_format("CAISSE EXO", {"dim_x": 600, "dim_y": 400, "dim_z": 220, "nb_div": 12})
    # Posage for AIDE CUT
    get_or_create_format("AC 001", {"dim_x": 22, "dim_y": 16, "dim_z": 0, "nb_div": 30, "typeBac": "AC001"})
    get_or_create_format("AC 002", {"dim_x": 21, "dim_y": 20, "dim_z": 0, "nb_div": 24, "typeBac": "AC002"})
    get_or_create_format("AC 003", {"dim_x": 26, "dim_y": 26, "dim_z": 0, "nb_div": 15, "typeBac": "AC003"})
    # Posage AIDE TRACK for AIDE PICK
    get_or_create_format("AP 001 S", {"dim_x": 30, "dim_y": 38, "dim_z": 0, "nb_div": 20, "typeBac": "AP001S"})
    get_or_create_format("AP 001 P", {"dim_x": 72, "dim_y": 92, "dim_z": 0, "nb_div": 25, "typeBac": "AP001P"})
    get_or_create_format("AP 002 P", {"dim_x": 42, "dim_y": 132, "dim_z": 0, "nb_div": 35, "typeBac": "AP002P"})
    get_or_create_format("AP 003 P", {"dim_x": 34, "dim_y": 62, "dim_z": 0, "nb_div": 75, "typeBac": "AP003P"})
    # Posage AIDE TRACK for SHELVES
    get_or_create_format("AT 001", {"dim_x": 17, "dim_y": 17, "dim_z": 0, "nb_div": 56, "typeBac": "AT001"})
    get_or_create_format("AT 002", {"dim_x": 22, "dim_y": 17, "dim_z": 0, "nb_div": 48, "typeBac": "AT002"})
    get_or_create_format("AT 003", {"dim_x": 23, "dim_y": 23, "dim_z": 0, "nb_div": 24, "typeBac": "AT003"})
    get_or_create_format("AT 004", {"dim_x": 28, "dim_y": 19, "dim_z": 0, "nb_div": 35, "typeBac": "AT004"})
    get_or_create_format("AT 005", {"dim_x": 33, "dim_y": 33, "dim_z": 0, "nb_div": 16, "typeBac": "AT005"})
    get_or_create_format("AT 006", {"dim_x": 46, "dim_y": 46, "dim_z": 0, "nb_div": 9, "typeBac": "AT006"})
    get_or_create_format("EXT_BOX", {"dim_x": 0, "dim_y": 0, "dim_z": 0, "nb_div": 9999, "typeBac": "EXT_BOX"})


def deenova_default_config():
    get_or_create_config(
        MEDIANWEB_POSTE,
        CONFIG_CFG_CLE,
        "k_eco_dir_dpm",
        ref_id="_DPM00001",
        type_value="C",
        value="C:\\MEDIAN\\Partage\\DPM_en_cours",
        hierarchie_menu="MEDIANWEB\\DPM\\Dossier en cours",
        type_champ="dir",
        ordre=1,
    )
    get_or_create_config(
        MEDIANWEB_POSTE,
        CONFIG_CFG_CLE,
        "k_eco_dir_fiche_pince",
        ref_id="_DPM00002",
        type_value="C",
        value="",
        hierarchie_menu="MEDIANWEB\\DPM\\Fiche produit pince",
        type_champ="dir",
        ordre=2,
    )
    get_or_create_config(
        MEDIANWEB_POSTE,
        CONFIG_CFG_CLE,
        "k_eco_dir_fiche_ventouse",
        ref_id="_DPM00003",
        type_value="C",
        value="",
        hierarchie_menu="MEDIANWEB\\DPM\\Fiche produit ventouse",
        type_champ="dir",
        ordre=3,
    )
    get_or_create_config(
        MEDIANWEB_POSTE,
        CONFIG_CFG_CLE,
        "k_eco_dir_fiche_track",
        ref_id="_DPM00004",
        type_value="C",
        value="",
        hierarchie_menu="MEDIANWEB\\DPM\\Fiche produit AIDE TRACK",
        type_champ="dir",
        ordre=4,
    )
    get_or_create_config(
        TOUS_POSTE,
        CONFIG_CFG_CLE,
        "k_eco_client",
        type_value="C",
        value="!! CLIENT !!",
        hierarchie_menu="Median\\General\\Client",
        type_champ="reg",
        ordre=0,
    )
    get_or_create_config(
        MEDIANWEB_POSTE,
        CONFIG_WEB_CLE,
        "k_web_gtin_management",
        type_value="C",
        value="",
        hierarchie_menu="MEDIANWEB\\GTIN\\Unconventionnal gtin use",
        type_champ="reg",
        ordre=0,
    )
    get_or_create_config(
        MEDIANWEB_POSTE,
        CONFIG_WEB_CLE,
        "k_web_datamatrix_management",
        type_value="C",
        value="",
        hierarchie_menu="MEDIANWEB\\DATAMATRIX\\Unconventionnal datamatrix scanning",
        type_champ="reg",
        ordre=0,
    )
    get_or_create_config(
        MEDIANWEB_POSTE,
        CONFIG_WEB_CLE,
        "k_web_serial_management",
        type_value="C",
        value="",
        hierarchie_menu="MEDIANWEB\\SERIALS\\Unconventionnal serial/stock association",
        type_champ="reg",
        ordre=0,
    )
    get_or_create_config(
        MEDIANWEB_POSTE,
        CONFIG_WEB_CLE,
        "k_web_gpao_management",
        type_value="C",
        value="",
        hierarchie_menu="MEDIANWEB\\GPAO\\Gpao configuration for WMS communication",
        type_champ="reg",
        ordre=0,
    )
    get_or_create_config(
        TOUS_POSTE,
        CONFIG_CFG_CLE,
        "k_ua_retour_externe",
        type_value="C",
        value="",
        hierarchie_menu="Mouvements\\UF\\UF retour externe",
        type_champ="reg",
        info_champ="!k",
        ordre=51,
    )
    get_or_create_config(
        TOUS_POSTE,
        CONFIG_CFG_CLE,
        "k_eco_nb_péremption_global",
        type_value="N",
        value="7",
        hierarchie_menu="Median\\Nombre de jour pour faisabilité<>périmés",
        type_champ="num",
        info_champ="",
        ordre=52,
    )
    get_or_create_config(
        MEDIANWEB_POSTE,
        CONFIG_WEB_CLE,
        "k_web_data_fetch_url_1",
        type_value="C",
        value="",
        hierarchie_menu="MEDIANWEB\\Data Fetcher\\URL 1",
        type_champ="reg",
        info_champ="",
        ordre=53,
    )
    get_or_create_config(
        MEDIANWEB_POSTE,
        CONFIG_WEB_CLE,
        "k_web_data_fetch_url_2",
        type_value="C",
        value="",
        hierarchie_menu="MEDIANWEB\\Data Fetcher\\URL 2",
        type_champ="reg",
        info_champ="",
        ordre=54,
    )
    get_or_create_config(
        MEDIANWEB_POSTE,
        CONFIG_WEB_CLE,
        "k_web_data_fetch_url_3",
        type_value="C",
        value="",
        hierarchie_menu="MEDIANWEB\\Data Fetcher\\URL 3",
        type_champ="reg",
        info_champ="",
        ordre=55,
    )
    get_or_create_config(
        MEDIANWEB_POSTE,
        CONFIG_WEB_CLE,
        "k_web_data_fetch_target",
        type_value="C",
        value="",
        hierarchie_menu="MEDIANWEB\\Data Fetcher\\target",
        type_champ="reg",
        info_champ="",
        ordre=56,
    )
    get_or_create_config(
        MEDIANWEB_POSTE,
        CONFIG_WEB_CLE,
        "k_web_data_fetch_upload_1",
        type_value="C",
        value="",
        hierarchie_menu="MEDIANWEB\\Data Fetcher\\upload",
        type_champ="dir",
        info_champ="",
        ordre=56,
    )
    get_or_create_config(
        MEDIANWEB_POSTE,
        CONFIG_WEB_CLE,
        "k_web_data_fetch_base_url",
        type_value="C",
        value="",
        hierarchie_menu="MEDIANWEB\\Data Fetcher\\Base Url",
        type_champ="dir",
        info_champ="",
        ordre=56,
    )
    get_or_create_config(
        MEDIANWEB_POSTE,
        CONFIG_CFG_CLE,
        "k_eco_cip_index_libre",
        type_value="C",
        value="",
        hierarchie_menu="Reference/Unlock CIP edit",
        type_champ="reg",
        info_champ="",
        ordre=57,
    )
    get_or_create_config(
        MEDIANWEB_POSTE,
        CONFIG_CFG_CLE,
        "k_web_ucd_fields",
        type_value="N",
        value="",
        hierarchie_menu="Reference/Unlock CIP edit",
        type_champ="bool",
        info_champ="",
        ordre=58,
    )


def update_password_web():
    users = User.select().where(User.passwordWeb == "")

    for usr in users:
        old_password = decripte(usr.password)
        new_password = bcrypt.hashpw(old_password.encode("UTF_8"), bcrypt.gensalt())
        User.update(passwordWeb=new_password, isTemporary=1).where(User.pk == usr.pk).execute()
        print(f"User: {usr.user} -> No Web password, set to temporary")


def update_language():
    """
    Enable English and french language by default
    """
    WebLang.update(enable=True).where(WebLang.code == "fr_FR").execute()
    WebLang.update(enable=True).where(WebLang.code == "en_GB").execute()


def update_loading_ids():
    load_counters = (
        Compteur.select(Compteur.val, Compteur.cle)
        .where(Compteur.cle.startswith("ID_CHARGEMENT_"))
        .order_by(Compteur.val.asc())
    )

    last_counter = None
    for counter in load_counters:
        equipment = counter.cle[14:]

        try:
            config = (
                Config.select()
                .where((Config.propriete == "k_min_chargement") & (Config.cle == equipment))
                .order_by(Config.value)
            ).get()
        except DoesNotExist:
            config = Config()
            config.propriete = "k_min_chargement"
            config.type_value = "N"
            config.cle = equipment
            config.poste = TOUS_POSTE

        if last_counter is None:
            value = 0
            last_counter = 0
        else:
            current = str(counter.val)
            value = current[:1].ljust(len(current), "0")
            last_counter = int(value)

        config.value = value
        config.save()


def update_contract_threshold():
    """Used for the stats (client objectives)"""
    _update_config(property="contract_dose_pillbox", value=8)
    _update_config(property="contract_cadence_pillbox", value=50)
    _update_config(property="contract_cadence_dose_pillbox", value=400)
    _update_config(property="contract_dose_stack", value=8)
    _update_config(property="contract_cadence_stack", value=40)
    _update_config(property="contract_cadence_dose_stack", value=320)
    _update_config(property="contract_completion_percentage", value=98)
    _update_config(property="contract_availability_percentage", value=96)
    _update_config(property="contract_cut_rejet_percentage", value=5)
    _update_config(property="contract_blocked_box_per_hour", value=10)
    _update_config(property="contract_duplicate_per_thousand", value=4)


def update_dpm_config():
    _update_config(property="k_eco_dir_dpm", value="tmp_export", type_value="C")
    _update_config(property="k_eco_dir_fiche", value="tmp_export", type_value="C")
    _update_config(property="k_eco_dir_fiche2", value="tmp_export", type_value="C")
    _update_config(property="k_eco_dir_track", value="tmp_export", type_value="C")

    # Check all warehouse to define the DPM Type
    try:
        for m in Magasin.select(Magasin).where(Magasin.eco_type << ["C", "T"], Magasin.type_dpm == ""):
            logger.debug("DPM Type empty for %s" % m.mag)
            if m.type_machine in DPM_PINCE:
                m.type_dpm = DpmType.Pince.value
            elif m.type_machine in DPM_VENTOUSE:
                m.type_dpm = DpmType.Ventouse.value
            m.save()
    except Exception:
        logger.error("Error when request DPM_TYPE")

    # TODO: Go in all these folders and set the DPM type

    # TODO: update the reap_mode following the DPM type

    # Check the type of DPM we have, to fix automatically
    Gtin.update({Gtin.type_dpm: "Z"}).where(Gtin.type_dpm == "", Gtin.dossier == "0").execute()
    dpm_type_list = Magasin.select(Magasin.type_dpm).distinct().where(Magasin.eco_type << ["C", "T"])
    if len(dpm_type_list) == 1:
        logger.info("Update all Gtin line with the DPM Type %s" % dpm_type_list[0].type_dpm)
        Gtin.update({Gtin.type_dpm: dpm_type_list[0].type_dpm}).where(Gtin.type_dpm == "").execute()
    else:
        logger.warning("f_ucd_cip automatic migration cannot be done, we have more than 1 DPM type")


def _update_config(property, value, type_value="N"):
    try:
        config = Config.select(Config).where((Config.propriete == property)).order_by(Config.value).get()
    except DoesNotExist:
        config = Config()
        config.propriete = property
        config.type_value = type_value
        config.poste = TOUS_POSTE
        config.value = value

    if not config.value:
        config.value = value
    config.save()


def update_threshold_parameters():
    equipments = Magasin.select(Magasin.pk).where(Magasin.eco_type << [EcoType.Coupe.value, EcoType.Cueillette.value])
    WebThresholdParameter.delete().where(WebThresholdParameter.equipment_pk.not_in(equipments)).execute()
    for equipment in equipments:
        parameter = WebThresholdParameter.select(WebThresholdParameter.pk).where(
            WebThresholdParameter.equipment_pk == equipment.pk
        )

        if not any(parameter):
            WebThresholdParameter.create(equipment_pk=equipment.pk)


def fix_addresse_state_a():
    """Add block =1 on stock and state X on address"""
    addrs_query = Adresse.select(Adresse.adresse).where(Adresse.etat == "A")
    addrs = [a.adresse for a in addrs_query]
    logger.info("We have %i addresses with bad state!" % len(addrs))
    Stock.update(bloque=1).where(Stock.adresse << addrs).execute()
    Adresse.update(etat="X", bloque=1, bloque_message="à controler").where(Adresse.etat == "A").execute()


def fix_equipement_group():
    """Add mag group for each equipmeent where group is empty"""
    try:
        for m in Magasin.select(Magasin).where(Magasin.eco_type << ["A", "L", "E"], Magasin.groupe == ""):
            logger.debug("Add group for %s" % m.mag)
            MagasinGroupe.get_or_create(code=m.mag, defaults={"nom": m.libelle})
            m.groupe = m.mag
            m.save()
    except Exception:
        pass


def fix_config_twincat_amsid():
    """Add 2 configurations variables for Twincat AMS NetId and Port"""
    try:
        for m in Magasin.select(Magasin).where(Magasin.eco_type << ["T", "C", "A", "K", "P"]):
            Config.get_or_create(
                poste=m.type_mag,
                cle="cfg",
                propriete="k_eco_ams_netid",
                defaults={
                    "ref_id": "_TC1{:0>5}".format(str(m.pk)),
                    "type_value": "C",
                    "value": "1.2.3.4.1.1",
                    "hierarchie_menu": "Twincat\\Configuration\\AMS NetId",
                    "type_champ": "reg",
                },
            )
            Config.get_or_create(
                poste=m.type_mag,
                cle="cfg",
                propriete="k_eco_ams_port",
                defaults={
                    "ref_id": "_TC2{:0>5}".format(str(m.pk)),
                    "type_value": "N",
                    "value": "851",
                    "hierarchie_menu": "Twincat\\Configuration\\AMS Port",
                    "type_champ": "num",
                },
            )
    except Exception:
        pass


def fix_list_notes():
    """Apply the right value for the with_notes field"""
    lists_with_messages = ListeErrorModel.select(ListeErrorModel.liste).where(ListeErrorModel.message != "").distinct()

    update_query = ListeModel.update(with_note=1).where(
        (ListeModel.liste.in_(lists_with_messages)) & (ListeModel.with_note == 0)
    )

    # Execute the update
    rows_updated = update_query.execute()
    if rows_updated > 0:
        logger.info(f"With_Notes fix : Updated {rows_updated} records")


def fix_freefields():
    # Add missing freefields (libelles)
    # 0 = Unlocked | 1 = Locked but visible | 2 = Locked and hidden
    new_fields_count = 0
    new_fields_count += get_or_create_freefield("ref_desig", "Designation", 1)
    new_fields_count += get_or_create_freefield("ref_dci", "DCI", 1)
    new_fields_count += get_or_create_freefield("ref_desig_bis", "Desig Bis", 1)
    new_fields_count += get_or_create_freefield("ref_com_med", "Commentaire", 0)
    new_fields_count += get_or_create_freefield("ref_forme", "Forme", 2, "/api/product/forme/selectbox")
    new_fields_count += get_or_create_freefield("ref_reap_mode", "Replenish Mode", 2, "/api/acced/reappro/selectbox")

    if new_fields_count > 0:
        print(f"    Added {new_fields_count} new freefields.")


def execute_ddl_query(query):
    """Execute a SQL query"""
    # print(query)
    mysql_db.execute_sql(query)


def get_or_create_profil(ressource, groups=[], view=1, create=0, update=1, delete=0):
    """
    Check if profil exists, if not we create it
    """
    for grp in groups:
        try:
            Profil.get(profil=grp, ressource=ressource)
        except DoesNotExist:
            prof = Profil()
            prof.profil = grp
            prof.ressource = ressource
            prof.visu = view
            prof.cree = create
            prof.edit = update
            prof.supp = delete
            prof.save()
            print(f"    Added profil {ressource} for group {grp}")


def get_or_create_default(code: int, label: str) -> None:
    _, created = CodeBlocage.get_or_create(
        valeur=code,
        defaults={
            "libelle": label,
        },
    )


def get_or_create_format(code: str, def_value: dict) -> Format:
    """Check if format exists or create it"""
    res, created = Format.get_or_create(format=code, defaults=def_value)
    return res


def get_or_create_poste(poste: str, def_value: dict) -> Poste:
    res, created = Poste.get_or_create(poste=poste, defaults=def_value)
    return res


def get_or_create_config(
    poste: str,
    cle: str,
    prop: str,
    *,
    ref_id: str = None,
    type_value: str = None,
    value: str = None,
    hierarchie_menu: str = None,
    type_champ: str = None,
    info_champ: str = None,
    ordre: int = 0,
    prive: int = 0,
) -> Config:

    params = {
        "ref_id": ref_id,
        "type_value": type_value,
        "value": value,
        "hierarchie_menu": hierarchie_menu,
        "type_champ": type_champ,
        "info_champ": info_champ,
        "ordre": ordre,
        "prive": prive,
    }

    def_value = {k: v for k, v in params.items() if v is not None}

    res, created = Config.get_or_create(poste=poste, cle=cle, propriete=prop, defaults=def_value)
    if created:
        print(f"    Added config {prop} for poste {poste}")
    return res


def get_or_create_freefield(code: str, libelle: str, readonly=2, value="") -> Label:
    res, created = Label.get_or_create(
        code=code, defaults={"libelle": libelle, "libelle_0": value, "readonly": readonly}
    )
    return 1 if created else 0


def generate_equipment_topography():
    equipments = (
        Magasin.select(Magasin)
        .join(EquipmentUnitStock, JOIN.LEFT_OUTER, on=EquipmentUnitStock.equipment_pk == Magasin.pk)
        .where(EquipmentUnitStock.pk.is_null())
    )

    for eq in equipments:
        regenerate(eq)


def update_catalog():
    add_riedls()
    add_acceds()
    add_aide()
    add_astus()
    add_externe()


def add_externe():
    equipment = add_type(EquipmentType.EXTERNE.value, EcoType.Externe.value)
    version = add_version(ExternalMachineType.Etagere.value, equipment.pk)
    add_option("LIGNE", 1, version.pk)
    add_option("COLONNE", 1, version.pk)
    add_option("PROFONDEUR", 1, version.pk)
    version = add_version(ExternalMachineType.Chariot.value, equipment.pk)
    add_option("LIGNE", 1, version.pk)
    add_option("COLONNE", 1, version.pk)
    add_option("PROFONDEUR", 1, version.pk)


def astus_version(label, col, equipment_pk, sub_version):
    version = add_version(f"ASTUS_{label}", equipment_pk, sub_version=sub_version)
    add_option("PROFONDEUR", 1 if sub_version == "XL" else 2, version.pk)
    nb_1_3 = 12 - col
    nb_lines = 14 if sub_version == "XL" else 23

    unit_stock = add_unit_stock(total_line=nb_lines, total_column=col, order=0, version_pk=version.pk)
    lines_h = add_unit_stock_lines(start=9, end=nb_lines - 1, unit_stock_pk=unit_stock.pk)

    end_1_7 = (2 * (6 - nb_1_3)) - 1
    if nb_1_3 > 0:
        if sub_version == "XL":
            add_unit_stock_columns(start=2 * (6 - nb_1_3), end=col - 2, line_pk=lines_h.pk, container_type="PIL 1/3 H")
        else:
            add_unit_stock_columns(start=2 * (6 - nb_1_3), end=col - 2, line_pk=lines_h.pk, container_type="BAC 1/3 H")
    else:
        end_1_7 = end_1_7 - 1

    add_unit_stock_columns(start=0, end=end_1_7, line_pk=lines_h.pk, container_type="BAC 1/7 H")

    lines_b = add_unit_stock_lines(start=0, end=8, unit_stock_pk=unit_stock.pk)
    add_unit_stock_columns(start=0, end=(2 * (6 - nb_1_3)) - 1, line_pk=lines_b.pk, container_type="BAC 1/7 B")
    if nb_1_3 > 0:
        if sub_version == "XL":
            add_unit_stock_columns(start=2 * (6 - nb_1_3), end=col - 1, line_pk=lines_b.pk, container_type="PIL 1/3 B")
        else:
            add_unit_stock_columns(start=2 * (6 - nb_1_3), end=col - 1, line_pk=lines_b.pk, container_type="BAC 1/3 B")


def add_astus():
    equipment = add_type(EquipmentType.ASTUS.value, EcoType.Astus.value)

    for col in range(6, 12):
        astus_version(label="V3", col=col, equipment_pk=equipment.pk, sub_version=str(col))

    astus_version(label="V3", col=6, equipment_pk=equipment.pk, sub_version="XL")


def add_aide():
    equipment = add_type(EquipmentType.AIDE.value, EcoType.Coupe.value)

    # Aide Cut V1
    version = add_version(CuttingMachineType.Aide_Cut.value, equipment.pk)
    unit_stock = add_unit_stock(total_line=3, total_column=6, order=0, version_pk=version.pk)
    lines = add_unit_stock_lines(start=0, end=2, unit_stock_pk=unit_stock.pk)
    add_unit_stock_columns(start=0, end=5, line_pk=lines.pk, container_type="BOITE PASS")

    # Aide Pick V1
    version = add_version(PickingMachineType.Aide_Pick.value, equipment.pk)
    for stock in range(18):
        unit_stock = add_unit_stock(total_line=18, total_column=4, order=stock, version_pk=version.pk)
        if stock == 1:
            lines = add_unit_stock_lines(start=0, end=6, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type="BOITE PASS")
            lines = add_unit_stock_lines(start=7, end=17, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type="BOITE PASS")
        elif stock == 2 or stock == 3:
            lines = add_unit_stock_lines(start=0, end=2, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type="BOITE PASS")
            lines = add_unit_stock_lines(start=3, end=17, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type="")
        elif 4 <= stock <= 5:
            lines = add_unit_stock_lines(start=0, end=9, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type="BOITE PASS")
            lines = add_unit_stock_lines(start=10, end=17, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type="")
        elif 6 <= stock <= 8:
            lines = add_unit_stock_lines(start=0, end=3, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type="BOITE PASS")
            lines = add_unit_stock_lines(start=4, end=9, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type="")
            lines = add_unit_stock_lines(start=10, end=17, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type="BAC")
        else:
            lines = add_unit_stock_lines(start=0, end=17, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type="BOITE PASS")

    # Aide Classic
    version = add_version(PickingMachineType.Aide_V2.value, equipment.pk)
    for stock in range(10):
        unit_stock = add_unit_stock(total_line=18, total_column=4, order=stock, version_pk=version.pk)
        lines = add_unit_stock_lines(start=0, end=17, unit_stock_pk=unit_stock.pk)
        add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type="BOITE PASS")


def add_acceds():
    equipment = add_type(EquipmentType.ACCED.value, EcoType.Cueillette.value)

    # ACCED V3
    version = add_version(PickingMachineType.Acced_Dose_Unit.value, equipment.pk)
    for stock in range(10):
        unit_stock = add_unit_stock(total_line=13, total_column=4, order=stock, version_pk=version.pk)
        if stock < 9:
            lines = add_unit_stock_lines(start=0, end=2, unit_stock_pk=unit_stock.pk)
        else:
            lines = add_unit_stock_lines(start=0, end=4, unit_stock_pk=unit_stock.pk)

        add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type="BOITE PASS")

        if stock <= 4:
            lines = add_unit_stock_lines(start=3, end=12, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=0, line_pk=lines.pk, container_type="TIROIR 1/4")
        elif stock <= 7:
            lines = add_unit_stock_lines(start=3, end=12, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=0, line_pk=lines.pk, container_type="TIROIR 1/5")
        elif stock == 8:
            lines = add_unit_stock_lines(start=3, end=12, unit_stock_pk=unit_stock.pk)
            add_unit_stock_columns(start=0, end=0, line_pk=lines.pk, container_type="TIROIR 1/3")

    add_option("PROFONDEUR", 2, version.pk)

    # ACCED V2
    version = add_version(PickingMachineType.Acced_Boite_Pass.value, equipment.pk)
    for stock in range(10):
        unit_stock = add_unit_stock(total_line=18, total_column=4, order=stock, version_pk=version.pk)
        lines = add_unit_stock_lines(start=0, end=17, unit_stock_pk=unit_stock.pk)
        add_unit_stock_columns(start=0, end=3, line_pk=lines.pk, container_type="BOITE PASS")

    add_option("PROFONDEUR", 2, version.pk)


def add_riedls():
    # RIEDL PHASYS
    equipment = add_type(EquipmentType.RIEDL.value, EcoType.Riedl.value)
    version = add_version(RiedlMachineType.Riedl_Version_2.value, equipment.pk)
    add_option("ENTREE", 1, version.pk)
    add_option("ENTREE_AUTO", 0, version.pk)
    add_option("SORTIE", 1, version.pk)
    add_option("URGENCE", 1, version.pk)
    unit_stock = add_unit_stock(total_line=1, total_column=1, order=0, version_pk=version.pk)
    lines = add_unit_stock_lines(start=0, end=0, unit_stock_pk=unit_stock)
    add_unit_stock_columns(start=0, end=0, line_pk=lines.pk, container_type="SHELF")


def fill_mag_format_table():
    equipments = Magasin.select(Magasin.mag).where(
        Magasin.eco_type << [EcoType.Coupe.value, EcoType.Cueillette.value, EcoType.Astus.value, EcoType.Riedl.value]
    )
    for equip in equipments:
        # Retrieve all format define per equipment,
        AdrFormatList = Adresse.select(Adresse.format).distinct().where(Adresse.magasin == equip.mag)
        for CurrentFormat in AdrFormatList:
            MagasinFormat.get_or_create(magasin=equip.mag, format=CurrentFormat.format)


def add_type(label, ecoType):
    try:
        type = CatalogType.get(CatalogType.type == label)
        versions = CatalogVersion.select(CatalogVersion.pk).where(CatalogVersion.type_pk == type.pk)
        unit_stocks = CatalogUnitStock.select(CatalogUnitStock.pk).where(CatalogUnitStock.version_pk << versions)
        lines = CatalogLines.select(CatalogLines.pk).where(CatalogLines.unit_stock_pk << unit_stocks)

        CatalogColumns.delete().where(CatalogColumns.line_pk << lines).execute()
        CatalogLines.delete().where(CatalogLines.unit_stock_pk << unit_stocks).execute()
        CatalogUnitStock.delete().where(CatalogUnitStock.pk << versions).execute()

        CatalogOption.delete().where(CatalogOption.version_pk << versions).execute()
        CatalogVersion.delete().where(CatalogVersion.type_pk == type.pk).execute()
        CatalogType.delete().where(CatalogType.type == label).execute()

    except DoesNotExist:
        type = None

    equipement = CatalogType()
    equipement.type = label
    equipement.eco_type = ecoType
    equipement.save()
    return equipement


def add_version(label, type_pk, sub_version=""):
    version = CatalogVersion()
    version.numVersion = label
    version.type_pk = type_pk
    version.subVersion = sub_version
    version.save()
    return version


def add_option(label, value, version_pk):
    option = CatalogOption()
    option.version_pk = version_pk
    option.type = label
    option.value = value
    option.save()


def add_unit_stock(total_line, total_column, order, version_pk):
    unit_stock = CatalogUnitStock()
    unit_stock.version_pk = version_pk
    unit_stock.total_line = total_line
    unit_stock.total_colonne = total_column
    unit_stock.order = order
    unit_stock.save()
    return unit_stock


def add_unit_stock_lines(start, end, unit_stock_pk):
    lines = CatalogLines()
    lines.unit_stock_pk = unit_stock_pk
    lines.start_line = start
    lines.end_line = end
    lines.save()
    return lines


def add_unit_stock_columns(start, end, container_type, line_pk):
    if start <= end:
        column = CatalogColumns()
        column.line_pk = line_pk
        column.start_column = start
        column.end_Column = end
        column.container_type = container_type
        column.save()


def check_episodes_from_patients():
    query_patients = Patient.select(Patient.ipp, Patient.nom, Patient.prenom).order_by(Patient.pk.asc())
    cursor_patient = mysql_db.execute(query_patients)
    updated_count = 0
    processed_patients_count = 0

    for patient_ipp, nom, prenom in cursor_patient:
        processed_patients_count += 1
        print("[%+8s] %s -> %s %s" % (processed_patients_count, patient_ipp, nom, prenom))

        query_episodes = (
            Sejour.select(Sejour.pk, Sejour.ipp, Sejour.date_entree, Sejour.date_sortie)
            .where(Sejour.ipp == patient_ipp)
            .order_by(Sejour.pk.desc())
        )

        cursor_sejours = mysql_db.execute(query_episodes)

        prec_date_entree = None
        for sejour_pk, sejour_ipp, date_entree, date_sortie in cursor_sejours:
            print(
                "%+14sentry:%s |  exit:%s | pk:%i | prec_date:%s"
                % (" ", str(date_entree), str(date_sortie), sejour_pk, str(prec_date_entree)),
                end=" ",
            )
            if prec_date_entree is None:
                # first and most recent episode, no action needed
                pass
            else:
                if date_sortie is None:
                    Sejour.update(date_sortie=prec_date_entree).where(Sejour.pk == sejour_pk).execute()
                    updated_count += 1
                    print("(updated)", end=" ")

            prec_date_entree = date_entree

            print("")

    print("%i entries updated" % updated_count)


def fix_equipment_types():
    oldNames = ["ETAGERE_V1", "CHARIOT_V1", "AIDE_CUT_V1", "AIDE_PICK_V1"]
    newNames = ["ETAGERE", "CHARIOT", "AIDE_CUT", "AIDE_PICK"]
    storesToCorrect = Magasin.select().where(Magasin.type_machine << oldNames)

    store_count = storesToCorrect.count()
    for store in storesToCorrect:
        index = oldNames.index(store.type_machine)
        store.type_machine = newNames[index]
        store.save()

    if store_count > 0:
        print(f"   {store_count} stores corrected")


def get_or_create_drug_county_system(country_code=None):
    if not country_code:
        country_code = os.environ.get("MEDIAN_DRUG_COUNTRY", "fr")
        print("Country code: %s" % country_code)
        Config.get_or_create(
            poste=TOUS_POSTE,
            cle="cfg",
            propriete="k_drug_country",
            defaults={
                "ref_id": "_DRUG0001",
                "type_value": "C",
                "value": country_code,
                "hierarchie_menu": "MEDIANWEB\\Drugs\\Country",
                "type_champ": "reg",
            },
        )


def set_default_printer_values():
    # Width x Height
    _get_or_create_printer_label("Sticky Label 102x68", 102, 68)
    _get_or_create_printer_label("Sticky Label 70x32", 70, 32)
    _get_or_create_printer_label("Sticky Label 32x40", 32, 40)
    _get_or_create_printer_label("Sticky Label 33x29", 33, 29)
    _get_or_create_printer_type("GX430t")
    _get_or_create_printer_type("ZD421")
    _get_or_create_printer_cmd("calibrate", "Calibration Label")
    _get_or_create_printer_cmd("user_barcode", "User barcode")
    _get_or_create_printer_cmd("drug_label", "Drug label")
    _get_or_create_printer_cmd("external_box_nominative_label", "External Nominative Box label")
    _get_or_create_printer_cmd("external_box_global_label", "External Global Box label")
    _get_or_create_printer_cmd("track_label", "Track label")
    _get_or_create_printer_cmd("box_uncomplete", "Riedl box uncomplete for taking mode")
    _get_or_create_printer_cmd("external_ud_label", "External unit dose label")
    _get_or_create_printer_cmd("container", "Container lable for Riedl")


def set_default_acced_printer():
    """For each ACCED we create 2 printers"""
    def_printer, _ = _get_or_create_printer_type("Zebra 300 dpi")
    def_label, _ = _get_or_create_printer_label("Continous Label 70", 67, -1)
    for m in Magasin.select(Magasin).where(Magasin.eco_type == "C"):
        for i in [1, 2]:
            p, _ = Printer.get_or_create(
                name="%s Zebra %i" % (m.type_mag, i),
                defaults={
                    "type_id": def_printer,
                    "current_label_id": def_label,
                    "address": "tcp://172.25.212.x:9100",
                    "option_cut": 1,
                    "cut_trigger": 1,
                },
            )
            PrinterMag.get_or_create(printer_id=p, mag_id=m, defaults={"primary": 1})


def set_default_riedl_printer():
    """Create printer for the taking mode"""
    def_printer, _ = _get_or_create_printer_type("Zebra 300 dpi")
    def_label, _ = _get_or_create_printer_label("Sticky Label 102x68", 102, 68)
    for m in Magasin.select(Magasin).where(Magasin.eco_type == "L"):
        p, _ = Printer.get_or_create(
            name="%s Zebra 1" % m.type_mag,
            defaults={
                "type_id": def_printer,
                "current_label_id": def_label,
                "address": "tcp://172.25.212.x:9100",
                "option_cut": 0,
                "cut_trigger": 0,
            },
        )
        PrinterMag.get_or_create(printer_id=p, mag_id=m, defaults={"primary": 1})


def _get_or_create_printer_label(pname, width_mm, height_mm):
    return PrinterLabel.get_or_create(name=pname, defaults={"width": width_mm, "height": height_mm})


def _get_or_create_printer_type(pname, language="ZPL", density=300):
    return PrinterType.get_or_create(
        name=pname,
        defaults={
            "lang": language,
            "density": density,
        },
    )


def _get_or_create_printer_cmd(template_name: str, command_name: str):
    """Create a printer command from"""
    import re
    import json
    from pathlib import Path
    from peewee import DoesNotExist

    # Create calibrate command if it doesn't exist
    printer_cmd, created = PrinterCommand.get_or_create(code=template_name, defaults={"name": command_name})
    if created:
        logger.info("Created PrinterCommand: Calibration Label")

    # Find all calibration files
    base_dir = Path(__file__).parent / "data" / "printer" / template_name
    calibrate_files = list(base_dir.glob("%s_*.tzpl" % template_name))
    logger.info(f"Found {len(calibrate_files)} %s files" % template_name)

    # Process each calibration file
    for file_path in calibrate_files:
        regexp_expression = template_name + r"_(\d+)x(\d+)_(\d+)\.tzpl"
        match = re.match(regexp_expression, file_path.name)
        if not match:
            logger.warning(f"Cannot parse dimensions and DPI from {file_path.name}")
            continue

        width, height, dpi = int(match.group(1)), int(match.group(2)), int(match.group(3))
        label_name = f"Sticky Label {width}x{height}"

        # Look for corresponding JSON file
        json_path = file_path.with_suffix(".json")
        json_content = "{}"
        if json_path.exists():
            try:
                with open(json_path, "r", encoding="utf-8") as json_file:
                    json_content = json_file.read()
                # Validate that it's valid JSON
                json.loads(json_content)
                logger.debug(f"Found and loaded JSON file: {json_path.name}")
            except json.JSONDecodeError:
                logger.warning(f"Invalid JSON in {json_path.name}, using empty object")
                json_content = "{}"
            except Exception as e:
                logger.warning(f"Error reading JSON {json_path.name}: {str(e)}")
                json_content = "{}"
        else:
            logger.debug(f"No JSON file found for {file_path.name}, using empty object")

        # Find or create printer label
        try:
            label = PrinterLabel.get(PrinterLabel.name == label_name)
            logger.debug(f"Found existing PrinterLabel: {label_name}")
        except DoesNotExist:
            label = PrinterLabel.create(name=label_name, width=width, height=height)
            logger.info(f"Created PrinterLabel: {label_name} ({width}x{height}mm)")

        # Check if command-label association already exists
        try:
            PrinterCommandLabel.get(
                (PrinterCommandLabel.label_id == label.pk)
                & (PrinterCommandLabel.command_id == printer_cmd.pk)
                & (PrinterCommandLabel.density == dpi)
            )
            logger.debug(f"PrinterCommandLabel already exists for {label_name} at {dpi} DPI")
            continue
        except DoesNotExist:
            # Read ZPL file content
            with open(file_path, "r") as f:
                zpl_content = f.read()

            # Create association
            PrinterCommandLabel.create(
                label_id=label.pk,
                command_id=printer_cmd.pk,
                print_code=zpl_content,
                print_dict=json_content,  # Use the JSON content from file
                enabled=1,
                density=dpi,
                unit="mm",
            )
            logger.info(f"Created PrinterCommandLabel for {label_name} at {dpi} DPI")


def fix_external_loading_counters():
    """ Add missing counters for external loading """
    extbox_compteur, extbox_isCreated = Compteur.get_or_create(cle="EXT_SHELVEBOX", defaults={"val": 1})
    if extbox_isCreated:
        logger.info(f"Compteur : Added {extbox_compteur.cle} with value {extbox_compteur.val}")

    external_mags = Magasin.select().where(Magasin.eco_type == EcoType.Externe.value)

    for external_mag in external_mags:
        new_compteur, isCreated = Compteur.get_or_create(cle=f"EXT_CHARGEMENT_{external_mag.mag}", defaults={"val": 1})
        if isCreated:
            logger.info(f"Compteur : Added {new_compteur.cle} with value {new_compteur.val}")

        new_compteur, isCreated = Compteur.get_or_create(cle=f"EXT_INCREMENT_{external_mag.mag}", defaults={"val": 1})
        if isCreated:
            logger.info(f"Compteur : Added {new_compteur.cle} with value {new_compteur.val}")


def fix_zonemag():
    """ Remove Zone and ZoneMag for newer machines"""
    newer_machines = [
        PickingMachineType.Acced_Boite_Pass.value,
        PickingMachineType.Acced_Dose_Unit.value,
        PickingMachineType.Aide_V2.value,
        PickingMachineType.Aide_V3.value,
        PickingMachineType.Aide_Pick.value,
    ]

    mag: Magasin
    for mag in Magasin.select().where(
        (Magasin.type_machine << newer_machines) & (Magasin.eco_type == EcoType.Cueillette.value)
    ):
        zone: Zone
        for zone in Zone.select().where(Zone.zone == mag.type_mag):
            logger.info("Delete Zone for %s" % zone.zone)
            zone.delete_instance()

        zonemag: ZoneMag
        for zonemag in ZoneMag.select().where(ZoneMag.zone == mag.type_mag):
            logger.info("Delete ZoneMag for %s" % zonemag.zone)
            zonemag.delete_instance()
